import colorconsole, colorconsole.terminal
import os, sys, traceback
import threading
import signal

if os.name == 'nt':
    import msvcrt
    import ctypes

    from ctypes import windll, byref, create_string_buffer
    from ctypes.wintypes import SMALL_RECT, _COORD

    def get_size_screen():
        return windll.user32.GetSystemMetrics(0), windll.user32.GetSystemMetrics(1)

    def get_size_console():
        '''Sets up the console size and buffer height.

        <at> param width {int} Width of console in column value.
        <at> param height {int} Height of console in row value.
        <at> param buffer_height {int} Buffer console height in row value.
        '''
        # Active console screen buffer
        # STD_OUTPUT_HANDLE -> -11, STD_ERROR_HANDLE -> -12)
        STDERR = -12
        # SMALL_RECT input
        LEFT = 0
        TOP = 0
        # handle
        hdl = windll.kernel32.GetStdHandle(STDERR)
        csbi = create_string_buffer(22)

        res = windll.kernel32.GetConsoleScreenBufferInfo(hdl, csbi)

        if res:
            import struct
            (bufx, bufy, curx, cury, wattr,
             left, top, right, bottom,
             maxx, maxy) = struct.unpack("hhhhHhhhhhh", csbi.raw)

        current_width = right - left + 1
        current_height = bottom - top + 1
        current_buffer_height = bufy
        return current_width, current_height, current_buffer_height

    def console_resize(width=80, height=24, buffer_height=600):
        '''Sets up the console size and buffer height.

        <at> param width {int} Width of console in column value.
        <at> param height {int} Height of console in row value.
        <at> param buffer_height {int} Buffer console height in row value.
        '''
        # Active console screen buffer
        # STD_OUTPUT_HANDLE -> -11, STD_ERROR_HANDLE -> -12)
        STDERR = -12
        # SMALL_RECT input
        LEFT = 0
        TOP = 0
        RIGHT = width - 1
        BOTTOM = height - 1
        # handle
        hdl = windll.kernel32.GetStdHandle(STDERR)
        csbi = create_string_buffer(22)

        res = windll.kernel32.GetConsoleScreenBufferInfo(hdl, csbi)

        if res:
            import struct
            (bufx, bufy, curx, cury, wattr,
             left, top, right, bottom,
             maxx, maxy) = struct.unpack("hhhhHhhhhhh", csbi.raw)

        current_width = right - left + 1
        current_height = bottom - top + 1
        current_buffer_height = bufy

        if buffer_height < height:
            buffer_height = height
        # order of resizing avoiding some problems
        if current_buffer_height > buffer_height:
            rect = SMALL_RECT(LEFT, TOP, RIGHT, BOTTOM)  # (left, top, right, bottom)
            windll.kernel32.SetConsoleWindowInfo(hdl, True, byref(rect))

            bufsize = _COORD(width, buffer_height)  # columns, rows
            windll.kernel32.SetConsoleScreenBufferSize(hdl, bufsize)
        else:
            bufsize = _COORD(width, buffer_height)  # columns, rows
            windll.kernel32.SetConsoleScreenBufferSize(hdl, bufsize)

            rect = SMALL_RECT(LEFT, TOP, RIGHT, BOTTOM)  # (left, top, right, bottom)
            windll.kernel32.SetConsoleWindowInfo(hdl, True, byref(rect))


    class _CursorInfo(ctypes.Structure):
        _fields_ = [("size", ctypes.c_int),
                    ("visible", ctypes.c_byte)]

    def hide_cursor():
        ci = _CursorInfo()
        handle = ctypes.windll.kernel32.GetStdHandle(-11)
        ctypes.windll.kernel32.GetConsoleCursorInfo(handle, ctypes.byref(ci))
        ci.visible = False
        ctypes.windll.kernel32.SetConsoleCursorInfo(handle, ctypes.byref(ci))

    def show_cursor():
        ci = _CursorInfo()
        handle = ctypes.windll.kernel32.GetStdHandle(-11)
        ctypes.windll.kernel32.GetConsoleCursorInfo(handle, ctypes.byref(ci))
        ci.visible = True
        ctypes.windll.kernel32.SetConsoleCursorInfo(handle, ctypes.byref(ci))

    def _Getch():
        return msvcrt.getch()


else:
    def get_size_screen():
        return -1, -1

    def get_size_console():
        def ioctl_GWINSZ(fd):
            try:
                import fcntl, termios, struct, os
                cr = struct.unpack('hh', fcntl.ioctl(fd, termios.TIOCGWINSZ,'1234'))
            except:
                return None
            return cr
        cr = ioctl_GWINSZ(0) or ioctl_GWINSZ(1) or ioctl_GWINSZ(2)
        if not cr:
            try:
                fd = os.open(os.ctermid(), os.O_RDONLY)
                cr = ioctl_GWINSZ(fd)
                os.close(fd)
            except:
                pass
        if not cr:
            try:
                cr = (env['LINES'], env['COLUMNS'])
            except:
                return None
        return int(cr[1])+1, int(cr[0])+1, int(cr[0])+1

    def console_resize(width=80, height=24, buffer_height=600):
        raise Exception('it is not possible on linux') 

    def hide_cursor():
        sys.stdout.write("\033[?25l")
        sys.stdout.flush()

    def show_cursor():
        sys.stdout.write("\033[?25h")
        sys.stdout.flush()

    def _Getch():
        import sys, tty, termios
        fd = sys.stdin.fileno()
        old_settings = termios.tcgetattr(fd)
        try:
            tty.setraw(sys.stdin.fileno())
            ch = sys.stdin.read(1)
        finally:
            termios.tcsetattr(fd, termios.TCSADRAIN, old_settings)
        return ch


LEFT_ALIGN = "{:<%s}"
RIGHT_ALIGN = "{:>%s}"
CENTER_ALIGN = "{:^%s}"


class Action(object):
    def __init__(self, screen):
        pass

    def __call__(self):
        pass

class Event(object):
    
    def __init__(self, name):
        self.name = name
        self._observateurs = []

    def __iadd__(self, observateur):
        if not callable(observateur):
            raise TypeError("object does not callable")
        observateurs = self._observateurs
        if observateur not in observateurs:
            observateurs.append(observateur)
        return self

    def __isub__(self, observateur):
        observateurs = self._observateurs
        if observateur in observateurs:
            observers.remove(observateur)
        return self

    def __call__(self, *args, **kw):
        [observateur(*args, **kw) for observateur in self._observateurs]



class ListenGetch(threading.Thread):

    def __init__(self, screen):
        threading.Thread.__init__(self)
        self.Terminated = False
        self.viewChar = False
        self.screen = screen

    def run(self):
        while not self.Terminated:
            char = ord(_Getch())#ord(msvcrt.getch())
            if self.viewChar: print(char)
            if char in self.screen._binds:
                self.screen._binds[char]()

    def stop(self):
        self.Terminated = True

class Screen(object):

    def __init__(self, term=None):
        if term == None:
            term = colorconsole.terminal.get_terminal()
        self.term = term
        self._listen = ListenGetch(self)
        self._binds = {}


    def resize(self, width=80, height=24, buffer_height=600):
        console_resize(width, height, buffer_height)

    def __getattr__(self, name):
        try:
            def fctdyn(*args, **kwargs):
                return getattr(self.term, name)(*args, **kwargs)
            return fctdyn 
        except Exception as e:
            raise AttributeError("Screen instance has no attribute'%s'" % name)

    def start_listen(self):
        self._listen.start()
    
    def stop_listen(self):
        self._listen.stop()

    def bind(self, key, obj):
        if key not in self._binds:
            self._binds[key]= Event(key)
        self._binds[key] += obj    

    def exit(self):
        try:
            os.kill(signal.CTRL_C_EVENT, 0)
        except:
            os.kill(os.getpid(), signal.SIGINT) 
        sys.exit(1)

    @property
    def width(self):
        return get_size_console()[0]

    @width.setter
    def width(self, value):
        self.resize(value, self.height, self.buffer_height)

    @property
    def height(self):
        return get_size_console()[1]

    @height.setter
    def height(self, value):
        self.resize(self.width, value, self.buffer_height)

    @property
    def buffer_height(self):
        return get_size_console()[2]

    @buffer_height.setter
    def buffer_height(self, value):
        self.resize(self.width, self.height, value)

    def no_cursor(self, v=True):
        if v: 
            hide_cursor()
            return
        show_cursor()    

class Text(object):

    def __init__(self, screen, x, y, l=1, bg='BLACK', fg='WHITE', default_bg='BLACK', default_fg='WHITE', align=LEFT_ALIGN):
        self.screen = screen
        self.change = True
        self._x     = x
        self._y     = y
        self._l     = l
        self._fg    = fg
        self._bg    = bg
        self._dfg   = default_fg
        self._dbg   = default_bg
        self._value = None
        self._align = align % l
        self.clear()

    @property
    def value(self):
        return self._value

    @value.setter
    def value(self, value):
        value = str(value or '')[0:self._l]
        if value != self.value:
            self.change = True
        self._value = value
        self._print()

    @property
    def bg(self):
        return self._bg

    @bg.setter
    def bg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        if self._bg != value:
            self.change = True
        self._bg = value
        self._print()

    @property
    def fg(self):
        return self._fg

    @fg.setter
    def fg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        if self._fg != value:
            self.change = True
        self._fg = value
        self._print()

    @property
    def dbg(self):
        return self._dbg

    @dbg.setter
    def dbg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        self._dbg = value

    @property
    def dfg(self):
        return self._dfg

    @dfg.setter
    def dfg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        self._dfg = value

    def refresh(self):
        self.change = True
        self._print()

    def _print(self):
        if self.change:
            self.screen.set_color(fg=colorconsole.terminal.colors[self._fg], bk=colorconsole.terminal.colors[self._bg])
            self.screen.print_at(self._x,self._y, self._align.format(self._value))
            self.screen.set_color(fg=colorconsole.terminal.colors[self._dfg], bk=colorconsole.terminal.colors[self._dbg]) 
        self.change = False    

    def clear(self):
        self.value = ' ' * self._l
    
    def _test(self):
        self.value = 'X' * self._l

class Menu(Text):

    def __init__(self, screen, x, y, l=1, bg='BLACK', fg='WHITE', default_bg='BLACK', default_fg='WHITE', align=LEFT_ALIGN):
        Text.__init__(self,screen, x, y, l=l, bg=bg, fg=fg, default_bg=default_bg, default_fg=default_fg, align=align)

    def _print(self):
        if self.change:
            txt = self._align.format(self._value)
            x = self._x
            for i in txt:
                if i.upper() == i and i != " ":
                    self.screen.set_color(fg=colorconsole.terminal.colors[self._bg], bk=colorconsole.terminal.colors[self._fg])
                else:
                    self.screen.set_color(fg=colorconsole.terminal.colors[self._fg], bk=colorconsole.terminal.colors[self._bg])
                self.screen.print_at(x, self._y, i)
                x = x +1
            self.screen.set_color(fg=colorconsole.terminal.colors[self._dfg], bk=colorconsole.terminal.colors[self._dbg]) 
        self.change = False    



class HProgressBar(object):

    def __init__(self, screen, x, y, l=1, bg='WHITE', fg='BLACK', default_bg='BLACK', default_fg='WHITE', align=CENTER_ALIGN, reverse=False):
        self.screen = screen
        self.change = True
        self._x     = x
        self._y     = y
        self._l     = l
        self._fg    = fg
        self._bg    = bg
        self._dfg   = default_fg
        self._dbg   = default_bg
        self._value = None
        self._align = align % l
        self._text = ''
        self._texts = []
        self._reverse = reverse
        self._set_texts()
        self.clear()

    def _set_texts(self):
        for i in range(0,self._l):
            self._texts.append(Text(self.screen, self._x + i, self._y, l=1, bg=self._bg, fg=self._fg, default_bg=self._dbg, default_fg=self._dfg))
        
    def clear(self):
        self.value = 0
        self.text = ''
    
    @property
    def value(self):
        return self._value

    @value.setter
    def value(self, value):
        try:
            if 0 <= float(value) <= 100:
                value = float(value)
            else:
                a = 1/0 #generate error
        except:
            raise AttributeError('%s not between 0 and 100' % value)
        if value != self.value:
            self.change = True
        self._value = value
        self._print()

    @property
    def bg(self):
        return self._bg

    @bg.setter
    def bg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        if self._bg != value:
            self.change = True
        self._bg = value
        self._print()

    @property
    def fg(self):
        return self._fg

    @fg.setter
    def fg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        if self._fg != value:
            self.change = True
        self._fg = value
        self._print()

    @property
    def dbg(self):
        return self._dbg

    @dbg.setter
    def dbg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        self._dbg = value

    @property
    def dfg(self):
        return self._dfg

    @dfg.setter
    def dfg(self, value):
        if value not in colorconsole.terminal.colors.keys():
            raise AttributeError('%s not in color of terminal' % value)
        self._dfg = value

    def refresh(self):
        self.change = True
        self._print()
 
    @property
    def text(self):
        return self._text

    @text.setter
    def text(self, text):
        if text != self.text:
            self.change = True
        self._text = text
        self._print()
    
    def _get_colors(self, src, level):
        return self.bg, self.fg

    def _print(self):
        if self.change:
            prt = self._align.format(self.text) 
            for i in range(0, self._l):
                j = i+1
                if self._reverse: j = self._l - j
                text = self._texts[i]
                text.value = prt[i]
                if (j) * (100.00/self._l) <= self.value:
                    text.bg, text.fg = self._get_colors(self, (j+1)*(100.00/self._l))
                else:
                    text.bg, text.fg = self.dbg, self.dfg
        self.change = False    
    
    def _test(self):
        self.value = 100
        txt =''.join([str(i) for i in range(1,11)]) * int(self._l/10) +''.join([str(i) for i in range(1,11)][0:(self._l % 10)])  
        self.text = txt

class VProgressBar(HProgressBar):

    def __init__(self, screen, x, y, l=1, bg='WHITE', fg='BLACK', default_bg='BLACK', default_fg='WHITE', align=CENTER_ALIGN, reverse=False):
        HProgressBar.__init__(self, screen, x, y, l, bg, fg, default_bg, default_fg, align, reverse)

    def _set_texts(self):
        for i in range(0,self._l):
            self._texts.append(Text(self.screen, self._x, self._y - i, l=1, bg=self._bg, fg=self._fg, default_bg=self._dbg, default_fg=self._dfg))


def test():
    t = colorconsole.terminal.get_terminal()
    for i in colorconsole.terminal.colors.keys():
        #t.set_color(colorconsole.terminal.colors[i], None)
        t.cprint(colorconsole.terminal.colors[i], colorconsole.terminal.colors["BLACK"], "{:<20}".format(i))
        t.cprint(None,colorconsole.terminal.colors[i], "{:<20}".format(' '*20+'\n'))
    t.set_color(colorconsole.terminal.colors["WHITE"], colorconsole.terminal.colors["BLACK"])


class TestBind(object):

    def __init__(self, term=None):
        self._listen = ListenGetch(self)
        self._listen.viewChar = True
        self._binds = {}
        self.run = False

    def start_listen(self):
        self._listen.start()
        self.run = True


    def stop_listen(self):
        self._listen.stop()
        self.run = False

    def bind(self, key, obj):
        if key not in self._binds:
            self._binds[key]= Event(key)
        self._binds[key] += obj    

    def test(self):
        print('coucou')
        self.stop_listen()



def testbind():
    import time
    print('click on q for exit')
    a = TestBind()
    a.bind(113,a.test)
    a.start_listen()
    while(a.run):
        time.sleep(1)

