"""An incredibly simple HTTP basic auth implementation.
"""
# stdlib imports
from base64 import b64decode
from base64 import b64encode
from urllib import quote
from urllib import unquote


class DecodeError(Exception):
    pass


def encode(username, password):
    """Returns an HTTP basic authentication encrypted string given a valid
    username and password.
    """
    return 'Basic ' + b64encode('%s:%s' % (quote(username), quote(password)))


def decode(encoded_str):
    """Decode an encrypted HTTP basic authentication string. Returns a tuple of
    the form (username, password), and raises a DecodeError exception if
    nothing could be decoded.
    """
    split = encoded_str.strip().split(' ')

    # If split is only one element, try to decode the username and password
    # directly.
    if len(split) == 1:
        try:
            username, password = b64decode(split[0]).split(':')
        except:
            raise DecodeError

    # If there are only two elements, check the first and ensure it says
    # 'basic' so that we know we're about to decode the right thing. If not,
    # bail out.
    elif len(split) == 2:
        if split[0].strip().lower() == 'basic':
            try:
                username, password = b64decode(split[1]).split(':')
            except:
                raise DecodeError
        else:
            raise DecodeError

    # If there are more than 2 elements, something crazy must be happening.
    # Bail.
    else:
        raise DecodeError

    return unquote(username), unquote(password)


def check_auth(request, username, password):
    """Check if the given request contains correct auth headers
    """
    if not 'Authorization' in request.headers:
        return False
    return (username, password) == decode(request.headers['Authorization'])
