# coding: utf-8

"""
    AUTHOR: Igor Kalnitsky <igor@kalnitsky.org>
    URL: http://www.kalnitsky.org/
"""

import nautilus
import pynotify
import urllib2
import gtk
import os

from poster.encode import multipart_encode
from poster.streaminghttp import register_openers


class XsnippetExtension(nautilus.MenuProvider):
    """
        Nautilus extension to send files to xsnippet.org

        Add 'Send to xsnippet.org' item to nautilus context menu.
        Link to the snippet is copied to the clipboard and displays
        by notification.
    """

    API_URL = "http://www.xsnippet.org/new"
    ICON = "/usr/share/pixmaps/nautilus-xsnippet.png"


    def __init__(self):
        pynotify.init("nautilus-xsnippet")
        self.clipboard = gtk.clipboard_get()

    def sendFile(self, filename):
        """
            Send file to xsnippet.org and return link to last one.
            Return 'None' if error occured.
        """

        register_openers()
        data, headers = multipart_encode({
            "title": os.path.basename(filename),
            "file": open(filename, "rt"),
            "author": os.environ.get("USER"),
        })

        request = urllib2.Request(XsnippetExtension.API_URL, data, headers)
        response = urllib2.urlopen(request)

        if response.getcode() == 200:
            return response.geturl()
        return None

    def menu_click(self, menu, sourcefile):
        """
            'Send to xsnippet.org' item handler.

            Send file to xsnippet and show notification.
        """
        title = ''
        message = ''
        snippeturl = self.sendFile(sourcefile.get_uri()[7:])

        if snippeturl is None:
            title = "Error"
            message = "Can't post file"
        else:
            title = "Posted to"
            message = snippeturl + '\n[the link is copied to clipboard]'

            self.clipboard.set_text(message)
            self.clipboard.store()

        notification = pynotify.Notification(title, message,
                                             XsnippetExtension.ICON)
        notification.show()

    def get_file_items(self, window, files):
        """
            Show 'Send to xsnippet.org' item only for 1 selected file.
        """

        if len(files) != 1 or files[0].is_directory():
            return

        item = nautilus.MenuItem(
                "XsnippetExtension::send_to_xsnippet",
                "Send to xsnippet.org",
                "Send the current file to xsnippet.org",
        )

        item.set_property("icon", "nautilus-xsnippet")
        item.connect("activate", self.menu_click, files[0])
        return [item]
