# -*- coding: utf-8 -*-
"""
The module contains helper functions to work with SDPA.

Created on Fri May 16 13:52:58 2014

@author: Peter Wittek
"""
from bisect import bisect_left
from subprocess import call
import tempfile


def read_sdpa_out(filename):
    """Helper function to parse the output file of SDPA
    """
    file_ = open(filename, 'r')
    for line in file_:
        if line.find("objValPrimal") > -1:
            primal = float((line.split())[2])
        if line.find("objValDual") > -1:
            dual = float((line.split())[2])

    file_.close()
    return primal, dual


def solve_sdp(sdp_problem):
    """Helper function to write out the SDP problem to a temporary
    file, call the solver, and parse the output.
    """
    primal, dual = 0, 0
    tempfile_ = tempfile.NamedTemporaryFile()
    tmp_filename = tempfile_.name
    tempfile_.close()
    tmp_dats_filename = tmp_filename + ".dat-s"
    tmp_out_filename = tmp_filename + ".out"
    write_to_sdpa(sdp_problem, tmp_dats_filename)
    call(["sdpa", tmp_dats_filename, tmp_out_filename])
    primal, dual = read_sdpa_out(tmp_out_filename)
    return primal, dual


def convert_row_to_sdpa_index(block_struct, row_offsets, row):
    """Helper function to map to sparse SDPA index values.
    """
    block_index = bisect_left(row_offsets[1:], row + 1)
    width = block_struct[block_index]
    row = row - row_offsets[block_index]
    i, j = divmod(row, width)
    return block_index, i, j


def write_to_sdpa(sdp_problem, filename):
    """Write the SDP relaxation to SDPA format.

    Arguments:
    filename -- the name of the file. It must have the suffix ".dat-s"
    """
    file_ = open(filename, 'w')
    file_.write('"file ' + filename + ' generated by ncpol2sdpa"\n')
    file_.write(str(sdp_problem.n_vars) + ' = number of vars\n')
    file_.write(str(len(sdp_problem.block_struct)) + ' = number of blocs\n')
    # bloc structure
    file_.write(str(sdp_problem.block_struct).replace('[', '(')
                .replace(']', ')'))
    file_.write(' = BlocStructure\n')
    # c vector (objective)
    file_.write(str(list(sdp_problem.obj_facvar)).replace(
        '[', '{').replace(']', '}'))
    file_.write('\n')
    # Coefficient matrices
    row_offsets = [0]
    cumulative_sum = 0
    for block_size in sdp_problem.block_struct:
        cumulative_sum += block_size ** 2
        row_offsets.append(cumulative_sum)
    for row in range(len(sdp_problem.F_struct.rows)):
        if len(sdp_problem.F_struct.rows[row]) > 0:
            col_index = 0
            for k in sdp_problem.F_struct.rows[row]:
                value = sdp_problem.F_struct.data[row][col_index]
                col_index += 1
                block_index, i, j = convert_row_to_sdpa_index(
                    sdp_problem.block_struct, row_offsets, row)
                if k == 0:
                    value *= -1
                file_.write('{0}\t{1}\t{2}\t{3}\t{4}\n'.format(
                    k, block_index + 1, i + 1, j + 1, value))
    file_.close()
