"""
Produce a batch file for plotting a lifeline from a netlogger log.
Supported output formats are R (eventually!) and gnuplot.
"""
__author__ = "Dan Gunter <dkgunter@lbl.gov>"
__rcsid__ = "$Id: lifeline.py 788 2008-06-02 17:11:49Z dang $"

from netlogger.parsers.base import TS_FIELD, EVENT_FIELD, NLFastParser
from netlogger import nllog

# Logging
log = nllog.NullLogger()
def activateLogging(name=__name__):
    global log
    log = nllog.getLogger(name)

class LifelineSet:
    """Set of lifelines with different identifiers.
    """
    def __init__(self, events=None, lineids=None, groupids=None,
                 prefix=''): 
        self.events = dict.fromkeys(events)
        self.event_pos = events
        self.lineids, self.groupids = lineids, groupids        
        self.event_prefix, self.event_prefix_len = prefix, len(prefix)
        self.lifelines = [ ]
        self._count = 0    

    def process(self, record):
        """Process a record (dictionary) and, potentially, return a lifeline
        """
        self._count += 1
        try:
            r = self._process(record)
        except KeyError, E:
            warn("error for record %d: missing %s" % (self._count, E))
            r = None
        return r
    
    def _process(self, record):
        result = None
        event = record[EVENT_FIELD]
        if self.event_prefix and event.startswith(self.event_prefix):
            event = event[self.event_prefix_len:]
        if self.events.has_key(event):
            pos = self.event_pos.index(event)
            tmp = [ ]
            for key in self.lineids:
                if not record.has_key(key):
                    raise ValueError("missing line-id key '%s'" % key)
                tmp.append(record[key])
            record_lineid = tuple(tmp)
            tmp = [ ]
            for key in self.groupids:
                if not record.has_key(key):
                    raise ValueError("missing group-id key '%s'" % key)
                tmp.append(record[key])
            record_groupid = tuple(tmp)
            # find lifeline that matches this record
            matched = False
            for i, lifeline in enumerate(self.lifelines):
                if lifeline.lineid == record_lineid:
                    if lifeline.groupid != record_groupid:
                        raise ValueError("same lineid, different group")
                    if lifeline.times[pos] is None:
                        lifeline.setTime(pos, float(record[TS_FIELD]))
                    else:
                        result = lifeline
                        # start a new lifeline on dup
                        lifeline = Lifeline(record_lineid, record_groupid, 
                                          len(self.events))
                        lifeline.setTime(pos, float(record[TS_FIELD]))
                        self.lifelines.append(lifeline)
                        # remove old one
                        self.lifelines = self.lifelines[:i] + \
                            self.lifelines[i+1:]
                    matched = True
                    break
            if not matched:
                # if no match found, add new lifeline
                lifeline = Lifeline(record_lineid, record_groupid, 
                                    len(self.events))
                lifeline.setTime(pos, float(record[TS_FIELD]))
                self.lifelines.append(lifeline)
        return result

class Lifeline:
    def __init__(self, lineid, groupid, n):
        self.lineid = lineid
        self.groupid = groupid
        self.times = [ ]
        for i in xrange(n):
            self.times.append(None)

    def setTime(self, pos, t):
        self.times[pos] = t

