"""
Loader configuration module.
"""
__author__ = 'Dan Gunter <dkgunter@lbl.gov>'
__rcsid__ = '$Id: loaderConfig.py 914 2008-08-12 16:02:09Z dang $'

import glob
import imp
from logging import INFO, DEBUG
import os
import re
import sys
from warnings import warn
#
from netlogger.configobj import ConfigObj, Section
from netlogger import nllog, util
from netlogger.util import IncConfigObj, ConfigError
from netlogger.analysis import loader

log = nllog.NullLogger()
def activateLogging(name=__name__):
    global log
    log = nllog.getLogger(name)

def _strOrList(item):
    if isinstance(item, str):
        return [item]
    return item

def _testopen(filename, mode='r'):
    f = None
    try:
        f = file(filename, mode)
    except IOError, E:
        if os.path.isfile(filename):
            log.warn("open('%s') failed: %s" % (filename, E))
    return f

class Configuration(IncConfigObj):
    # section names
    INPUT = 'input'
    DB = 'database'
    DB_PARAM = 'parameters'
    LOGGING = 'logging'
    GLOBAL = 'global'
    # variables
    BASE = 'filename'
    IS_NUM = 'numbered_files'
    DELETE = 'delete_old_files'
    MOVE_TO = 'move_files_dir'
    MOVE_SFX = 'move_files_suffix'
    DB_BATCH = 'batch'
    DB_URI = 'uri'
    DB_CREATE = 'create'
    DB_UNIQUE = 'unique'
    DB_SCHEMA_FILE = 'schema_file'
    DB_SCHEMA_INIT = 'schema_init'
    DB_SCHEMA_FIN = 'schema_finalize'
    STATE_FILE = 'state_file'
    STATE_FILE_DEFAULT = '/tmp/netlogger_loader_state'

    def __init__(self, path_or_lines):
        global log
        try:
            IncConfigObj.__init__(self, path_or_lines, file_error=True,
                                  interpolation='Template')
        except SyntaxError, E:
            raise ConfigError("Illegal configuration syntax: %s" % E)
        except IOError, E:
            raise  ConfigError("Error reading config file: %s" % E)
        log = nllog.getLogger("netlogger.analysis.loader.config")
        # global section
        global_section = self._section(self.GLOBAL, {})
        self.state_file = global_section.get(self.STATE_FILE, 
                                             self.STATE_FILE_DEFAULT)
        if self.state_file == 'None':
            self.state_file = None
        # input files section
        input_section = self._section(self.INPUT, {self.IS_NUM:False,
                                                   self.DELETE:False})
        try:
            self.base = input_section[self.BASE]
        except KeyError:
            raise  ConfigError("Missing required '%s' in %s section" %
                                    (self.BASE, self.INPUT))
        self.numbered = input_section.as_bool(self.IS_NUM)
        self.eof_event = self.numbered # one implies the other!
        self.delete = input_section.as_bool(self.DELETE)
        if self.numbered and not self.delete:
            self.moveto_dir, self.moveto_suffix = None, None
            if input_section.has_key(self.MOVE_TO):
                self.moveto_dir =  input_section[self.MOVE_TO]
            elif input_section.has_key(self.MOVE_SFX):
                self.moveto_suffix = input_section[self.MOVE_SFX]
            else:
                raise  ConfigError("either '%s' or '%s' is required "
                                        "when %s is False" % (
                        self.MOVE_TO, self.MOVE_SFX, self.DELETE))
        # database section
        db_section = self._section(self.DB, {self.DB_BATCH:100, 
                                             self.DB_CREATE:True})
        try:
            uri = db_section[self.DB_URI]
        except KeyError:
            raise  ConfigError("Missing required '%s' in %s section" %
                                    (self.DB_URI, self.DB))
        try:
            scheme, dsn = self._splitURI(uri)
        except ValueError,E:
            raise  ConfigError("Parsing URI %s: %s" % (uri, E))
        self.db_dsn = dsn
        self.db_module = self._getModule(scheme)
        if self.db_module is None:
            raise  ConfigError("database module '%s' not in available: %s" 
                                    % (scheme, loader.AVAIL_DB))
        self.batch = db_section.as_int(self.DB_BATCH)
        create = db_section.get(self.DB_CREATE, '0')
        self.create = int(create)
        self.unique = util.as_bool(db_section.get(self.DB_UNIQUE, 'True'))
        self.schema_file = db_section.get(self.DB_SCHEMA_FILE)
        keys = db_section.get(self.DB_SCHEMA_INIT, "")
        self.schema_init_keys = self._commaList(keys)
        keys = db_section.get(self.DB_SCHEMA_FIN, "")
        self.schema_finalize_keys = self._commaList(keys)
        # database parameters sub-section
        self.db_param = db_section.get(self.DB_PARAM, {})        
        # logging section
        nllog.configureLogging(self)

    def _commaList(self, s):
        if isinstance(s, str):
            return (s,)
        return s
        
    def _section(self, name, defaults):
        section = self.get(name, {})
        for k,v in defaults.items():
            section.setdefault(k, v)
        return section

    def _splitURI(self, uri):
        parts = uri.split('://',1)
        if len(parts) == 1:
            raise ValueError("bad format, expected <db-module>://<DSN>")
        return parts

    def _getModule(self, scheme):
        return loader.DB_MODULES.get(scheme, None)

    def dump(self):
        s = "input: "
        s += ','.join(["%s=%s" % (a, getattr(self,a))
                       for a in ('base', 'numbered', 
                                 'eof_event', 'delete')])
        s += "\ndatabase: "
        s += ','.join(["%s=%s" % (a, getattr(self,a))
                       for a in ('db_module', 'db_dsn', 'batch', 'create')])
        return s
