from __future__ import with_statement
"""
Schema for the NetLogger database loader.

Direct access to the schema is discouraged, instead use the
module functions getSchema() and getCreateStatements().
"""
__rcsid__ = "$Id: schema.py 1039 2008-09-15 22:44:44Z dang $"
__author__ = 'Dan Gunter'

# Imports
from logging import DEBUG
# Local imports
from netlogger.configobj import ConfigObj
from netlogger import nllog

class DBStatements:
    """Execute pre-load (init) and post-load (finalize) 
    statements for a given database type.

    The statements to run are configured from a file.

    The 'type' of the database should match the section names in
    the configuration file, and then the 'init' and 'finalize'
    operations will use SQL statements listed under the
    'init' and 'finalize' values in that section. Both init and
    finalize should be present, even if not desired -- in this
    case, just use an empty string.

    For example (in the file, last double-quote wouldn't be escaped):
          [fake_db]
          init = ""\" create table if not exists event (
                 id integer primary key
                 ,hash char(32) UNIQUE NOT NULL
                 ,time double precision NOT NULL""\"
          finalize = ""
    """
    # Names of constants
    INDEX_PFX_LEN = 'INDEX_PFX_LEN'
    NAME_MAX = 'NAME_MAX'
    VALUE_MAX = 'VALUE_MAX'
    def __init__(self, infile, type="mysql", execute_fn=None):        
        """Configure for type from infile.

        If there is no section for 'type', raise a KeyError
        """
        self.log = nllog.getLogger("netlogger.loader")
        self.type = type
        self._exec = execute_fn
        self._exec_id = 0 # identifier for execution attempts
        # Read/parse configuration file.
        # The "template" interpolation style uses $Value not %(Value)s
        self._c = ConfigObj(infile, interpolation="template")
        try:
            # make sure there is a section matching 'type'
            self._section = self._c[self.type]
        except KeyError:
            raise KeyError("non-existent section '%s'" % self.type)
        # internalize some required integer constants
        self._const = { }
        for key in self.INDEX_PFX_LEN, self.NAME_MAX, self.VALUE_MAX:
            if not self._c.has_key(key):
                raise KeyError("Missing required constant '%s'" % key)
            self._const[key] = int(self._c[key])
        # make sure finalize/init are idempotent
        self._did_init, self._did_fin = False, False

    def getConstant(self, name):
        return self._const[name]

    def setExecuteFunc(self, execute_fn):
        """Set the function to call to execute a statement.
        This will replace any existing value.
        """
        self._exec = execute_fn

    def init(self, keywords=[]):
        """Initialize by calling execute_fn with one or 
        more statements appropriate for the DB type and initialization
        type.

        For the special empty tuple, the first initialization
        type found in the file will be used.

        If no initialization statements of that type exist,
        a KeyError will be raised.

        Errors executing the statements are simply propagated.
        """
        with nllog.logged(self.log, event="schema.init"):
            if not self._did_init:
                init_type = self._getType(keywords)
                with nllog.logged(self.log, level=DEBUG, event="schema.init.execall"):
                    self._executeAll(self._getStatements("init", init_type))
                self._did_init = True

    def finalize(self, keywords=[]):
        """Finalize (at close) by calling execute_fn with one or 
        more statements appropriate for DB type and finalization
        type.

        This function behaves just like init().
        """
        if self._did_fin: return
        finalize_type = self._getType(keywords)
        self._executeAll(self._getStatements("finalize", finalize_type))
        self._did_fin = True

    def _getType(self, keywords):
        """Return name of init/finalize type from a series of string
        args, e.g. ('index', 'unique').

        This essentially encodes the naming convention for the variables
        in each subsection.
        """
        return '_'.join(sorted(keywords))

    def _executeAll(self, stmt_list):
        """Execute all statements in stmt_list.
        """
        # filter out empty strings that come from
        # blank lines at the beginning or end
        statements = ["%s;" % s for s in filter(lambda x: x.strip(), stmt_list)]
        self._exec_id = self._exec_id + 1
        if self.log.isEnabledFor(DEBUG):
            self.log.debug("schema.executeAll.start", id=self._exec_id, statements=';'.join(statements))
        map(self._exec, statements)
        if self.log.isEnabledFor(DEBUG):
            self.log.debug("schema.executeAll.end", id=self._exec_id)

    def _getStatements(self, subsection, name):
        s = self._section.get(subsection)
        if s is None:
            raise KeyError("non-existent subsection '%s'" % subsection)
        if name:
            s = s.get(name)
            if s is None:    
                raise KeyError("non-existent type '%s' "
                               "in subsection '%s'" % (
                        name, subsection))
        else: 
            # try to use first one
            name_list = list(s)
            if not name_list:
                raise KeyError("cannot choose default type because "
                               "there are no types in '%s'" % subsection)
            s = s.get(name_list[0])
        return s.replace('\n', ' ').split(';')
