"""
Module with functions/classes for matching .start and .end events
"""
__author__ = "Dan Gunter (dkgunter (at) lbl.gov)"
__rcsid__ = "$Id$"

import sys
#
from netlogger import nllog
from netlogger.parsers.base import parseDate

# Logging
log = nllog.NullLogger()
def activateLogging(name=__name__):
    global log
    log = nllog.getLogger(name)

class StartEndMatcher:
    """Class with logic for matching .start/.end events
    with a fixed set of identifying fields.
    
    Expected usage is something like the following::
    
        m = StartEndMatcher(idlist=('event','guid'), max_time=3600)
        for event in event_source:
            if not m.add(event):
                # not a start/end...
            if len(m) > 0:
                for start, end, key in m.getResults():
                    if start and not end:
                        # missing end..
                    elif end and not start:
                        # missing start..
                    else:
                        # OK.. 
                
    """
    # modify these for different suffixes
    START, END = ".start", ".end"
    # ..and this for a different keyword for the event name (type)
    EVENT = "event"
    # ..and this for the timestamp
    TIME = "ts"
    
    def __init__(self, idlist=(), max_time=None, null='#', scan=100):
        """Match .start and .end events using the 'idlist' fields.
        If not present, substitute 'null' for a value.
        Every 'scan' events, scan for things older than 'max_time'
        seconds ago.
        """
        self._idlist = idlist
        self._maxt = max_time
        self._null = null
        self._scan_num = max(scan,1)
        self._lastt, self._num = 0, 0
        self._ids = { }
        self._result = [ ]
        
    def add(self, value_dict):
        """Add an event, represented as a Python dictionary.
        Returns True if the event was a start or end, False otherwise.
        """
        event = value_dict[self.EVENT]
        is_start = event.endswith(self.START)
        if not is_start and not event.endswith(self.END):
            return False
        if self._num == self._scan_num:
            self.scan()
            self._num = 0
        else:
            self._num += 1
        tm = value_dict[self.TIME]
        if isinstance(tm, str):
            tm = value_dict[self.TIME] = parseDate(tm)
        ekey = self._getKey(value_dict, event, is_start)
        if is_start:
            self._addStart(value_dict.copy(), ekey)
        else:
            self._addEnd(value_dict.copy(), ekey)
        self._lastt = tm
        return True
            
    def _search(self, unfinished, idx):
        match = None
        for i, e in enumerate(unfinished):
            if e[idx] is not None:
                match = i
                break
        return match
                    
    def _addStart(self, value_dict, key):
        # check if this key already has unfinished events
        if self._ids.has_key(key):
            # look for 'end' event
            unfinished = self._ids[key]
            match = self._search(unfinished, 1)
            # if no 'end' event, add this event to the list
            if match is None:
                unfinished.append([value_dict, None]) 
            # otherwise, remove that 'end' event      
            else:
                self._result.append((value_dict, unfinished[match][1], key))
                self._ids[key] = unfinished[:match] + unfinished[match+1:]
        # if not, initialize it with this one
        else:
            self._ids[key] = [[value_dict, None]]

    def _addEnd(self, value_dict, key):
        # check if key has unfinished events
        if self._ids.has_key(key):
            # look for 'start' event
            unfinished = self._ids[key]
            match = self._search(unfinished, 0)            
            # if no 'start', add this event to list
            if match is None:
                unfinished.append([None, value_dict])
            else:
                self._result.append((unfinished[match][0], value_dict, key))
                if len(unfinished) == 1:
                    del self._ids[key]
                else:
                    self._ids[key] = unfinished[:match] + unfinished[match+1:]
        # if not, initialize with this one
        else:
            self._ids[key] = [[None, value_dict]]
    
    def scan(self):
        """Scan entire set of saved events, looking for the aged and weak.
        """
        if self._maxt is None: # no timeout
            return
        tgt = self._lastt - self._maxt
        remove_list = [ ]
        for k, eventlist in self._ids.items():
            keep = [ ]
            for i, start_end in enumerate(eventlist):
                t =  start_end[start_end[0] is None][self.TIME]
                if t < tgt:
                    self._result.append((start_end[0], start_end[1], k))                    
                else:
                    keep.append(start_end)
            if len(keep) < len(eventlist):
                if len(keep) > 0:
                    self._ids[k] = keep
                else:
                    remove_list.append(k)
        for k in remove_list:
            del self._ids[k]
        self._num = 0 # in case this is not called from add()

    def flush(self):    
        """Flush all 'unfinished' events to the result.
        """
        for k,v in self._ids.items():
            #print "flush",k,'=',v
            for start, end in v:
                self._result.append((start, end, k))
        self._ids = { }
        self._num = 0
                
    def __len__(self):
        """For convenience, act like a list for the purposes of
        determining how many results are stored.
        """
        return len(self._result)
        
    def getResults(self):
        """Get all results at once, clearing for next event.
        Return value is a list of triples: (start-event, end-event, key),
        where either 'start-event' or 'end-event' (but not both) may be None.
        """
        r = self._result
        self._result = [ ]
        return r
                
    def _getKey(self, d, event, is_start):
        key = [ ]
        for _id in self._idlist:
            if _id == self.EVENT:
                n = len((self.END, self.START)[is_start])
                key.append(event[:-n])
            else:
                key.append(d.get(_id, self._null))
        return tuple(key)
