"""
Do an end-to-end test of the nl_loader and
nl_parser interaction.
"""
import os
import signal
import sqlite3
import subprocess
from subprocess import Popen, PIPE
import sys
import tempfile
import threading
import time
import unittest
try:
    import testoob
except ImportError:
    testoob = None
#
import netlogger.nlapi
import testBase
from netlogger.util import rm_rf

class TestCase(testBase.BaseTestCase):

    LOGGING_CONF = """
[logging]
[[loggers]]
[[[netlogger]]]
level=DEBUG
handlers=hand1
qualname=netlogger
propagate=0
netlogger = yes
[[[program]]]
level=DEBUG
handlers=hand1
qualname="netlogger.%(prog)s"
propagate=0
[[handlers]]
[[[hand1]]]
class="FileHandler"
level=DEBUG
args="('%%(logging_dir)s/%(prog)s.log', 'a')"
"""
    PARSER_CONF = """# Parser configuration
[global]
files_root = %(log_dir)s
output_file = %(parser_output_file)s
state_file = %(base_dir)s/nl_parser.state
eof_event = True
tail = True
rotate = 1
[foo1]
[[bp]]
files = *.log""" + \
LOGGING_CONF % {'prog':'nl_parser'}

    LOADER_CONF = """# Loader configuration
[global]
state_file = None
[input]
numbered_files = yes
# delete_old_files = yes
move_files_suffix = ".DONE"
filename = %(parser_output_file)s
[database]
uri = %(db_module)s://%(db_dsn)s
[[parameters]]
database = %(db_database)s""" +  \
LOGGING_CONF % {'prog':'nl_loader'}

    def setUp(self):
        """Create and populate temporary directories."""
        # pick temporary directory
        stamp = time.strftime("%Y%m%d%H%M%S",time.gmtime(time.time()))
        name = "%s-%s" % (os.path.basename(sys.argv[0]), stamp)
        self.BASE_DIR = os.path.join('/', 'tmp', name)
        os.mkdir(self.BASE_DIR)
        #  raw log directory
        self.LOG_DIR = os.path.join(self.BASE_DIR, 'logs')
        os.mkdir(self.LOG_DIR)
        # parsed log directory
        self.PARSED_DIR = os.path.join(self.BASE_DIR, 'parsed_logs')
        os.mkdir(self.PARSED_DIR)
        # configuration directory
        self.CONF_DIR = os.path.join(self.BASE_DIR, 'etc')
        os.mkdir(self.CONF_DIR)
        # PID file directory
        self.PID_DIR = os.path.join(self.BASE_DIR, 'var')
        os.mkdir(self.PID_DIR)
        # Logs of program activity directory
        self.LOGGING_DIR = os.path.join(self.BASE_DIR, 'var', 'log')
        os.mkdir(self.LOGGING_DIR)

    def tearDown(self):
        """Delete temporary files."""
        if self.DEBUG:
            self.debug_("will NOT remove %s" % self.BASE_DIR)
        else:
            rm_rf(self.BASE_DIR)
        try:
            os.unlink('/tmp/nl_parser.exit')
        except OSError:
            pass
        
    def testDaemon(self):
        """Test both the loader and parser running as daemons.
        """
        self.runPipeline(3, True, 'sqlite')

    def NotestForeground(self):
        """Test both the loader and parser running in the foreground.
        """
        self.runPipeline(3, False, 'sqlite')

    def runPipeline(self, n, daemon_mode, db_mod):
        """Run Loader and Parser pipeline.

        'n' is the number of total iterations for rolling over files, etc.
        'daemon_mode' is a flag for whether to run them both as a daemon
        'db_mod' is the name of the database module
        """
        # figure out DSN
        if db_mod == 'test':
            dsn = ''
        elif db_mod == 'sqlite':
            dsn = os.path.join(self.BASE_DIR, 'sqlite.dat')
        else:
            self.fail("unknown database module '%s'" % db_mod)
        db_database = 'test_pipeline'
        # create config files
        # ~ config file parameter dictionary
        p = { 'base_dir':self.BASE_DIR,
              'log_dir': self.LOG_DIR , 'logging_dir':self.LOGGING_DIR,
              'db_module' : db_mod, 'db_database' : db_database,
              'db_dsn' : dsn } 
        # ~ create parser config file
        p['parser_conf_file'] = os.path.join(self.CONF_DIR, "parser.conf")
        f = file(p['parser_conf_file'], "w")
        p['parser_output_file'] = os.path.join(self.PARSED_DIR, 'bp.log')
        f.write(self.PARSER_CONF % p)
        f.close()
        # ~ create loader config file
        p['loader_conf_file'] = os.path.join(self.CONF_DIR, "loader.conf")
        f = file(p['loader_conf_file'], "w")
        f.write(self.LOADER_CONF % p)
        f.close()
        # put a log file in logs
        self.newLogFile(0)
        # run the loader
        cmd = testBase.scriptPath('nl_loader')
        if daemon_mode:
            loader_pid_file = os.path.join(self.PID_DIR, "nl_loader.pid")
            args = ['-d', loader_pid_file, '-c',  p['loader_conf_file']]
        else:
            args = ['-c',  p['loader_conf_file']]
            loader_pid_file = None
        self.debug_("nl_loader command + args: %s %s" % (cmd, ' '.join(args)))
        loader_pid = subprocess.Popen([cmd] + args).pid
        # run the parser
        cmd = testBase.scriptPath('nl_parser')
        if daemon_mode:
            parser_pid_file = os.path.join(self.PID_DIR, "nl_parser.pid")
            args = ['-d', parser_pid_file, '-c',  p['parser_conf_file']]
        else:
            args = ['-c',  p['parser_conf_file']]
            parser_pid_file = None
        self.debug_("nl_parser command + args: %s %s" % (cmd, ' '.join(args)))
        parser_pid = subprocess.Popen([cmd] + args).pid
        # Need lots of time to go daemonic
        time.sleep(3)
        for i in xrange(1, n+1):
            # let the whole system run for a bit
            time.sleep(0.5)
            # <debugging> look at process table
            #proc = subprocess.Popen("ps auxwww | grep nl_", shell=True)
            #sts = os.waitpid(proc.pid, 0)
            # </debugging>
            # send a kill -USR1 to the parser so it rotates
            # ~ this makes sure the next set of events is in a a new log
            #   i.e. so the loader must advance to see them
            if daemon_mode:
                self.debug_("kill USR1 daemon parser")
                self._killpid(parser_pid_file, signal.SIGUSR1)
            else:
                self.debug_("kill USR1 non-daemon parser")
                os.kill(parser_pid, signal.SIGUSR1)
            # Add a new log file
            if i < n:
                self.newLogFile(i)
            # send a kill -HUP to the parser
            # ~ this makes sure it 'sees' the new file
            n_open = self._countOpenFiles()
            if daemon_mode:
                self.debug_("send HUP to daemn parser")
                self._killpid(parser_pid_file, signal.SIGHUP)
            else:
                self.debug_("send HUP to non-daemon parser")
                os.kill(parser_pid, signal.SIGHUP)
            n_open2 = self._countOpenFiles()
            self.assert_(n_open == n_open2, "File descriptor leak")
        # let the whole system run for a bit more
        time.sleep(1)
        # kill the parser
        self.debug_("kill nl_parser")
        if daemon_mode:
            self._killpid(parser_pid_file, signal.SIGTERM)
        else:
            os.kill(parser_pid, signal.SIGTERM)
        # give the loader a second to finish getting the
        # last few events
        time.sleep(2)
        # kill the loader
        self.debug_("kill nl_loader")
        if daemon_mode:
            self._killpid(loader_pid_file, signal.SIGTERM)
        else:
            os.kill(loader_pid, signal.SIGTERM)
        # check that the DB has the events e0 .. e(N-1)
        time.sleep(2)
        if db_mod == 'sqlite':
            self.verifySqlite(dsn, n)

    def _countOpenFiles(self):
        import re
        try:
            output = Popen("lsof", stdout=PIPE).communicate()[0]
        except OSError:
            self.debug_("lsof command failed")
            # always return 0
            return 0
        n = 0
        pat = self.LOGGING_DIR + ".*\.log"
        for line in output:
            if re.search(pat, line) is not None:
                n += 1
        return n

    def verifySqlite(self, filename, n):
        """Verify results in an SQLite file 'filename' 
        from a runPipeline() that had 'n' iterations.
        """
        for i in xrange(10):
            conn = sqlite3.connect(filename)
            self.assert_(conn)
            self.debug_("connected to %s" % filename)
            c = conn.cursor()
            c.execute("select distinct(name) from event")
            x = [row for row in c.fetchall()] 
            if len(x) == n:
                break
            time.sleep(1)
        self.assert_(len(x) == n, "wrong num. events in database, "
                     "got %d expected %d" % (len(x), n))
        for i, row in enumerate(x):
            e = row[0]
            expect = "e%d" % i
            self.assert_(e == expect, "bad event name %s, expected %s" % (e, expect))

    def _killpid(self, pidfile, signo):
        if pidfile is None: return
        try:
            f = file(pidfile)
        except IOError, E:
            self.debug_("could not open pid file %s" % pidfile)
            return
        s = f.readline().strip()
        if not s:
            self.debug_("pid file %s is empty" % pidfile)
        else:
            pid = int(s)
            self.debug_("sending signal %d to pid %d" % (signo,pid))
            os.kill(pid, signo)

    def newLogFile(self, i):
        name = "out%d.log" % i
        path = os.path.join(self.LOG_DIR, name)
        f = file(path, "w")
        log = netlogger.nlapi.Log(f, flush=True)
        event = "e%d" % i
        for i in xrange(20):
            log.write(event, num=i)

# Standard unittest goop


# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
