"""
Unittests for netlogger.analysis.loaderConfig
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testLoaderConfig.py 836 2008-06-19 00:17:36Z dang $'

import os
import sys
import unittest
#
from netlogger.analysis import loader, loaderConfig
import testBase

class TestCase(testBase.BaseTestCase):
    TMPDIR = "/tmp/netlogger-loader-test"
    CONFIGS = [
        ('simple.cfg', """
        [input]
        filename = /tmp/foo.log
        
        [database]
        uri = sqlite:///tmp/some-file.sqlite
        """),
        ('simple-mysql.cfg', """
        [input]
        filename = /tmp/foo.log
        
        [database]
        uri = mysql://localhost
        """),
        ('bad1.cfg', """
        [disaster]
        # no input section
        foo = bar
        << argh >>
        """),
        ('bad2.cfg', """
        [input]
        # this is ok
        filename = /tmp/bar
        numbered_files = yes
        delete_old_files = true
        [database]
        # but this uri is wrong
        uri = foo://no-such-module
        """),
        ]
    def setUp(self):        
        self.has_mysql = loader.DB_MODULES['mysql']
        if not os.path.isdir(self.TMPDIR):
            os.mkdir(self.TMPDIR)
        for filename, contents in self.CONFIGS:
            ofile = file(self.tmp(filename), "w")
            ofile.write(contents % dict(TMPDIR=self.TMPDIR))

    def tmp(self, filename):
        return os.path.join(self.TMPDIR, filename)

    def tearDown(self):
        for filename in os.listdir(self.TMPDIR):
            os.unlink(self.tmp(filename))
        os.rmdir(self.TMPDIR)

    def _load(self, config_file):
        """Load a config file."""
        path = self.tmp(config_file)
        cfg = loaderConfig.Configuration(path)
        self.debug_("%s configuration:\n%s" % (config_file, cfg.dump()))

    def testSimpleSqliteConfig(self):
        """Load a simple sqlite configuration.
        """
        self._load('simple.cfg')

    def testSimpleMySQLConfig(self):
        """Load a simple MySQL configuration.
        """
        if self.has_mysql:
            self._load('simple-mysql.cfg')
        else:
            self.debug_("MySQL module not found, skipping test")

    def testBadConfig(self):
        """Make sure the proper error occurs on a bad config.
        """
        try:
            self._load('bad1.cfg')
            self.fail("Bad config (1) passed")
        except loaderConfig.ConfigError, E:
            self.debug_("config error caught: %s" % E)
        try:
            self._load('bad2.cfg')
            self.fail("Bad config (2) passed")
        except loaderConfig.ConfigError, E:
            self.debug_("config error caught: %s" % E)
        try:
            self._load('/a/file/that/does/not/exist')
            self.fail("Non-existent file passed")
        except loaderConfig.ConfigError, E:
            self.debug_("config error caught: %s" % E)            


# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
