#!/usr/bin/env python
"""
Test basic, common functionality from each of the scripts,
e.g. that it has at least a working -h option.
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testScripts.py 1039 2008-09-15 22:44:44Z dang $'

import re
import tempfile
import testBase

sample_log = """
2008/04/22 16:15:10 DEBUG> Request.pm:75 perfSONAR_PS::Request::parse - ts=2008-04-22T23:15:10.266296Z event=org.perfSONAR.Services.MA.clientRequest.start guid=e47c1ec9-10c1-11dd-9586-000f1f6fc256
2008/04/22 16:15:13 DEBUG> SNMP.pm:734 perfSONAR_PS::Services::MA::SNMP::handleEvent - ts=2008-04-22T23:15:13.146751Z event=org.perfSONAR.Services.MA.handleEvent.end guid=e47c1ec9-10c1-11dd-9586-000f1f6fc256
2008/04/22 16:15:13 DEBUG> Request.pm:248 perfSONAR_PS::Request::finish - ts=2008-04-22T23:15:13.286661Z event=org.perfSONAR.Services.MA.clientRequest.end guid=e47c1ec9-10c1-11dd-9586-000f1f6fc256

"""
sample_log2 = """ ts=2008-04-22T23:15:10.266296Z event=org.perfSONAR.Services.MA.clientRequest.start guid=e47c1ec9-10c1-11dd-9586-000f1f6fc256
 ts=2008-04-22T23:15:13.286661Z event=org.perfSONAR.Services.MA.clientRequest.end guid=e47c1ec9-10c1-11dd-9586-000f1f6fc256
"""

class TestCase(testBase.BaseTestCase):
    scripts = ['nl_cfgmon', 'nl_check', 'nl_cpuprobe', 'nl_csv',
               'nl_date', 'nl_findbottleneck', 'nl_findmissing',
               'nl_ganglia', 'nl_interval', 'nl_loader', 'nl_parser',
               'nl_query', 'nl_tbl', 'nl_view', 'nl_wflowgen', 'nl_write']
    
    def makeLog(self, log):
        f = tempfile.NamedTemporaryFile()
        f.write(log)
        return f

    def testHelp(self):
        """Test -h/--help for each script.
        """
        for script in self.scripts:
            self.program = script  
            self.debug_("\nscript: %s" % self.program)
            self.cmd(['-h'], 'wait')
            self.cmd(['--help'], 'wait')

    def testNLView(self):
        """Test nl_view on some input.
        """
        self.program = 'nl_view'
        f = self.makeLog(sample_log)
        self.cmd(['-diIgmt','--namespace=org.perfSONAR.', f.name], 'wait')
        
    def testNLCsv(self):
        """Test nl_csv on some input.
        """
        self.program = 'nl_csv'
        f = self.makeLog(sample_log)
        self.cmd(['-diIgmt','--namespace=org.perfSONAR.', f.name], 'wait')

    def testNLParserVanilla(self):
        """nl_parser
        """
        self.program = 'nl_parser'
        f = self.makeLog(sample_log)
        self.cmd(['-m','bp', f.name], 'wait')

    def testNLParserUnparsed(self):
        """nl_parser, unparsed data
        """
        self.program = 'nl_parser'
        # take 1: all should be parseable
        f = self.makeLog(sample_log2)
        f.flush()
        f2 = tempfile.NamedTemporaryFile()
        self.cmd(['-m','bp', '-u', f2.name, f.name], 'wait')
        lines = f2.readlines()
        self.assert_(len(lines) == 0, "%d bogus unparsed lines" % len(lines))
        # take 2: unparseable ones
        f = self.makeLog(sample_log2)
        for i in range(10):
            f.write("can't parse me %d\n" % i)
        f.flush()
        f2 = tempfile.NamedTemporaryFile()
        self.cmd(['-m','bp', '-u', f2.name, f.name], 'wait')
        lines = f2.readlines()
        self.assert_(len(lines) > 0, "should be unparsed lines")
        for i, line in enumerate(f2):
            self.assert_(line == "can't parse me %d\n" % i)
        
    def testNLNotify(self):
        """nl_notify
        """
        self.program = 'nl_notify'
        from_user =  'user@otherhost.org'
        to_user =  'user@somehost.com'
        prog = 'false'
        args = ['--from',  from_user, '--to', to_user, '--test', prog]
        self.debug_("Args=%s" % args)
        self.cmd(args, action='communicate')
        s = self.cmd_stdout.split('\n')
        self.debug_("notify output: %s" % s)
        self.assert_(re.match("To:.*%s" % to_user, s[1]))
        self.assert_(re.match("From:.*%s" % from_user, s[2]))
        self.assert_(re.match("Subject:.*Error.*%s" % prog, s[3]))

# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
