#!/usr/bin/env python
"""
Unittests for talktome.py
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testTalktome.py 971 2008-09-05 13:55:32Z dang $'

import os
import unittest
import testBase
import tempfile
import time
from netlogger import talktome

class TestCase(testBase.BaseTestCase):
    """Unit test cases.
    """

    def setUp(self):
        fd, self.secret_file = tempfile.mkstemp()
        fd, self.bad_secret_file = tempfile.mkstemp()

    def tearDown(self):
        os.unlink(self.secret_file)
        os.unlink(self.bad_secret_file)

    def testBasic(self):
        """Good messages are received and 'bad' ones rejected
        """
        ntot = 10
        port = 25000
        parent = talktome.Parent(port)
        bad_parent = talktome.Parent(port)
        ppid = os.getpid()
        cpid = os.fork()
        if cpid == 0:
            # child
            child = talktome.Child(port, parent.getSecret())
            cthread = talktome.ChildThread(child)
            cthread.start()
            time.sleep(1.5) # get commands later..
            cmd, nrcv = None, 0
            while cmd != 'DONE':
                cmd = cthread.nextCommand()
                if cmd is None:
                    time.sleep(0.1)
                else:
                    self.debug_("Child got command: %s" % cmd)
                    if cmd != 'DONE':
                        nrcv += 1
            self.assert_(nrcv == ntot/2, "Number of received messages, %d,"
                         " does not match number expected, %d" % 
                         (nrcv, ntot/2)) 
            # avoids duplicate test result on success
            os._exit(0)
        else:
            # parent
            time.sleep(0.75) # wait for child to start
            for i in xrange(ntot):
                if i%2:
                    self.debug_("send command %d" % i)
                    parent.sendCommand("TEST")
                else:
                    self.debug_("bad command %d" % i)
                    bad_parent.sendCommand("FOOZ")
                time.sleep(0.1)
            parent.sendCommand("DONE")
            self.debug_("parent is done")
            # join child
            os.wait()

    def testConvenience(self):
        """Convenience functions
        """
        ntot = 10
        sender = talktome.initSender(self.secret_file)
        bad_sender = talktome.initSender(self.bad_secret_file)
        pid = os.fork()
        if pid == 0:
            rcvr = talktome.initReceiver(self.secret_file)
            time.sleep(1.5) # get commands later..
            cmd, nrcv = None, 0
            while cmd != 'DONE':
                cmd = rcvr.nextCommand()
                if cmd is None:
                    time.sleep(0.1)
                else:
                    self.debug_("Child got command: %s" % cmd)
                    if cmd != 'DONE':
                        nrcv += 1
            self.assert_(nrcv == ntot/2, "Number of received messages, %d,"
                         " does not match number expected, %d" % 
                         (nrcv, ntot/2)) 
            # avoids duplicate test result on success
            os._exit(0)
        else:
            # parent
            time.sleep(0.75) # wait for child to start
            for i in xrange(ntot):
                if i%2:
                    self.debug_("send command %d" % i)
                    sender.sendCommand("TEST")
                else:
                    self.debug_("bad command %d" % i)
                    bad_sender.sendCommand("FOOZ")
                time.sleep(0.1)
            sender.sendCommand("DONE")
            self.debug_("parent is done")
            # join child
            os.wait()
            
# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
