from datetime import datetime
import logging
import time
from itertools import imap, izip
from turbogears.database import PackageHub
from turbogears import config
# import some basic SQLObject classes for declaring the data model
# (see http://www.sqlobject.org/SQLObject.html#declaring-the-class)
from sqlobject import SQLObject, SQLObjectNotFound, RelatedJoin
# import some datatypes for table columns from SQLObject
# (see http://www.sqlobject.org/SQLObject.html#column-types for more)
from sqlobject import StringCol, UnicodeCol, IntCol, DateTimeCol
from turbogears import identity

__connection__ = hub = PackageHub('cedps_ts')

# the "real" data
import warnings
try:
    import MySQLdb as mysql
    from _mysql_exceptions import Warning
    # don't print certain MySQLdb warnings
    warnings.filterwarnings("error", "Table.*already exists", Warning)
    warnings.filterwarnings("ignore", "Converting column", Warning)
except ImportError:
    import sys
    print >> sys.stderr, """Required Python MySQL DB-API module, MySQLdb,
is missing. Please install this and try again.

"""
    sys.exit(1)

# logging
log = logging.getLogger("cedps_ts.model")
log.setLevel(logging.DEBUG)

class DB:
    DATABASE = config.get("mydb.database")
    print '@@DB',DATABASE
    MAX_RESULTS = 1000
    def __init__(self):
        self.host = config.get('mydb.host', 'localhost')
        self.params = { }
        for key, param_key in (('user','user'), ('pass','passwd')):
            cfg_key = 'mydb.' + key
            self.params[param_key] = config.get(cfg_key)
            if self.params[param_key] is None:
                raise KeyError("no %s in [global] config section" % cfg_key)
        self._db = mysql
        self.conn = self._db.connect(self.host, **self.params)
        self.postConnect()
    def postConnect(self):
        c = self.conn.cursor()
        c.execute("use %s" % self.DATABASE)
    def runQuery(self, sql):
        log.debug("runQuery.start query=%s" % sql.replace('\n',' '))
        c = self.conn.cursor()
        c.execute(sql)
        r = c.fetchmany(self.MAX_RESULTS)
        log.debug("runQuery.end num_results=%d" % len(r))
        return r
    def gridftpFailedJobsQuery(self, **kw):
        _ = """
select DATE_ADD('1970-01-01', interval event.time second) as 'date', bad.value as 'id', status, dn.value
from (
    select attr_val.e_id, attr_val.name_id, attr_val.value, status
    from (
        select distinct attr_val.value, attr_val.name_id, 'error' as 'status'
        from (select event.id 
              from event 
              join event_type on event.et_id = event_type.id
              where event_type.name = 'globus-gridftp-server.transfer.error'
        ) as e
        join attr_val on e.id = attr_val.e_id
        join attr_name on attr_val.name_id = attr_name.id
        where attr_name.name = 'id'
        union
        select distinct attr_val.value, attr_val.name_id, 'hang' as 'status'
        from (select event.id
              from event 
              join event_type on event.et_id = event_type.id
              join attr_val as bytes on event.id = bytes.e_id
              join attr_name as bname on bytes.name_id = bname.id
              where event_type.name = 'globus-gridftp-server.transfer.end'
                    and bname.name = 'bytes' and bytes.value = '0'
        ) as e
        join attr_val on e.id = attr_val.e_id
        join attr_name on attr_val.name_id = attr_name.id
        where attr_name.name = 'id'
     ) as te_id
    join attr_val on (te_id.value = attr_val.value and te_id.name_id = attr_val.name_id)
) as bad
join event on event.id = bad.e_id
join event_type on event.et_id = event_type.id
join dn on event.id = dn.e_id
where ( event.time >= %(start_sec)lf and event.time <= %(end_sec)lf )
group by bad.value
""" % kw
        return ('date', 'id', 'status', 'dn'), self.runQuery(_)

    def grepQuery(self, **kw):
        
        for item in 'event', 'dn':
            if kw[item] == '' or kw[item] == '%':
                kw['%s_where' % item] = ""
            else:
                if item == 'event':
                    kw['event_where'] = (
                        "event_type.name like '%s' and " % kw[item])
                elif item == 'dn':
                    kw['dn_where'] = ("where dn.value like '%s'" % 
                                      kw[item])
#        _ = """select event.time, event_type.name, dn.value, ident.value
#            from event 
#                 join event_type on event_type.id = event.et_id
#                 left join dn on dn.e_id = event.id
#                 left join ident on ident.e_id = event.id 
#            where %(event_where)s
#                  event.time >= %(start_sec)lf and 
#                     event.time <= %(end_sec)lf 
#                   and ident.relationship = 'guid'
#            limit %(limit)d
#            """ % kw
        _ = """select event.time, event_type.name, dn.value, ident.value
            from event 
                 join event_type on event_type.id = event.et_id
                 left join dn on dn.e_id = event.id
                 left join ident on ident.e_id = event.id 
            where %(event_where)s
                  event.time >= %(start_sec)lf and 
                     event.time <= %(end_sec)lf 
            limit %(limit)d
            """ % kw
        return ('date', 'event', 'dn', 'guid'), self.runQuery(_)


# class YourDataClass(SQLObject):
#     pass


# the identity model


class Visit(SQLObject):
    """
    A visit to your site
    """
    class sqlmeta:
        table = 'visit'

    visit_key = StringCol(length=40, alternateID=True,
                          alternateMethodName='by_visit_key')
    created = DateTimeCol(default=datetime.now)
    expiry = DateTimeCol()

    def lookup_visit(cls, visit_key):
        try:
            return cls.by_visit_key(visit_key)
        except SQLObjectNotFound:
            return None
    lookup_visit = classmethod(lookup_visit)


class VisitIdentity(SQLObject):
    """
    A Visit that is link to a User object
    """
    visit_key = StringCol(length=40, alternateID=True,
                          alternateMethodName='by_visit_key')
    user_id = IntCol()


class Group(SQLObject):
    """
    An ultra-simple group definition.
    """
    # names like "Group", "Order" and "User" are reserved words in SQL
    # so we set the name to something safe for SQL
    class sqlmeta:
        table = 'tg_group'

    group_name = UnicodeCol(length=16, alternateID=True,
                            alternateMethodName='by_group_name')
    display_name = UnicodeCol(length=255)
    created = DateTimeCol(default=datetime.now)

    # collection of all users belonging to this group
    users = RelatedJoin('User', intermediateTable='user_group',
                        joinColumn='group_id', otherColumn='user_id')

    # collection of all permissions for this group
    permissions = RelatedJoin('Permission', joinColumn='group_id',
                              intermediateTable='group_permission',
                              otherColumn='permission_id')


class User(SQLObject):
    """
    Reasonably basic User definition.
    Probably would want additional attributes.
    """
    # names like "Group", "Order" and "User" are reserved words in SQL
    # so we set the name to something safe for SQL
    class sqlmeta:
        table = 'tg_user'

    user_name = UnicodeCol(length=16, alternateID=True,
                           alternateMethodName='by_user_name')
    email_address = UnicodeCol(length=255, alternateID=True,
                               alternateMethodName='by_email_address')
    display_name = UnicodeCol(length=255)
    password = UnicodeCol(length=40)
    created = DateTimeCol(default=datetime.now)

    # groups this user belongs to
    groups = RelatedJoin('Group', intermediateTable='user_group',
                         joinColumn='user_id', otherColumn='group_id')

    def _get_permissions(self):
        perms = set()
        for g in self.groups:
            perms |= set(g.permissions)
        return perms

    def _set_password(self, cleartext_password):
        """Runs cleartext_password through the hash algorithm before saving."""
        password_hash = identity.encrypt_password(cleartext_password)
        self._SO_set_password(password_hash)

    def set_password_raw(self, password):
        """Saves the password as-is to the database."""
        self._SO_set_password(password)


class Permission(SQLObject):
    """
    A relationship that determines what each Group can do
    """
    permission_name = UnicodeCol(length=16, alternateID=True,
                                 alternateMethodName='by_permission_name')
    description = UnicodeCol(length=255)

    groups = RelatedJoin('Group',
                         intermediateTable='group_permission',
                         joinColumn='permission_id',
                         otherColumn='group_id')
