"""
Perform grep-like queries on the database.
"""
__rcsid__ = "$Id$"
__author__ = 'Dan Gunter'

import logging
from netlogger import nlapi
from base import writeEvent, equalOrLike

# Logging
log = None
def activateLogging(name=__name__, **kw):
    global log
    log = nllog.getLogger(name, **kw)

def doQuery(module, conn, ofile=None, batch=50, attr=None, **kw):
    conn.execute("set time_zone = '+0:00'") # UTC
    subquery_table = _doSubquery(conn, **kw)
    params = dict( tmp=subquery_table, datefmt= '%Y-%m-%dT%H:%i:%S.%fZ')
    params['datefmt'] = params['datefmt'].replace('%', '%%')
    s = [ ]
    # EVENT table, timestamp
    s.append("""select %(tmp)s.id, 'ts', from_unixtime(ev.time, '%(datefmt)s')
              from event ev
              inner join %(tmp)s on ev.id = %(tmp)s.id""")
    # EVENT_TYPE table
    s.append("""select %(tmp)s.id, 'event', et.name
              from event_type et
              inner join %(tmp)s on %(tmp)s.et_id = et.id""")
    # EVENT table, severity
    s.append("""select %(tmp)s.id, 'level', ev.severity
               from event ev 
               inner join %(tmp)s on ev.id =  %(tmp)s.id""")
    # ATTR_NAME, ATTR_VAL tables
    s.append("""select  %(tmp)s.id, an.name, av.value 
               from attr_val av
               inner join %(tmp)s on av.e_id = %(tmp)s.id 
               inner join ATTR_NAME an on an.id = av.name_id""")
    # IDENT_NAME, IDENT_VAL tables
    s.append("""select %(tmp)s.id, idn.name, idv.value 
               from ident_val idv 
               inner join %(tmp)s on idv.e_id = %(tmp)s.id 
               inner join ident_name idn on idn.id = idv.name_id""")
     # DN table
    s.append("""select %(tmp)s.id, 'dn', dn.value from dn 
               inner join %(tmp)s on dn.e_id = %(tmp)s.id""")
    # make temporary table for event list
    event_tmp = "etmp"
    create = "create temporary table %s" % event_tmp
    create += " (id integer, name varchar(255), value varchar(255))"
    conn.execute(create)
    # populate table with select statements
    for stmt in s:
        stmt = "insert into %s " % event_tmp + stmt
        log.debug("SQL statement: %s", stmt % params)
        conn.execute(stmt % params)
    # initialize log writer and list of excluded/included attributes
    writer = nlapi.Log(logfile=ofile, level=nlapi.Level.ALL, guid=False)
    exclude, only = None, None
    if attr is not None:
        is_exclude = attr[0]
        if is_exclude:
            exclude = attr[1]
        else:
            only = attr[1]
    # write results to log writer
    cursor = conn.cursor()
    cursor.execute("select * from %s order by id" % event_tmp)
    old_eid, event = -1, { }
    while 1:
        results = cursor.fetchmany(batch)
        if not results:
            break
        for eid, name, value in results:
            #log.debug("event %s attr %s=%s", eid, name, value)
            if old_eid != eid:
                log.debug("old event id = %s, new = %s, event = %s", 
                          old_eid, eid, event)
                if event:
                    writeEvent(writer, event, exclude=exclude, only=only)
                    event = { }
                old_eid, event = eid, { }
            event[name] = value
    if event:
        writeEvent(writer, event, exclude=exclude, only=only)

def _doSubquery(conn, time=None, event=None, dn=None, depth=None,
                restrict=None, ids=None, maxresult=None):
    create = "create temporary table tmp"
    select = "select event.id, event.et_id from event"
    if maxresult != 0:
        options = "limit %d" % maxresult
    else:
        options = ""
    joins = [ ]
    rstr = [ ]
    params = { }
    if dn:
        joins.append("inner join dn on event.id = DN.e_id")
        restrictions.append(equalOrLike(dn, 'DN', 'dn'))
        params['dn'] = dn
    e_rstr = ""
    if event and event != '*':
        joins.append("inner join event_type on event.et_id = event_type.id")
        e_rstr = equalOrLike(event, 'event_type', 'event', 'name')
        rstr.append(e_rstr)
        params['event'] = event
    if time:
        t1, t2 = time
        if t2:
            t_rstr = "event.time >= %s AND event.time <= %s" % (t1, t2)
        else:
            t_rstr = "EVENT.time = %s" % t1
        rstr.append(t_rstr)
    if restrict:
        a_rstr = ""
        for i,(name,expr) in enumerate(restrict):
            joins.append("inner join attr_val av%d on event.id = av%d.e_id"
                         % (i,i))
            joins.append("inner join attr_name an%d on "
                         "av%d.name_id = an%d.id" % (i,i, i))
            if i > 0:
                a_rstr += " AND "
            a_rstr += "(an%d.name='%s' AND av%d.value %s)" % (
                i, name, i, expr)
        rstr.append(a_rstr)
    if ids:
        i_rstr = ""
        joins.append("inner join ident_val on event.id = ident_val.e_id")
        joins.append("inner join ident_name on ident_vAL.name_id = "
                     "ident_name.id")
        for name,expr in ids:
            i_rstr += ("(ident_name.name=%%(id_%s)s AND "
                       " ident_val.value %s)" % (name, expr))
            params['id_%s' % name] = name
        rstr.append(i_rstr)
    join_str, restrict_str = '', ''
    if joins:
        join_str = ' '.join(joins)
    if t_rstr or e_rstr or i_rstr or a_rstr:
        restrict_str = "WHERE " + ' AND '.join(filter(None,rstr))
    stmt = ' '.join((create, select, join_str, restrict_str, options))
    log.info("SQL statement to get event ids: %s ;; params = %s",stmt, params)
    conn.execute(stmt, params)
    # keep looking for "similar" events until depth is reached,
    # the result set doesn't expand, or the result set is too big
    maxdepth, depth = (depth, 1e9)[depth < 0], 0
    result_size, last_result_size = 0, -1
    while (depth < maxdepth and
           last_result_size < result_size < maxresult):
        if maxresult == 0:
            limit = ""
        else:
            result_space = maxresult - result_size
            limit = "limit %d" % result_space
        stmt = """create temporary table tmp2 
                  select distinct event.id, event.et_id from ident_val 
                  join tmp on ident_val.e_id = tmp.id 
                  join ident_val IV on iv.value = ident_val.value 
                   join event on event.id = iv.e_id """ + limit
# This will not match identifiers with a different name:
#                  join IDENT_VAL IV on (IV.value = IDENT_VAL.value AND 
#                       IV.name_id = IDENT_VAL.name_id) 
        log.debug("SQL (depth=%d): %s", depth, stmt)
        conn.execute(stmt)
        # 
        conn.execute("insert into tmp select * from tmp2")
        conn.execute("drop table tmp2")
        cursor = conn.cursor()
        cursor.execute("select count(id) from tmp")
        row = cursor.fetchone()
        last_result_size = result_size
        if row:
            result_size = int(row[0])
        depth += 1
    # all done; return name of temporary table
    return 'tmp'

