#!/usr/bin/env python
"""
Base class for Unittests for NLA
"""
__author__ = 'Keith Jackson KRJackson@lbl.gov'
__rcsid__ = '$Id: testBase.py 1039 2008-09-15 22:44:44Z dang $'

try:
    import cStringIO as StringIO
except ImportError:
    import StringIO
import logging
import os
import re
import sys
import signal
import subprocess
import time
from subprocess import PIPE, Popen, call
import unittest
#
from netlogger.parsers.base import NLFastParser
from netlogger.parsers.modules import dynamic

def scriptPath(script):
    p = script
    # look in local dirs first
    for d in '.', os.path.join('..','..'):
        pd = os.path.join(d, 'scripts')
        if os.path.isdir(pd):
            p = os.path.join(pd, script)
            break
    #print "found: %s" % p
    return p

def fileDir():
    return os.path.dirname(os.path.abspath(__file__))

def getNextEvent(parser):
    while 1:
        e = parser.next()
        if e is None:
            continue
        break
    return e

def isiterable(obj):
    try: 
        _ = iter(obj)
        result = True
    except TypeError:
        result = False
    return result

class BaseTestCase(unittest.TestCase):
    DEBUG = os.getenv('TEST_DEBUG')
    TRACE = 0
    try:
        x = int(DEBUG)
        if x > 1:
            TRACE = 1
    except ValueError:
        pass

    # directory for data files
    data_dir = 'data'

    def setUp(self):
        """Base setUp actions
        """
        self.stderr_save = None
        self._setDataDir()

    def captureStderr(self):
        self.stderr_save = sys.stderr
        my_stderr = StringIO.StringIO()
        sys.stderr = my_stderr
        return my_stderr

    def restoreStderr(self):
        if self.stderr_save:
            sys.stderr = self.stderr_save

    def tearDown(self):
        """Return stdout to its proper state"""
        #sys.stdout = self.stdout_save

    def log(self, msg, *args):
        if args:
            sys.stderr.write(msg % args)
        else:
            sys.stderr.write(msg)
        sys.stderr.write("\n")

    error = log

    def debug_(self, msg, *args):
        """Note: unittest.TestCase already has a debug() method
        """
        if self.DEBUG:
            self.log(msg, *args)

    def trace_(self, msg, *args):
        """Write even finer-grained debugging events
        Does nothing unless self.TRACE is True.
        """
        if self.TRACE:
            self.log(msg, *args)

    trace = trace_

    def getOutput(self):
        """Get captured stdout"""
        return self.sio.getvalue()

    def setInput(self, s):
        """Write to object and seek back to 0"""
        self.sio = StringIO.StringIO(s)

    def cmd(self, args, action='kill', should_fail=False):
        """Run command"""
        try:
            self._run(args, action)
            if should_fail:
                self.fail("unexpected success: %s" % E)
        except RuntimeError, E:
            if not should_fail:
                self.fail("failed: %s" % E)
    
    def _run(self, params, action):
        cmd = [self.program] + params
        self.debug_("run command: %s" % ' '.join(cmd))
        result = -1
        p = Popen(cmd, stdout=PIPE, stderr=PIPE)
        if action == 'kill':
            time.sleep(0.5)
            result = p.poll()
            if result is None:
                os.kill(p.pid, signal.SIGKILL)
                result = 0    
        elif action == 'wait':
            if p:
                result = p.wait()
        elif action == 'communicate':
            self.cmd_stdout, self.cmd_stderr = p.communicate()
            result = p.returncode
        else:
            raise ValueError("internal error: bad action")
        if result != 0:
            if p:
                output, errs = p.communicate()
            else:
                output, errs = '', ''
            indent = lambda s : '\n'.join(["   %s" % x 
                                           for x in s.split('\n')])
            raise RuntimeError("command '%s' failed (%d).\nOUTPUT:\n%s\n"
                               "ERRORS:\n%s" % (
                    ' '.join(cmd), result, indent(output), indent(errs)))

    def _setDataDir(self):
        if not self.data_dir.startswith('/'):
            if not os.path.isdir(self.data_dir):
                # try to prepend path to this script to basedir
                path = os.path.join(fileDir(), self.data_dir)
                if not os.path.isdir(path):
                    raise ValueError("Data directory '%s' not found in "
                                    "'%s' or current directory" % 
                                    (path, fileDir()))
                self.data_dir = path

class BaseParserTestCase(BaseTestCase):
    """Additional shared functionality for parser-module unit tests
    """
    # Base directory for data files; usually a relative path
    basedir = BaseTestCase.data_dir
    # Base filename, typically parser class name, ending with a '.'
    basename = ''
    # Subclasses should set this to their parser class
    parser_class = None

    def setUp(self):
        # For internal use
        self._nlparser = NLFastParser()
        self._dynamic_parser = False

    def checkGood(self, filename='xyz', test=None, parser_kw={},
                  num_expected=-1):
        """Check if data in filename basename + 'filename' is good by running
        the function/regex in 'test' against the results of the parser
        instantiated as 'parser_class(opened-file, **parser_kw)'.
        
        If the value of test is a list, it is interpreted as a list
        of regular expressions to match; if it is a function, it is invoked
        for each returned value as test(event, num=number-of-event).
        If it is None, then it passes for any result at all (this really
        makes sense only when num_expected is set).
        """
        if test is None:
            test = self._trueTest
        elif not callable(test) and not isiterable(test):
            self.fail("Test '%s' is not a function or list of regexes" % test) 
        parser = self._initParser(filename, parser_kw)
        n = 0
        for i, event in enumerate(parser):
            self.debug_("(%d) event: %s" % (i,event))
            if event is None:
                continue
            self._check(test, n, event)
            n += 1
        self._checkNum(n, num_expected)
        return parser

    def checkBad(self, filename=None, parser_kw={}, num_expected=-1):
        """Same as checkGood() except that there are no tests.

        The 'num_expected' parameter can be used to control how many
        good events are expected.
        """
        parser = self._initParser(parser_class, filename, parser_kw)
        n = 0
        for event in parser:
            if event is not None:
                n += 1
        self._checkNum(n, num_expected)        
        return parser

    def getFullPath(self, filename):
        self._setDataDir()
        self.basedir = self.data_dir
        return os.path.join(self.basedir, self.basename + filename)

    def setParseDynamic(self, enable, **kw):
        """Enable or disable the use of the 'dynamic' parser, which
        routes to a given parser based on a header. The assumption here
        is that the same parser_class will be the only possible parser,
        but that it will have access to header fields as well.
        This makes it easier to test how parsers behave if they are
        invoked via the dynamic parser module.
        If 'enable' is True, the keywords in '**kw' will be used when the
        dynamic parser gets instantiated (in _initParser).
        """
        if not enable:
            self._dynamic_parser = False
        else:
            self._dynamic_parser = True
            self._dynamic_parser_kw = kw

    def _initParser(self, filename, kw):
        """Open file and init parser class with it
        """
        self.assert_(self.parser_class,
                     "You forgot to set the 'parser_class' attribute in "
                     "the test-case class")
        path = self.getFullPath(filename)
        self.debug_("parsing data file: %s" % path)
        self.failUnless(os.path.isfile(path), "File not found: %s" % path)
        f = file(path)
        if self._dynamic_parser:
            instance = dynamic.Parser(f, **self._dynamic_parser_kw)
            sub_instance = self.parser_class(f, **kw)
            instance.add('foo', {}, sub_instance)
        else:
            instance = self.parser_class(f, **kw)
        return instance

    def _check(self, test, i, event):
        """Check one event
        """
        self.debug_("check event #%d: %s" % (i, event))
        if callable(test):
            test(event, num=i)
        else:
            self.assert_(re.search(test[i], event) is not None,
                         "Regex '%s' failed to match event '%s'" % (
                    test[i], event))

    def _checkNum(self, i, num_expected):
        if num_expected < 0:
            return
        self.assert_(i == num_expected, "Number of events found (%d) "
                     "does not match number expected (%d)" % 
                     (i, num_expected))

    def _trueTest(self, i, num=0):
        return True

    def parseEvent(self, line):
        """Convenience function to parse an event, e.g. as fed
        to the function provided to checkGood(), into a dictionary
        """
        return self._nlparser.parseLine(line)

def suite(clazz):
    suite = unittest.makeSuite(clazz,'test')
    return suite

def main():
    """Run either TestOOB or standard unittest"""
    try:
        import testoob
        testoob.main()
    except ImportError:
        unittest.main(defaultTest="suite")
