"""
Unittests for netlogger.parsers.config
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testNlParser.py 971 2008-09-05 13:55:32Z dang $'

import os
import sys
import tempfile
import unittest
#
from netlogger.parsers import nlparser, config
from netlogger.util import rm_rf
import testBase

class Options:
    pass

class TestCase(testBase.BaseTestCase):
    TMPDIR = "/tmp/netlogger-parser-test"

    CONFIGS = [('simple.cfg', """
TMPDIR = %(TMPDIR)s
[global]
modules_root = netlogger.parsers.modules
use_system_path = yes
state_file = $TMPDIR/nlparser-test.state
[foo1]
files_root = $TMPDIR
[[generic]]
files = *.log
[logging]
[[loggers]]
[[[all]]]
handlers=h1
level=ERROR
qualname=netlogger
[[handlers]]
[[[h1]]]
level=NOTSET
class="StreamHandler"
args="(sys.stderr,)"
formatter=f1
[[formatters]]
[[[f1]]]
format="*(message)s"
"""),
               ('output_file.cfg',"""
TMPDIR = %(TMPDIR)s
[global]
modules_root = netlogger.parsers.modules
output_file = $TMPDIR/nlparser-test.out
rotate = 1 minutes
[foo1]
files_root = $TMPDIR
[[generic]]
files = *.log
"""),]

    LOGS = [('simple_hello.log', """Hello
world
"""),
            ('nl_hello.log', """ts=2008-03-14T19:26:19.160217Z event=nlwrite.event level=Info msg="hello, world" gid=1234
ts=2008-03-14T19:26:24.872916Z event=nlwrite.event level=Info msg="hello, again" gid=1234
"""),
            ('syslog_hello.log', """
[123] INFO/foo: Hello
[99] DEBUG/netlogger: ts=2008-03-14T19:26:19.160217Z event=nlwrite.event level=Info msg="hello, world" gid=1234 syslog=yes
[456] INFO/bar: Hello
[123] INFO/foo: World
[99] INFO/NetLogger: ts=2008-03-14T19:26:24.872916Z event=nlwrite.event level=Info msg="goodbye" gid=1234  syslog=no
[99] DEBUG/NetLogger: ts=2008-03-14T19:26:24.872916Z event=nlwrite.event level=Info msg="hello, again" gid=1234 syslog=yes
[456] INFO/bar: World
"""),
]
    MORE_LOGS = [('simple_hello2.log', """Hello
world
"""),
]

    def setUp(self):
        # first, clean up from prev. runs; this is
        # important so assumptions about numbered files hold true
        if os.path.exists(self.TMPDIR):
            rm_rf(self.TMPDIR)
        if os.path.exists(config.Configuration.STATE_FILE_DEFAULT):
            os.unlink(config.Configuration.STATE_FILE_DEFAULT)
        os.mkdir(self.TMPDIR)
        for filename, contents in self.CONFIGS:
            ofile = file(self.tmp(filename), "w")
            ofile.write(contents % dict(TMPDIR=self.TMPDIR))
        for filename, contents in self.LOGS:
            ofile = file(self.tmp(filename), "w")
            ofile.write(contents)

    def addMoreLogs(self):
        for filename, contents in self.MORE_LOGS:
            ofile = file(self.tmp(filename), "w")
            ofile.write(contents)

    def removeMoreLogs(self):
        for filename, _ in self.MORE_LOGS:
            os.unlink(self.tmp(filename))

    def tearDown(self):
        rm_rf(self.TMPDIR)
        #print '@@',self.TMPDIR

    def testSaveState(self):
        """Check that the saveState() produces a good file.
        """
        path = self.tmp('simple.cfg')
        app = self.getApp(path)
        cfg = app.cfg
        app.saveState()
        f = file(cfg.state_file)
        line = f.readline().strip()
        saved_logs = [s.split()[0] for s in f]
        known_logs = [os.path.join(self.TMPDIR,x[0]) for x in self.LOGS]
        saved_logs.sort()
        known_logs.sort()
        self.failUnless(saved_logs == known_logs,
                        "expected %s, got %s" % (known_logs, saved_logs))

    def testRestoreState(self):
        """Check that restoreState() handles good files, rejects bad ones
        """
        path = self.tmp('simple.cfg')
        app = self.getApp(path)
        app.saveState()
        app.restoreState()

    def testChangingFilesState(self):
        """Check that state-file tracks existence of files.
        """
        path = self.tmp('simple.cfg')
        app = self.getApp(path)
        cfg = app.cfg
        app.saveState()
        app.flush()
        # add more logs, see if they appear
        self.debug_("add more logs")
        self.addMoreLogs()
        app = self.getApp(path)
        cfg = app.cfg
        expected_keys = map(self.tmp, 
                            [x[0] for x in self.LOGS + self.MORE_LOGS])
        found_keys = cfg.parsers.keys()
        def _compare_files(ekeys, fkeys):
            self.assert_(len(ekeys) == len(fkeys),
                     "wrong # of files: got %d but expected %d" %
                     (len(fkeys), (len(ekeys))))
            ekeys.sort() ; fkeys.sort()
            for a,b in zip(ekeys, fkeys):
                self.failUnless(a == b, "got %s, expected %s" % (b, a))
                self.debug_("match: %s" % a)
        _compare_files(expected_keys, found_keys)
        self.debug_("yes: new logs were found")
        # remove extra logs and try again
        self.debug_("remove extra logs, try again")
        self.removeMoreLogs()
        app = self.getApp(path)
        cfg = app.cfg
        expected_keys = map(self.tmp, [x[0] for x in self.LOGS])
        found_keys = cfg.parsers.keys()
        _compare_files(expected_keys, found_keys)        
        self.debug_("yes: removed logs are gone")

    def testRotate(self):
        """Test file rotation
        """
        # first try with stdout output
        path = self.tmp('simple.cfg')
        app = self.getApp(path)        
        self.assert_(app.ofile == sys.stdout, "output (%s) not stdout" % app.ofile)
        # next try with file output
        path = self.tmp('output_file.cfg')
        app = self.getApp(path)
        self.failIf(app.ofile == sys.stdout, "output is stdout")
        ofileName = lambda n: os.path.join(self.TMPDIR, 
                                           'nlparser-test.out.%d' % n)
        self.assert_(os.path.exists(ofileName(1)), 
                     "output file %s not found" % ofileName(1))
        # ~ try rotating, shouldn't do anything
        app.rotate()
        self.failIf(os.path.exists(ofileName(2)),
                     "output file %s should not be found" % ofileName(2))
        # ~ put some data in there, then try rotating again: should work
        app.ofile.write("some data\n")
        app.ofile.flush()
        app.rotate()
        self.failUnless(os.path.exists(ofileName(2)),
                        "output file %s not found" % ofileName(2))
        
    def getApp(self, path):
        options = Options()
        options.config = path
        options.progress = False
        options.noexec = True
        options.secret_file = tempfile.mkstemp()
        app = nlparser.Application(options)
        app.configure()
        return app

    def tmp(self, filename):
        return os.path.join(self.TMPDIR, filename)


# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
