"""
Unittests for nl_pipeline script
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testNlPipeline.py 971 2008-09-05 13:55:32Z dang $'

import os
import signal
import subprocess
import sys
import tempfile
import time
import unittest
#
import testBase
from netlogger.util import rm_rf

def _cp(p1, p2):
    f1 = open(p1, 'r')
    f2 = open(p2, 'w')
    for line in f1:
        f2.write(line)
    f2.close()
    f1.close()

class TestCase(testBase.BaseTestCase):
    FAKE_PROG = """#!/usr/bin/env python
import os, signal, sys, time
# get PID file after -d option
pidfile = sys.argv[sys.argv.index('-d') + 1]
# write your PID
file(pidfile, 'w').write(str(os.getpid()))
# remove pid file on signal
def onterm(signo, frame):
    if os.path.exists(pidfile): os.unlink(pidfile)
    sys.exit(0)
# setup signal handling
signal.signal(signal.SIGTERM, onterm)
# sleep for a long time
time.sleep(1000)
"""
    def setUp(self):
        self.work_dir = tempfile.mkdtemp()
        # copy 'nl_pipeline' script to work_dir
        p = testBase.scriptPath('nl_pipeline')
        self.debug_("script path = %s" % p)
        self.failUnless(os.path.isfile(p))
        self.pipeline_path = os.path.join(self.work_dir, 'nl_pipeline')
        _cp(p, self.pipeline_path)
        os.chmod(self.pipeline_path, 0700)
        # put fake nl_loader, nl_parser in work_dir
        for prog in 'nl_loader', 'nl_parser':
            prog_path = os.path.join(self.work_dir, prog)
            file(prog_path, 'w').write(self.FAKE_PROG)
            os.chmod(prog_path, 0700)
        #
        self.remove_dir = True

    def tearDown(self):
        # wipe out temp dir
        if self.remove_dir:
            rm_rf(self.work_dir)
            self.debug_("removed working dir: %s" % self.work_dir)
        else:
            self.debug_("saved working dir: %s" % self.work_dir)

    def testBasic(self):
        "Test simple invocation and kill"        
        self.remove_dir = False
        w = self.work_dir
        # run nl_pipeline from self.work_dir        
        args = " -c %s -i 10 -w 5 -l %s -p %s" % (w, w, w)
        cmd = self.pipeline_path + args
        self.debug_("run pipeline: %s" % cmd)
        retcode = subprocess.call(cmd, shell=True)
        self.debug_("pipeline returned %d" % retcode)
        self.assert_(retcode == 0, "cmd '%s' failed" % cmd)
        # now kill it
        self._killPipeline()
        # All PID files should be gone
        time.sleep(0.5)
        for tries in xrange(10):
            count = 0
            for prog in 'nl_loader', 'nl_parser', 'nl_pipeline':
                prog_path = os.path.join(w, prog + '.pid')
                if not os.path.exists(prog_path):
                    count += 1
            time.sleep(0.2)
        for prog in 'nl_loader', 'nl_parser', 'nl_pipeline':
            self.failIf(os.path.exists(prog_path),
                        "%s still exists" % prog_path)
        self.remove_dir = True

    def _killPipeline(self):
        w = self.work_dir
        pidfile = os.path.join(w, 'nl_pipeline.pid')
        infile = None
        for i in range(5):
            time.sleep(0.5)
            try:
                infile = file(pidfile)
                break
            except IOError:
                continue
        self.assert_(infile, "Can't read from PID file '%s'" % pidfile)
        pid = int(infile.read(100))
        count = 0
        os.kill(pid, signal.SIGTERM)
        time.sleep(0.5)
        while os.path.exists(os.path.join(w, 'nl_pipeline.pid')):
            self.failIf(count > 10, "nl_pipeline.pid not going away")
            count += 1
            os.kill(pid, signal.SIGTERM)
            time.sleep(0.5)

    def testBadOptions(self):
        "Handling of improper command-line options"
        for args in ([], ['-x foo'], ['-c /x -i badint -l /x -p /x'],
                     ['-c /x -i 10 -l /x -p /x']):
            cmd = self.pipeline_path + ' ' + ' '.join(args)
            if self.DEBUG:
                retcode = subprocess.call(cmd, shell=True)
            else:
                retcode = subprocess.Popen(cmd, shell=True, 
                                          stderr=subprocess.PIPE, 
                                          stdout=subprocess.PIPE)
            if retcode == 0:
                self._killPipeline()
                self.assert_(0, "cmd '%s' succeeded?" % cmd)

    def testLogging(self):
        "Logging and verbosity"
        self.remove_dir = False
        w = self.work_dir
        for verbosity in 0, 1, 2:
            vb = ' '.join(['-v'] * verbosity)
            # run nl_pipeline from self.work_dir        
            args = " -c %s -i 10 -w 5 -l %s -p %s %s" % (w, w, w, vb)
            cmd = self.pipeline_path + args
            self.debug_("running pipeline")
            retcode = subprocess.call(cmd, shell=True)
            self.assert_(retcode == 0, "cmd '%s' failed" % cmd)
            self.debug_("killing pipeline")
            self._killPipeline()
            # check that messages in log match verbosity
            counts = { 'INFO': 0, 'DEBUG': 0 }
            log_file = file(os.path.join(w, 'nl_pipeline.log'))
            for line in log_file:
                level_start = line.find('level=')
                level_end = line.find(' ', level_start)
                level_str = line[level_start + 6:level_end]
                if counts.has_key(level_str):
                    counts[level_str] += 1
            if verbosity == 0:
                self.failIf(counts['INFO'] > 0 or counts['DEBUG'] > 0,
                            "%d INFO and %d DEBUG messages; in default, "
                            "only ERROR expected")
            elif verbosity == 1:
                self.failIf(counts['DEBUG'] > 0,
                            "%d DEBUG messages; with -v, only INFO expected")
                self.failIf(counts['INFO'] == 0, "no INFO messages with -v")
            elif verbosity == 2:
                self.failIf(counts['INFO'] == 0, "no INFO messages with "
                            "-v -v")
                self.failIf(counts['INFO'] == 0, "no DEBUG messages with "
                            "-v -v")
        self.remove_dir = True

# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
