#!/usr/bin/env python
"""
Unittests for nllog.py
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testNllog.py 788 2008-06-02 17:11:49Z dang $'

import logging
import re
import sys
from StringIO import StringIO
from tempfile import mktemp, NamedTemporaryFile
import time
#
import testBase
import unittest
#
from netlogger import configobj
from netlogger import nllog
from netlogger.parsers.base import NLFastParser

class TestCase(testBase.BaseTestCase):
        
    def _is_nllogger(self, obj, msg):
        self.failUnless(isinstance(obj, nllog.NLLogger),
                        "%s: not a NLLogger instance" % msg)

    def _is_nlslogger(self, obj, msg):
        self.failUnless(isinstance(obj, nllog.NLScriptLogger),
                        "%s: not a NLScriptLogger instance" % msg)

    def testGetLogger(self):
        msg = "return from nllog.getLogger() with %s"
        # no args
        log = nllog.getLogger()
        self._is_nllogger(log, msg % "no args")
        # name
        log = nllog.getLogger("hello")
        self._is_nllogger(log, msg % "name=hello")
        
    def testGetScriptLogger(self):
        msg = "return from nllog.getScriptLogger() with %s"
        # no args
        try:
            log = nllog.getScriptLogger()
        except TypeError:
            pass
        else:
            self.fail("getScriptLogger() with no name should fail")
        # name
        log = nllog.getScriptLogger("hello")
        self._is_nlslogger(log, msg % "name=hello")

    def testVerbosityToLevel(self):
        for (vb, expected_level) in ((-2, logging.CRITICAL),
                                     (-1, logging.ERROR),
                                     (0, logging.WARN),
                                     (1, logging.INFO),
                                     (2, logging.DEBUG)):
            level = nllog.verbosityToLevel(vb)
            self.failUnless(level == expected_level)

#    def testLogged(self):
#        self.assert_(False)

    def testNLLogger(self):
        name = "testNLLogger"
        log = logging.getLogger(name)
        x = nllog.NLLogger(log)
        self._is_nllogger(x, "constructor")
        # add a handler, set the level
        tmp = NamedTemporaryFile()
        x.addHandler(logging.FileHandler(tmp.name, "w"))
        x.setLevel(logging.INFO)
        self._logSomething(x, name, tmp)

    def _logSomething(self, x, name, tmp):
        # log some things
        x.info("foo.info", name="value")
        x.warn("foo.warn", name="value")
        x.debug("foo.debug", name="value") # not logged
        # look for logs
        del x
        tmp.seek(0)
        line = tmp.readline()
        self.failUnless(line, "no log data")
        self.debug_("Log line :: %s" % line)
        for expected in  "event=%s.foo.info" % name, "name=value":
            self.failUnless(expected in line, "missing '%s'" % expected)
        line = tmp.readline()
        self.failUnless(line, "no log data")
        self.debug_("Log line :: %s" % line)
        for expected in  "event=%s.foo.warn" % name, "name=value":
            self.failUnless(expected in line, "missing '%s'" % expected)
        line = tmp.readline()
        self.failIf(line, "debug message shouldn't have been written")

    def testNLScriptLogger(self):
        name = "testNLScriptLogger"
        log = logging.getLogger(name)
        x = nllog.NLScriptLogger(log)
        self._is_nlslogger(x, "constructor")
        x.setDefaultLevel(logging.INFO)
        tmp = NamedTemporaryFile()
        x.redirectDefaultOutput(tmp)
        self._logSomething(x, name, tmp)
        
    def testNullLogger(self):
        x = nllog.NullLogger()
        x.addHandler(1)
        self.failIf(x.isEnabledFor(logging.ERROR))
        x.setLevel("")
        x.info("event", a=1, b=2)
        x.critical("event", a=1, b=2)
        x.error("event", a=1, b=2)
        x.warn("event", a=1, b=2)
        x.debug("event", a=1, b=2)
        x.trace("event", a=1, b=2)
        x.exc("event", "moo")

    def testConfigureLogging(self):
        # try with nothing
        ok = nllog.configureLogging({})
        self.failIf(ok, "empty input should not return True")
        ok = nllog.configureLogging([])
        self.failIf(ok, "empty input should not return True")
        ok = nllog.configureLogging(NamedTemporaryFile())
        self.failIf(ok, "empty input should not return True")
        # try for real
        text_tmp = NamedTemporaryFile()
        nl_tmp = NamedTemporaryFile()
        cfg_text = """
[logging]

[[loggers]]

[[[base]]]
handlers=text_file
qualname="test"

[[[text]]]
level="ERROR"
#handlers=text_file
qualname="test.text"

[[[netlogger]]]
level="DEBUG"
handlers=nl_file
qualname="test.netlogger"
propagate=0

[[handlers]]

[[[text_file]]]
class="FileHandler"
level="ERROR"
formatter="form1"
args="('%s',)"

[[[nl_file]]]
class="FileHandler"
level="NOTSET"
args="('%s',)"

[[formatters]]

[[[form1]]]
format="*(asctime)s *(levelname)s: *(message)s"
""" % (text_tmp.name, nl_tmp.name) 
        self.debug_("CONFIG: %s" % cfg_text)
        ok = nllog.configureLogging(cfg_text.split("\n"))
        self.assert_(ok, "[logging] section not found")
        nl = nllog.getLogger("test.netlogger")
        self._is_nllogger(nl, "getLogger failed")
        txt = logging.getLogger("test.text")
        base = logging.getLogger("test")
        # write data
        txt.error("text error: elmo loves you")
        txt.debug("text debug: elmo is ambivalent")
        nl.error("test.error", elmo="loves you")
        nl.debug("test.debug", elmo="ambivalent")
        base.error("base error: elmo loves you")
        base.debug("base debug: elmo is ambivalent")
        # check data
        # ~ text log
        n = 0
        for i, line in enumerate(file(text_tmp.name)):
            self.assert_( i < 2, "too many lines in log file")
            self.assert_("elmo loves you" in line)
            if i == 0:
                self.assert_("text error" in line)
            else:
                self.assert_("base error" in line)
            self.debug_("TEXTLOG: %s" % line.strip())
            n = i + 1
        self.assert_(n == 2, "too few lines in log file")
        # ~ netlogger log
        p = NLFastParser()
        for i, line in enumerate(file(nl_tmp.name)):
            self.debug_("NETLOG: %s" % line.strip())
            self.assert_( i < 2, "too many lines in log file")
            data = p.parseLine(line)
            self.debug_("NETLOG DATA: %s" % data)
            self.assert_(data['event'].startswith("test.netlogger.test."))
            t = time.time()
            self.assert_(data['ts'] <= t, "event is in the future")
            if i == 0:
                self.assert_(data['elmo'] == "loves you")
            else:
                self.assert_(data['elmo'] == "ambivalent")
            n = i + 1
        self.assert_(n == 2, "too few lines in log file")
        
# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
