#!/usr/bin/env python
"""
Unittests for parsers/modules/dynamic.py
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testParseDynamic.py 880 2008-07-18 21:56:09Z dang $'

import re
#
import testBase
import unittest
from netlogger.parsers.modules.dynamic import Parser

class FakeBaseParser:
    def setHeaderValues(self, arg):
        return

class DictParser(FakeBaseParser):
    def process(self, line):
        return ({'elmo':'loves_you'}, {'cookie-monster':'loves_cookies'})

class PrefmtParser(FakeBaseParser):
    def process(self, line):
        return ('elmo=loves_you', 'cookie-monster=loves_cookies')
        
class MixedParser(FakeBaseParser):
    def process(self, line):
        return ('elmo=loves_you', {'cookie-monster':'loves_cookies'})

class NoResultParser(FakeBaseParser):
    def process(self, line):
        return ()

class TestCase(testBase.BaseTestCase):
    HDR_PAT = "(?P<one>\S+) (?P<two>\S+) "
    ADD_PAT = {'one':re.compile('uno'), 'two':re.compile('dos')}
    def _getParsers(self, name, inst):
        NF = file('/dev/null')
        p = [Parser(NF, pattern=self.HDR_PAT),
             Parser(NF, pattern=self.HDR_PAT, show_header_groups=True),
             Parser(NF, pattern=self.HDR_PAT, show_header_groups=['one'])]
        for _ in p:
            _.add(name, self.ADD_PAT, inst)
        return p

    def _formatResults(self, raw):
        formatted = [ ]
        for r in raw:
            if isinstance(r, dict):
                formatted.append(' '.join(["%s=%s" % (k,v) for k,v in 
                                           r.items()]))
            else:
                formatted.append(r)
        return formatted

    def _checkResults(self, parser_results):
        """Note: Depends on order of parsers in _getParsers()"""
        r = parser_results[0]
        self.debug_("No header: %s" % r)
        self.assert_("elmo=loves_you" in r[0])
        self.assert_("one=uno" not in r[0])
        self.assert_("cookie-monster=loves_cookies" in r[1])
        self.assert_("one=uno" not in r[1])
        r = parser_results[1]
        self.debug_("All headers: %s" % r)
        self.assert_("elmo=loves_you" in r[0])
        self.assert_("one=uno" in r[0])
        self.assert_("two=dos" in r[0])
        self.assert_("cookie-monster=loves_cookies" in r[1])
        self.assert_("one=uno" in r[1])
        self.assert_("two=dos" in r[1])
        r = parser_results[2]
        self.debug_("First header: %s" % r)
        self.assert_("elmo=loves_you" in r[0])
        self.assert_("one=uno" in r[0])
        self.assert_("two=dos" not in r[0])
        self.assert_("cookie-monster=loves_cookies" in r[1])
        self.assert_("one=uno" in r[1])
        self.assert_("two=dos" not in r[1])
            
    def _testParser(self, clazz):
        name = clazz.__name__
        parsers = self._getParsers(name, clazz())
        results = [parser.process("uno dos tres quatro cinco cinco seis") 
                   for parser in parsers]
        formatted = map(self._formatResults,results)
        self.debug_("Output for Parser %s:" % name)
        self._checkResults(formatted)

    def testDict(self):
        self._testParser(DictParser)
            
    def testPrefmt(self):
        self._testParser(PrefmtParser)

    def testMixed(self):
        self._testParser(MixedParser)

    def testNoResult(self):
        parser = NoResultParser()
        results = parser.process("uno dos tres quatro cinco cinco seis") 
        self.assert_(results == (), "unexpected values from NoResultParser")

# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()


