"""
Unittests for netlogger.parsers.config
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testParserConfig.py 764 2008-05-16 19:50:34Z dang $'

import os
import sys
import unittest
#
from netlogger.parsers import config
import testBase

class ConfigOnceTestCase(testBase.BaseTestCase):
    TMPDIR = "/tmp/netlogger-parser-test"
    CONFIGS = [('simple.cfg', """
[global]
modules_root = netlogger.parsers.modules
use_system_path = yes
state_file = %(TMPDIR)s/nlparser-test.state

[foo1]
files_root = %(TMPDIR)s
[[generic]]
files = *.log
"""),
               ('complex.cfg', """
[global]
modules_root = netlogger.parsers.modules
use_system_path = yes
state_file = %(TMPDIR)s/nlparser-test.state
eof_event = True
tail = True
rotate = 60
throttle = 0.5

[static1]
files_root = %(TMPDIR)s
[[generic]]
files = simple*.log
[[bp]]
files = nl*.log
[[[parameters]]]
has_gid = yes

[dynamic1]
files_root = %(TMPDIR)s
files = syslog*.log
pattern = "\[(?P<pid>\d+)\] (?P<level>[A-Z]+)/(?P<app>\S+):"
[[generic]]
[[[match]]]
app = "foo|bar"
[[bp]]
[[[match]]]
app = "[Nn]et[Ll]ogger"
level = 'DEBUG'
[[[parameters]]]
has_gid = true

"""),
               ('bad1.cfg', """
[disaster]
# no global section
# no subsections like this:
# [something]
[[badname]]
foo = bar
<< argh >>
"""),
               ('bad2.cfg', """
[global]
something = whatever
[disaster]
files_root = /usr
# no subsections like this:
# [something]
[[badname]]
foo = bar
"""),
]
    LOGS = [('simple_hello.log', """Hello
world
"""),
            ('nl_hello.log', """ts=2008-03-14T19:26:19.160217Z event=nlwrite.event level=Info msg="hello, world" gid=1234
ts=2008-03-14T19:26:24.872916Z event=nlwrite.event level=Info msg="hello, again" gid=1234
"""),
            ('syslog_hello.log', """
[123] INFO/foo: Hello
[99] DEBUG/netlogger: ts=2008-03-14T19:26:19.160217Z event=nlwrite.event level=Info msg="hello, world" gid=1234 syslog=yes
[456] INFO/bar: Hello
[123] INFO/foo: World
[99] INFO/NetLogger: ts=2008-03-14T19:26:24.872916Z event=nlwrite.event level=Info msg="goodbye" gid=1234  syslog=no
[99] DEBUG/NetLogger: ts=2008-03-14T19:26:24.872916Z event=nlwrite.event level=Info msg="hello, again" gid=1234 syslog=yes
[456] INFO/bar: World
"""),
]
    def setUp(self):
        if not os.path.isdir(self.TMPDIR):
            os.mkdir(self.TMPDIR)
        for filename, contents in self.CONFIGS:
            ofile = file(self.tmp(filename), "w")
            ofile.write(contents % dict(TMPDIR=self.TMPDIR))
        for filename, contents in self.LOGS:
            ofile = file(self.tmp(filename), "w")
            ofile.write(contents)

    def tmp(self, filename):
        return os.path.join(self.TMPDIR, filename)

    def tearDown(self):
        for filename in os.listdir(self.TMPDIR):
           os.unlink(self.tmp(filename))
        os.rmdir(self.TMPDIR)

    def readLoop(self, cfg):
        something, total = 1, 0
        while something:
            something = 0
            for name, parser in cfg.parsers.items():
                try:
                    for _ in xrange(50):
                        self.debug_(testBase.getNextEvent(parser))
                        something += 1
                except StopIteration:
                    pass
                total += something
        return total

    def basicTest(self, config_file):
        """Parse, save, and restore using a a given configuration.
        """
        path = self.tmp(config_file)
        cfg = config.Configuration(path)
        cfg.keys()
        self.debug_("%s configuration:\n%s" % (config_file, cfg.dump()))
        total = self.readLoop(cfg)
        self.failUnless(total > 1, "Did not process any log data")

    def testSimpleConfig(self):
        """Parse, save, and restore using a simple configuration.
        """
        self.basicTest('simple.cfg')

    def testComplexConfig(self):
        """Use a more complex configuration.
        """
        self.basicTest('complex.cfg')
        # now look to see
        #  (1) has_gid parameter took effect
        #  (2) the syslog was parsed at all
        #  (3) the syslog wasn't "over-parsed"
        #  (4) the throttle parameter was read
        # do NOT restoreState(), otherwise there will be nothing to parse
        path = self.tmp('complex.cfg')
        cfg = config.Configuration(path)
        got_syslog = False
        for _, parser in cfg.parsers.items():
            while 1:
                try:
                    line=testBase.getNextEvent(parser)
                except StopIteration:
                    break
                self.failIf('gid=' in line, 
                            'The parameter has_gid is being ignored')
                if 'syslog=' in line:
                    self.failIf('syslog=no' in line,
                                'Unexpectedly parsed syslog line')
                    got_syslog = True
        self.failUnless(got_syslog, "Missing syslog (dynamic) data")
        self.failUnless(cfg.throttle == 0.5, 
                        "bad 'throttle': wanted 0.5, got %lf" % cfg.throttle)

    def testBadConfig(self):
        """Make sure the proper error occurs on a bad config.
        """
        try:
            self.basicTest('bad1.cfg')
        except config.ConfigError, E:
            self.debug_("config error caught: %s" % E)
        try:
            self.basicTest('bad2.cfg')
        except config.ConfigError, E:
            self.debug_("config error caught: %s" % E)
        try:
            cfg = config.Configuration('/a/file/that/does/not/exist')
        except config.ConfigError, E:
            self.debug_("config error caught: %s" % E)            


# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
