#!/usr/bin/env python
"""
Tests of nl_pipeline sending commands to nl_parser and nl_loader
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testPipelineCommands.py 971 2008-09-05 13:55:32Z dang $'

# Imports
#import glob
import os
import re
import signal
import subprocess
import tempfile
import time
import unittest
# Local imports
from netlogger.tests import testBase
from netlogger import pipeline
from netlogger import talktome
from netlogger.util import rm_rf

class TestCase(testBase.BaseTestCase):
    """Unit test cases.
    """
    NUM_CMD = 10
    CONF_DIR = os.path.realpath("data/pipelineCommands")
    OUT_FILE = "out.bplog"
    KILL_WAIT_SEC, START_WAIT_SEC = 10, 10

    def setUp(self):
        """Configure and start pipeline.
        """
        # Configure
        self.output_dir = tempfile.mkdtemp(prefix="pipeline-")
        self.sharedConf()
        # Start pipeline
        pipeline = testBase.scriptPath('nl_pipeline')
        args = ["--config-dir=%s" % self.CONF_DIR,
                "--log-dir=%s" % self.output_dir, 
                "--pid-dir=%s" % self.output_dir,
                "-v", "-v"]
        full_command = [pipeline] + args
        self.debug_("pipeline command = '%s'" % ' '.join(full_command))
        subprocess.Popen(full_command)
        # wait for file to exist
        started = False        
        for i in xrange(self.START_WAIT_SEC):
            try:
                f = self.pipelinePidFile()
            except IOError:
                time.sleep(1)
                continue
            started = True
            self.debug_("pipeline is started")
            break
        self.failUnless(started,"nl_pipeline didn't start")

    def sharedConf(self):
        """Put variable values into shared configuration file.
        """
        fname = os.path.join(self.CONF_DIR, "shared.conf.in")
        f = file(fname, 'r')
        lines = [ ]
        for line in f:
            line2 = line.replace('@BASEDIR@', self.CONF_DIR)
            line2 = line2.replace('@OUTPUTDIR@', self.output_dir)
            line2 = line2.replace('@OUTPUTFILE@', self.OUT_FILE)
            lines.append(line2)
        f.close()
        fname = os.path.join(self.CONF_DIR, "shared.conf")
        f = file(fname, 'w')
        for line in lines:
            f.write(line)

    def pipelinePidFile(self):
        fname = os.path.join(self.output_dir, "nl_pipeline.pid")
        return file(fname)
        
    def tearDown(self):
        """Stop pipeline.
        """
        # kill and wait for file to not exist
        killed = False
        f = self.pipelinePidFile()
        pid = int(f.read(20))
        self.debug_("kill pipeline pid %d" % pid)
        os.kill(pid, signal.SIGTERM)
        for i in xrange(self.KILL_WAIT_SEC):
            if not os.path.exists(f.name):
                self.debug_("pipeline is dead")
                killed = True
                break
            self.debug_("wait for pipeline to die")
            time.sleep(1)
        self.failUnless(killed, "could not kill nl_pipeline")
        # 
        if not self.DEBUG:
            rm_rf(self.output_dir)
        else:
            self.debug_("do NOT remove output dir %s" % self.output_dir)

    def testSendMany(self):
        """Send many commands to loader/parser
        """
        secret_file = self.getSecretFile()
        self.failUnless(secret_file, "can't find secret file in log")
        self.debug_("secret file = '%s'" % secret_file)
        secret = file(secret_file).read(talktome.Base.SECRET_LEN)
        p = talktome.Parent(talktome.PARSER_PORT)
        p.setSecret(secret)
        l = talktome.Parent(talktome.LOADER_PORT)
        l.setSecret(secret)
        for i in xrange(self.NUM_CMD):
            self.debug_("send command set %d" % i)
            p.sendCommand('SAVE')
            p.sendCommand('ROTT')
            p.sendCommand('RECF')
            l.sendCommand('SAVE')
        self.debug_("done with commands")

    def getSecretFile(self):
        """Get filename of 'secret' file by parsing the logs
        """
        fname = os.path.join(self.output_dir, "nl_pipeline.log")
        f = file(fname, 'r')
        for line in f:
            m = re.search("secret_file=(\S+)", line)
            if m:
                return m.group(1)
        return None

# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
