#!/usr/bin/env python
"""
Unittests for netlogger/util.py
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testUtil.py 788 2008-06-02 17:11:49Z dang $'

from glob import glob
import os
from tempfile import NamedTemporaryFile
import time
import unittest
#
import testBase
from netlogger import util

class TestCase(testBase.BaseTestCase):
    NNF_DIR = "/tmp"
    NNF_PFX = "netlogger-test-util-nnf"

    def setUp(self):
        self.named_temp_files = [ ]
        self._eraseNNF()

    def cleanUp(self):
        for ntf in self.named_temp_files:
            os.unlink(ntf)
        self._eraseNNF()

    def _eraseNNF(self):
        pattern = os.path.join(self.NNF_DIR, self.NNF_PFX) + "*"
        for old_file in glob(pattern):
            os.unlink(old_file)
        
    def testGetNNF(self):
        """Test of getNextNumberedFile() function
        """
        path = os.path.join(self.NNF_DIR, self.NNF_PFX)
        f = util.getNextNumberedFile(path)
        self.assert_(f)
        self.failUnless(f.name == path + '.1',
                        "Numbered file '%s' did not end in .1" % f.name)
        f = util.getNextNumberedFile(path)
        self.assert_(f)
        self.failUnless(f.name == path + '.2',
                        "Numbered file '%s' did not end in .2" % f.name)
        # test w/strip
        f = util.getNextNumberedFile(f.name, strip=True)
        self.assert_(f)
        self.failUnless(f.name == path + '.3',
                        "Numbered file '%s' did not end in .3" % f.name)
    
    def testGetLNF(self):
        """Test of getLowestNumberedFile() function
        """
        path = os.path.join(self.NNF_DIR, self.NNF_PFX)
        f = util.getLowestNumberedFile(path)
        self.assert_(f is None)
        for n in range(9,1,-2):
            fname = path + ".%d" % n
            file(fname, "w")
            f = util.getLowestNumberedFile(path)
            self.failUnless(f.name == fname, 
                            "expected '%s', got '%s'" % (fname, f.name))
        for n in range(9,1,-2):
            fname = path + ".%d" % n
            os.unlink(fname)

    def testMostRecentFile(self):
        """Test the mostRecentFile function
        """
        dir = "/tmp"
        pfx = "netlogger-test-util"
        # make some files
        start = time.time()
        f1 = NamedTemporaryFile(prefix=pfx, dir=dir)
        self.named_temp_files.append(f1)
        time.sleep(1.1)
        f2 = NamedTemporaryFile(prefix=pfx, dir=dir)
        self.named_temp_files.append(f2)
        end = time.time()
        # get most recent, without after_time
        mrf = util.mostRecentFile(dir, pfx + "*")
        self.failUnless(mrf, "Most recent file not found")
        self.failIf(len(mrf) > 1, "Too many recent files found: %s" % mrf)
        self.failUnless(mrf[0] == f2.name, "Wrong most recent file")
        # try the same with after_time before creation
        mrf = util.mostRecentFile(dir, pfx + "*", after_time=start)
        self.failUnless(mrf, "Most recent file not found")
        self.failIf(len(mrf) > 1, "Too many recent files found: %s" % mrf)
        self.failUnless(mrf[0] == f2.name, "Wrong most recent file")
        # one more time, with after_time after creation;
        # expect no results
        mrf = util.mostRecentFile(dir, pfx + "*", after_time=end+1)
        self.failIf(mrf, "Too many recent files found: %s" % mrf)
        # now test 'ties'; probabilistic since it is possible that
        # two files fall on either side of a second
        tied = False
        for i in xrange(10):
            f1 = NamedTemporaryFile(prefix=pfx, dir=dir)
            f2 = NamedTemporaryFile(prefix=pfx, dir=dir)
            self.named_temp_files.extend([f1, f2])
            mrf = util.mostRecentFile(dir, pfx + "*", after_time=end)
            self.debug_("tied: %s" % mrf)
            if len(mrf) > 1:
                tied = True
                break
        self.failUnless(tied, "Missed ties")

    def testThrottleTimer(self):
        """Test the ThrottleTimer class.
        """
        # allowed +/- slop between expected and measured
        slop = 0.5
        minus_slop, plus_slop = 1-slop, 1+slop
        # loop over these run-time to sleep-time ratios
        for rr in (0.2, 0.5, 0.8):
            timers = (util.ThrottleTimer(rr), util.NullThrottleTimer(rr))
            work = [0, 0]
            num_exp = 3
            # run 'num_exp' experiments, throw out first as "warm-up"
            for i in range(num_exp):
                # alternate between working and null timer 
                for tnum, tt in enumerate(timers):
                    tt.start()
                    t0 = time.time()
                    n = 0
                    # run for 0.5 seconds
                    while 1:
                        n += 1
                        x, y = 1, 1
                        for k in xrange(1000):
                            z = x + y
                            x, y = y, z
                        tt.throttle()
                        if time.time() - t0 > 0.2:
                            break
                    # if not warm-up, add work done to total
                    if i > 0:
                        work[tnum] += n
                    self.debug_("run %d, timer %d: %d iters" % (i, tnum, n))
            # calc average work
            avg_work = map(lambda x: 1. * x / (num_exp - 1), work)
            # compare work to ratio
            work_ratio = avg_work[0] / avg_work[1]
            self.debug_("ratio %lf: throttled=%d null=%d throttled/null=%lf",
                        rr, avg_work[0], avg_work[1], work_ratio)
            if not ((rr * minus_slop) < work_ratio < (rr * plus_slop)):
                self.fail("work ratio %lf not in (%lf, %lf)" % (
                        work_ratio, (rr * minus_slop), (rr * plus_slop)))
                                                                

# Boilerplate to run the tests
def suite(): 
    return testBase.suite(TestCase)
if __name__ == '__main__':
    testBase.main()
