"""
Remove data older than a given number of days 
from all tables.

Parameters:
   days = Delete beyond this number of days
   destdb = Database to move all data to; empty or absent means just delete.
            Currently, the only implemented option is to just delete.
"""

## Imports

import time
from netlogger.actions.base import BaseAction

## Classes

class Action(BaseAction):
    """Delete old data from the database.

    The definition of old is: more than self.param[DAYS_PARAM] days old.
    Because the data is deleted by event id, and these are not
    strictly ordered, there is up to a 1 day of "slop" in the 
    implementation.
    """

    DAYS_PARAM = 'days'
    DEST_PARAM = 'destdb'

    def execute(self):
        self.log.info("execute.start")
        # Setup
        self.log.debug("setup.start")
        conn = self.getConnection()
        param = self.getParameters()
        try:
            strdays = param[self.DAYS_PARAM]
            days = int(strdays)
        except KeyError:
            raise KeyError("Missing parameter '%s' for the #days" % 
                           self.DAYS_PARAM)
        except ValueError:
            raise KeyError("Bad integer value '%s' for  parameter '%s' "
                           "for the #days" % (strdays, self.DAYS_PARAM))
        destdb = param.get(self.DEST_PARAM, '')
        self.log.debug("setup.end", days=days, destdb=destdb)
        # Calculate time of boundary event
        ndays = days + 1 # 1 day of slop
        ts = time.time() - 60*60*24*ndays
        # Find the boundary event id
        self.log.debug("get.maxid.start")
        curs = conn.cursor()
        curs.execute("select max(id) from event where time < %lf" % ts)
        row = curs.fetchone()
        if row[0] is None:
            self.log.info("execute.end", status=0, msg="nothing to do")
            return # nothing to do
        max_id = int(row[0])
        self.log.debug("get.maxid.end", value=max_id)
        # Process all rows, in all tables, at or before the boundary event
        if destdb:
            # Insert rows in destination DB
            raise NotImplementedError("Destination database option not done")
        # Delete rows from source DB
        self.log.debug("delete.event", event__id=max_id)
        curs.execute("delete from event where id <= %d" % max_id)
        for table in ("attr", "ident", "dn", "text"):
            self.log.debug("delete.%s" % table, event__id=max_id)
            curs.execute("delete from %s where e_id <= %d" % (table, max_id))
        self.log.info("execute.end", status=0, event__id=max_id)
        conn.commit()
        conn.close()
