"""
Logging Best Practices messages in Python.

Usage:
  logging_example STYLE [LEVEL [COUNT]]
    STYLE is 'bp' or 'pretty'
    LEVEL is 0, 1, 2, or 3 which is WARN, INFO, DEBUG, or TRACE (default=1)
    COUNT is the number of iterations (default=100)

Covers two use-cases:
(1) You want to log in methods of a class. 
    In this case, inherit from the DoesLogging mix-in.
    Change the class variable NAMESPACE for your project
(2) You want to log in top-level functions.
    In this case, call get_logger() in your functions to grab
    the log object. It is handy to have a top-level get_log() function
    that will do this for the right namespace. The reason we do *not*
    want to create a global 'log' object is that we often want to set
    the attributes of the loggers at runtime, and the global objects
    are created at import time.
"""
import logging
import sys
from netlogger import nllog

nllog.PROJECT_NAMESPACE = "my.project"

def get_log():
    return nllog.get_logger("example")

class DoSomething(nllog.DoesLogging):
    def __init__(self, count):
        nllog.DoesLogging.__init__(self)
        self._n = count

    def run(self):
        self.log.info("run.start")
        self.do_work()
        self.log.info("run.end", status=0)

    def do_work(self):
        self.log.debug("do_work.start")
        for i in xrange(self._n):
            if self.log.isEnabledFor(nllog.TRACE):
                self.log.trace("do_work.iter", num=i)
        self.log.debug("do_work.end", status=0)

def run(iterations):
    log = get_log()
    log.info("run.start")
    do_something = DoSomething(iterations)
    do_something.run()
    log.info("run.end", status=0)
    return 0

def error(msg):
    sys.stderr.write("ERROR: " + msg + "\n")
    sys.stderr.write(__doc__)
    sys.exit(-1)

PRETTY, BP = 'pretty', 'bp'

if __name__ == '__main__':
    level, count = 1, 100
    if len(sys.argv) < 2:
        error("Style argument is missing")
    style = sys.argv[1].lower()
    if style not in (PRETTY, BP):
        error("Bad value for 'style': %s" % style)
    if len(sys.argv) > 2:
        try:
            level = int(sys.argv[2])
            if level < 0 or level > 3:
                raise ValueError()
        except ValueError:
            error("Bad level: %s" % sys.argv[2])
        if len(sys.argv) > 3:
            try:
                count = int(sys.argv[3])
            except ValueError:
                error("Bad number of iterations: %s" % sys.argv[2])
    if style == PRETTY:
        logging.setLoggerClass(nllog.PrettyBPLogger)
    else:
        logging.setLoggerClass(nllog.BPLogger)
    # get a logger at the top-level of the namespace
    log = logging.getLogger(nllog.PROJECT_NAMESPACE)
    # add standard error handler
    handler = logging.StreamHandler()
    handler.setFormatter(logging.Formatter("%(message)s"))
    log.addHandler(handler)
    # set level
    log_level = (logging.WARN, logging.INFO, logging.DEBUG, 
                 nllog.TRACE)[level]
    log.setLevel(log_level)
    # run
    sys.exit(run(count))

