#!/bin/sh

#
# File: nl_parse_example.sh
# Examples of running nl_parse.
#
# usage: nl_parse_example NUM [options]
#
# NUM - Example number
# options - Add -v options to increase verbosity of
# nl_parse and, if applicable, nl_load.
#

# constant: TMP
#
# Temporary directory under which input files
# and output files will be written.
# Must be writable, and allow directory creation,
# by whoever runs the script.
#
TMP="."

# Functions common to example scripts
. common.sh

#
# Function: Main
#
# Program entry point.
# Create input files, pick an example to run,
# and count the number of lines in the output.
#
# After running the desired example, the
# number of lines in all the input files (as
# returned by the write_logs function) are
# printed, and then the number of lines in the
# output file produced by running the example.
# For examples 1, 2, and 3 these numbers should match.
#

main ()
{
    which="$1"
    shift
    # Create some logfiles under $TMP
    wl=`write_logs 150`
    logdir=`echo $wl | cut -f1 -d' '`
    lognum=`echo $wl | cut -f2 -d' '`
    # Run appropriate example
    case $which in
        1) example1 "$*";;
        2) example2 "$*";;
        3) example3 "$*";;
        4) example4 "$*";;
    esac
    ofile="example${which}.log"
    # Check line count
    printf "Input lines : %d\n" $lognum
    printf "Output lines: " 
    printf `wc -l $ofile` 
    printf "\n"
    # Cleanup output file
    /bin/rm -f $ofile
    # Cleanup temporary directory
    cleanup
}

#
# Function: Example 1
#
# Parse NL-BP files and write them to a file.
#
# Steps:
# * run nl_parse in foreground
#
example1 () 
{
    opts="$1"
    nl_parse $opts -f $TMP/example1.log bp "$logdir/*.log"
}

#
# Function: Example 2
#
# Parse NL-BP files, send them to the info_broker,
# and use the nl_load program to pull them out and
# write them to a file.
#
# Steps:
# * Run nl_broker in background
# * Run nl_load, connected to nl_broker, in background
# * Run nl_parser, connected to nl_broker, in foreground
# * When it finishes, kill nl_broker and nl_load
#
example2 () 
{
    opts="$1"
    ofile=$TMP/example2.log
    # Run broker
    nl_broker &
    brokerid=$!
    printf "starting broker..\n"
    sleep 1
    # Run "loader" that just writes the data to a file
    nl_load $opts -c localhost bp_loader > $ofile &
    loaderid=$!
    printf "starting loader..\n"
    sleep 1
    # Run parser that sends the logs to the loader
    printf "running parser..\n"
    nl_parse $opts -c localhost bp "$logdir/*.log"
    printf "parser is done\n"
    # Kill backgrounded processes
    printf "killing loader and broker..\n"
    sleep 1    
    kill $loaderid 
    sleep 1
    kill $brokerid
    sleep 1
}

#
# Function: Example 3
#
# Parse files and send to file, in 'tail' mode.
# This is similar <Example 1>, but with an added
# test of the ability to 'tail' files indefintely
# and also notice when files come and go.
#
# Steps:
# * Rename one of the input files, so the
#   parser will not see it initially
# * Run nl_parse in the background
# * After a couple of seconds, re-rename the input
#   file to fit the pattern the parser is looking for.
# * Wait a couple more seconds for nl_parse to 
#   scan the directory and parse the file.
# * Kill the parser
#
example3 ()
{
    opts="$1"
    ofile="$TMP/example3.log"
    # pick a file, rename it
    myfile=`/bin/ls -1 $logdir | head -1`
    mv $logdir/$myfile $logdir/${myfile}.moved
    # run parser
    nl_parse $opts -f "$ofile" -t bp -s 1 "$logdir/*.log" &
    parserid=$!
    # wait a sec
    sleep 2
    # 'add' back the renamed file
    printf "add back file..\n"
    mv $logdir/${myfile}.moved $logdir/$myfile
    # wait for it to get picked up
    sleep 1
    # stop parser
    kill $parserid
    sleep 1
}

#
# Function: Example 4
#
# A test - start the loader client and parser
# before the broker to test the timeout/reconnect
# stuff.
#
# Steps:
# * Run nl_load in background with retry set to 3 seconds
# * Run nl_parser in background retry set to 15 seconds.
# * Run nl_broker in background.
# * Wait for the parser to connect and write data.
# * Parser will exit when done.
# * When it finishes, kill nl_broker and nl_load
#
#
example4 () 
{
    opts="$1"
    ofile=$TMP/example4.log
    # Run "loader" that just writes the data to a file
    nl_load $opts -c localhost bp_loader -r 3 > $ofile &
    loaderid=$!
    printf "starting loader..\n"
    sleep 1
    # Run parser that sends the logs to the loader
    nl_parse $opts -c localhost bp -r 15 "$logdir/*.log" &
    parserid=$!
    printf "starting parser and waiting..\n"
    sleep 5
    nl_broker &
    brokerid=$!
    printf "starting broker..\n"
    # give parser time to do its thing
    printf "waiting on parser..\n"
    sleep 25
    # Kill backgrounded processes
    printf "killing loader and broker..\n"
    sleep 1    
    kill $loaderid 
    sleep 1
    kill $brokerid
    sleep 1
}


main $*
