"""
Create a table with a summary of event types in the database.

Parameters:
   table = Database table to create. Default: 'event_types'.
   start = Use this date, in format YYYY-MM-DD[Thh:mm:ss], as the
           starting date to look for event types. 
           If not defined, look at the whole database. Default: <undef>.
"""
__author__ = "Dan Gunter <dkgunter@lbl.gov>"
__rcsid__ = "$Id: event_types.py 23923 2009-09-18 22:42:26Z ksb $"

## Imports

import time
from netlogger.actions.base import BaseAction
from netlogger.nllog import DoesLogging, get_logger
from netlogger import nldate

## Classes

class Action(BaseAction,DoesLogging):
    """Create a table with a summary of event types in the database.
    """

    TABLE_PARAM = 'table'    
    DATE_PARAM = 'start'
    TABLE_DEFAULT = 'event_types'

    MAX_ATTR_LEN = 4096 # max. length of concatenated attribute names

    def __init__(self):
        DoesLogging.__init__(self)
        BaseAction.__init__(self)

    def execute(self):
        self.log.info("execute.start")
        # Setup
        self.log.debug("setup.start")
        conn = self.getConnection()
        curs = conn.cursor()
        # : parse parameters
        param = self.getParameters()
        table_name = param.get(self.TABLE_PARAM, self.TABLE_DEFAULT)
        start_time = None
        if param.has_key(self.DATE_PARAM):
            dt = param[self.DATE_PARAM]
            try:
                iso_dt = nldate.completeISO(dt)
                start_time = nldate.parseISO(iso_dt)
            except nldate.DateFormatError, E:
                raise ValueError("Bad '%s' value: %s" % (self.DATE_PARAM, E))
        self.log.debug("setup.end")
        # Drop and (re-)create target table
        self.log.debug("create.table.start", value=table_name)
        curs.execute("drop table if exists %s" % table_name)
        curs.execute("create table %s ("
                     "name varchar(300), "
                     "num integer(15), "
                     "min_time double, "
                     "max_time double, "
                     "attr_list varchar(%d)"
                     ")" % (table_name, self.MAX_ATTR_LEN))
        self.log.debug("create.table.end", value=table_name, status=0)
        # Get list of event types
        restrict = ""
        if start_time:
            restrict = "where time >= %lf" % start_time
        curs.execute("select name, startend, count(*), "
                     "min(time), max(time), min(id), max(id) from event "
                     "%s "
                     "group by name, startend" % restrict)
        # Loop through event types, fetch related attributes,
        # and insert results into target table.
        for nm, se, cnt, mt, xt, mi, xi in curs.fetchall():
            curs2 = conn.cursor()
            self.log.debug("insert.event_type.start", name=nm, startend=se)
            # look at first/last event of this type for the
            # attributes (faster than looking at ALL events)
            restrict = "e_id = %d or e_id = %d" % (mi, xi)
            # get list of attributes
            curs2.execute("select name from attr where %s group by name" % 
                         restrict)
            attrs = [r[0] for r in curs2.fetchall()]
            # get list of identifiers
            curs2.execute("select name from ident where %s group by name" % 
                         restrict)
            ids = [r[0] for r in curs2.fetchall()]
            attrs.extend(ids)
            # get yes/no for DN
            curs2.execute("select count(*) from dn where %s" % restrict)
            if curs2.fetchone()[0] > 0:
                attrs.insert(0, "DN")
            # get yes/no for text
            curs2.execute("select count(*) from text where %s" % restrict)
            if curs2.fetchone()[0] > 0:
                attrs.insert(0, "text")
            # insert result            
            ename = nm + (".start", ".end", "")[se]
            attr_str = ",".join(attrs)
            # If string overflows, add '...' indicating this
            if len(attr_str) > self.MAX_ATTR_LEN:
                attr_str = attr_str[self.MAX_ATTR_LEN-3:] + '..'
            # Insert values
            curs2.execute("insert into %s values "
                         "( '%s', %d, %lf, %lf, '%s')" % 
                         (table_name, 
                          ename, cnt, mt, xt, attr_str))
            self.log.debug("insert.event_type.end", status=0)
            curs2.close()
        self.log.info("execute.end", status=0)
        conn.commit()
        conn.close()
