"""
Abstract (base) functions, classes, and variables for data mining.
"""
__author__ = "Dan Gunter <dkgunter@lbl.gov>"
__rcsid__ = "$Id: base.py 26773 2010-11-13 03:12:07Z dang $"

from netlogger.nllog import DoesLogging

class PlotError(Exception):
    pass

class DataSource:
    """Source of data for a plot or analysis.
    """
    def __init__(self, data=None):
        """Constructor.

        Kwargs:
          - data: Iterable rows of data.
        """
        self.data = self.preprocess(data)

    def preprocess(self, obj):
        """Subclasses should override this to transform the data
        as it's provided.
        """
        return obj

class PlotDevice:
    """Enumeration of plot devices, corresponding to the
    names R has for them.
    """
    PNG = "png"
    PDF = "pdf"
    
class PlotParam:
    """Plot parameters, which is just some syntactic sugar on top
    of a dictionary. Selected dictionary values can be accessed as
    if they were attributes.

    Subclasses should be declared as nested classes inside the
    subclass for a given plot function
    """
    def __init__(self, filename="/tmp/plot",
                 device=PlotDevice.PDF, xlab="X", ylab="Y",
                 title="Plot", **kw):
        self.device = device
        self.xlab = xlab
        self.ylab = ylab
        self.filename = filename
        self.title=title
        self._kw = kw
        self._attrs = ['device', 'xlab', 'ylab', 'filename', 'title']
        
    def as_dict(self):
        d = self._kw
        for key in self._attrs:
            d[key] = getattr(self, key)
        return d
                                     
class PlotFunction(DoesLogging):
    """Abstract superclass of a plotting function.
    """
    def __init__(self, data=None, param=None):
        """Constructor.
        Initialize the plot with optional data and parameters.
        These can also be set later.
        
        Kwargs:
          - data (DataSource): The data for the plot.
          - param (PlotParam): Parameters for the plot.
        """
        DoesLogging.__init__(self)
        self.param = param
        self.data = data
    
    def create(self):
        """Create/re-create plot.
        Subclasses should override with real behavior.

        Return: Plot instance
        """
        return Plot()

    def create_common(self, min_points=2, rpy=None):
        if self.data is None:
            raise PlotError("Empty data")
        elif len(self.data) < min_points:
            raise PlotError("Need at least %d points to plot" % min_points)
        # don't understand why this is necessary?
        rpy.importr("plyr")
        rpy.importr("ggplot2")
        rpy.importr("lattice")

class Plot:
    """Created plot.

    Public attributes:
      - path (str): Location of plot.
      - device (str): Name of plot device.
    """
    def __init__(self, path=None, device=None):
        """Constructor.

        Kwargs:
          - path (str): Location of plot.
          - device (str): Name of plot device.
        """
        self.path = path
        self.device = device

    def __str__(self):
        return "Plot: path=%s device=%s" % (self.path, self.device)
