"""
For a time-series of values with a duration, where the values which
overlap in time are additive, calculate the average total value in
each fixed interval of time. Applications include the summing of
multiple concurrent data transfers over a single path.
"""
from math import ceil, floor

def reg(ts, delta, window):
    """Smooth an irregular univariate time-series 'ts' into regular one
    of size delta. This algorithm is linear in the size of 'ts',
    and uses constant (and small) temporary storage.
    The result vector uses memory O(time-range of ts / delta).

    For example:

    >>> a = ((105, 110, 115), (28, 10, 2), (1, 1, 2))
    >>> for interval in (10, 20, 40):
    ...     r = regularize(a, interval, (100, 140))
    ...     print ', '.join(["%.2f" % x for x in r])
    0.50, 2.40, 1.00, 0.30
    1.45, 0.65
    1.05
    >>> a = ((0,), (7,), (1.0,))
    >>> r = regularize(a, 5, (0,5))
    >>> print ', '.join(["%.2f" % x for x in r])
    1.00
    
    Args:
      - ts (3xN floats): Input matrix, with 3 columns: time(stamp), duration,
        value.
      - delta (float): Interval.
      - window (float,float): Times for start/end of interval. This is
        rounded up to an integer multiple of the interval.
      
    Return:
      N floats) Time, and value for each interval of size
                delta starting from window[0].
    """
    # Variable for each column in input array
    tm, dur, val = ts
    # Start of first interval
    t0 = floor(window[0] / delta) * delta
    # End of last interval
    t1 = ceil(window[1] / delta) * delta
    # Number of intervals
    n = int((t1 - t0) / delta)
    # Result vector
    a = [0.0] * n
    # Span of values
    span = t1 - t0
    # Main loop, over input elements
    for i in xrange(len(tm)):
        # Shorthand vars
        offs = tm[i] - t0
        v = float(val[i])
        # Interval numbers for this value:
        #  - beginning of first
        x1 = int(floor(offs / delta))
        #  - beginning of last, clipped to range
        x2 = int(ceil((min(offs + dur[i], span)) / delta)) - 1
        if x1 == x2:
            # If only one interval, add fraction of time in
            # the interval to the total. min() clips the duration in case
            # it goes outside the window.
            a[x1] += v * min(dur[i], delta) / delta
        else:
            # If more than one interval, do three things.
            # 1) Add part of duration at end of first interval
            a[x1] += v * (((x1 + 1) * delta) - offs) / delta
            # 2) Add middle part(s)
            for j in xrange(x1 + 1, x2):
                a[j] += v
            # 3) Add part of duration at end of last interval
            a[x2] += v * (offs + dur[i] - x2*delta) / delta
    # End of main loop.
    # Return result
    return a

if __name__ == "__main__":
    import doctest, sys
    if len(sys.argv) > 1 and sys.argv[1] == '-d':
        doctest.testmod()
    else:
        a = ((105, 110, 115), (28, 10, 2), (1, 1, 2))
        for interval in (10, 20, 40):
            r = regularize(a, interval, (100, 140))
            print ', '.join(["%.2f" % x for x in r])
