"""
Parse output file from SGI csa process accounting
"""

__author__ = 'Tina Declerck tinad@nersc.gov'
__rcsid__ = '$Id: csa_acct.py 000 2008-07-31 12:42:00 tinad $' 

import sys
import time
from netlogger import nldate
from netlogger.parsers.base import BaseParser

"""
Sample input:

NOTE:  This is running accounting as follows:  csacom -pPXY

#usrmod           root     11:05:03 11:05:03     0.02     0.01        0   7638   7637 Mon Jul 14 2008 Mon Jul 14 2008
usrmod            cam      11:05:03 11:05:03     0.08     0.02        0   7637   7561 Mon Jul 14 2008 Mon Jul 14 2008
perl              cam      11:05:02 11:05:04     2.52     0.39        0   7561   7559 Mon Jul 14 2008 Mon Jul 14 2008
sh                cam      11:05:02 11:05:04     2.53     0.00        0   7559      1 Mon Jul 14 2008 Mon Jul 14 2008
#cron             root     11:05:02 11:05:04     2.56     0.00        0   7539  26996 Mon Jul 14 2008 Mon Jul 14 2008
sleep             fogal1   11:04:50 11:05:05    15.00     0.00        0   7532   7419 Mon Jul 14 2008 Mon Jul 14 2008
sleep             fogal1   11:04:50 11:05:05    15.00     0.00        0   7536   7418 Mon Jul 14 2008 Mon Jul 14 2008
sleep             fogal1   11:04:50 11:05:05    15.00     0.00        0   7535   7421 Mon Jul 14 2008 Mon Jul 14 2008
ps                fogal1   11:05:05 11:05:05     0.11     0.10        0   7642   7641 Mon Jul 14 2008 Mon Jul 14 2008
grep              fogal1   11:05:05 11:05:05     0.11     0.00        0   7643   7641 Mon Jul 14 2008 Mon Jul 14 2008

"""

NS = 'csa.'
EVENT_BOTH = "%sprocess" % NS
EVENT_START = "%sprocess.start" % NS
EVENT_END = "%sprocess.end" % NS

class Parser(BaseParser):
    """SGI Comprehensive System Accounting (CSA) process accounting parser.
    See also http://oss.sgi.com/projects/csa/.

    Parameters:
       - one_event {yes,no,yes*}: csa.process instead of csa.process.start/csa.process.end

    """
    def __init__(self, f, one_event='yes', **kw):
        """
        """
        BaseParser.__init__(self, f, fullname=__name__, **kw)
        self._one_event = self.boolParam(one_event)

    def process(self, line):
        parts = line.split()
        if len(parts) != 17:
            return () # ignore unknown
        e = { 'cmd':parts[0],
              'local_user':parts[1],
              'walltime' : parts[4],
              'cputime' : parts[5],
              'ignore':parts[6],
              'pid':parts[7],
              'ppid':parts[8] }
        # parse start/end times
        tm = ' '.join((parts[9], parts[10], parts[11], parts[2], parts[12]))
        start = nldate.parseSyslogDate(tm)
        tm = ' '.join((parts[13], parts[14], parts[15], parts[3], parts[16]))
        end = nldate.parseSyslogDate(tm)
        # build and return events
        if self._one_event:
            e.update({'ts':start, 'event': EVENT_BOTH, 'dur' : end-start})
            return (e,)
        else:
            e1 = { 'ts':start, 'event':EVENT_START,
                    'pid':e['pid'], 'ppid':e['ppid'], }
            e.update({'ts':end, 'event':EVENT_END})
            return (e1, e)
