#!/usr/bin/env python
"""
Base class for Unittests for NLA
"""
__author__ = 'Keith Jackson KRJackson@lbl.gov'
__rcsid__ = '$Id: shared.py 26501 2010-09-23 00:34:43Z mgoode $'

try:
    import cStringIO as StringIO
except ImportError:
    import StringIO
import logging
import os
import re
import sys
import signal
import subprocess
import time
from subprocess import PIPE, Popen, call
import unittest
#
from netlogger.nlapi import Level
from netlogger.nllog import PrettyBPLogger, get_root_logger, setLoggerClass, get_logger
from netlogger.parsers.base import NLFastParser
from netlogger.parsers.modules import dynamic

def script_path(script, search_path=('.', 'scripts', 
                                     os.path.join('..','..', 'scripts'))):
    """Look for 'script' in the list of paths in 'search_path'.
    If found, return the full path to the script.
    Otherwise, return the input argument.
    """
    script_path = script
    script_name = os.path.basename(script)
    for dirname in search_path:
        if os.path.isdir(dirname):
            path = os.path.join(dirname, script_name)
            if os.path.exists(path):
                script_path = path
    return script_path

def fileDir():
    return os.path.dirname(os.path.abspath(__file__))

def getNextEvent(parser):
    while 1:
        e = parser.next()
        if e is None:
            continue
        break
    return e

def isiterable(obj):
    try: 
        _ = iter(obj)
        result = True
    except TypeError:
        result = False
    return result

class BaseTestCase(unittest.TestCase):
    DEBUG = os.getenv('TEST_DEBUG', default=0)
    TRACE = 0
    try:
        x = int(DEBUG)
        if x > 1:
            TRACE = 1
    except (ValueError, TypeError), E:
        pass

    # directory for data files
    data_dir = os.path.join(fileDir(), 'data')

    # Default one-time logging setup
    setLoggerClass(PrettyBPLogger)
    log = get_logger("tests")
    log.setLevel(logging.CRITICAL)
    log.addHandler(logging.StreamHandler())
    # Set level
    if DEBUG:
        if DEBUG == '1':
            log.setLevel(logging.INFO)
        elif DEBUG == '2':
            log.setLevel(logging.DEBUG)
        else:
            log.setLevel(Level.TRACE)

    def __init__(self, *args, **kwargs):
        self._setDataDir()
        unittest.TestCase.__init__(self, *args, **kwargs)

    def setUp(self):
        """Base setUp actions
        """
        self.stderr_save = None

    def set_up_nl_logger(self):
        """ guarantee a netlogger logger """
        get_root_logger().addHandler(logging.StreamHandler())
        if self.DEBUG == '1':
            get_root_logger().setLevel(logging.INFO)
        elif self.DEBUG == '2':
            get_root_logger().setLevel(logging.DEBUG)
        else:
            get_root_logger().setLevel(logging.CRITICAL)


    def captureStderr(self):
        self.stderr_save = sys.stderr
        my_stderr = StringIO.StringIO()
        sys.stderr = my_stderr
        return my_stderr

    def restoreStderr(self):
        if self.stderr_save:
            sys.stderr = self.stderr_save

    def tearDown(self):
        """Return stdout to its proper state"""
        #sys.stdout = self.stdout_save

    def log_(self, msg, *args):
        if args:
            sys.stderr.write(msg % args)
        else:
            sys.stderr.write(msg)
        sys.stderr.write("\n")

    error = log_

    def debug_(self, msg, *args):
        """Note: unittest.TestCase already has a debug() method
        """
        if self.DEBUG:
            self.log_(msg, *args)

    def trace_(self, msg, *args):
        """Write even finer-grained debugging events
        Does nothing unless self.TRACE is True.
        """
        if self.TRACE:
            self.log_(msg, *args)

    trace = trace_

    def getOutput(self):
        """Get captured stdout"""
        return self.sio.getvalue()

    def setInput(self, s):
        """Write to object and seek back to 0"""
        self.sio = StringIO.StringIO(s)

    def setProgram(self, prog):
        self.program = prog

    def cmd(self, args, action='kill', should_fail=False, **kw):
        """Run command.

        Action specifies what to do after starting the process.
          - kill: Wait for it to exit, then kill it
          - wait: Wait for it to exit
          - communicate: Wait for it to exit, put stdout and stderr
            into self.cmd_stdout and self.cmd_stderr
          - fork: Just fork the process and return the Popen obj

        For all but 'fork' actions, this method evaluates whether
        the process failed. If 'should_fail' is True then a RuntimeError
        is expected.

        Additional keyword arguments are passed to the run_cmd() method.
        """
        proc = None
        try:
            proc = self.run_cmd(args, action, **kw)
            if should_fail:
                self.fail("unexpected success for: %s" % args)
        except RuntimeError, E:
            if not should_fail:
                self.fail("failed: %s" % E)
        return proc

    def run_cmd(self, params, action, pipe_stdin=False):
        cmd = [self.program] + params
        self.debug_("run command: %s" % ' '.join(cmd))
        result = -1
        if pipe_stdin:
            stdin=PIPE
        else:
            stdin = None
        #print "@@ CMD='%s'" % ' '.join(cmd)
        p = Popen(cmd, stdin=stdin, stdout=PIPE, stderr=PIPE)
        if action == 'kill':
            time.sleep(0.5)
            result = p.poll()
            if result is None:
                os.kill(p.pid, signal.SIGKILL)
                result = 0
        elif action == 'wait':
            if p:
                result = p.wait()
        elif action == 'communicate':
            self.cmd_stdout, self.cmd_stderr = p.communicate()
            result = p.returncode
        elif action == 'fork':
            return p
        else:
            raise ValueError("internal error: bad action")
        if result != 0:
            if p:
                output, errs = p.communicate()
            else:
                output, errs = '', ''
            indent = lambda s : '\n'.join(["   %s" % x 
                                           for x in s.split('\n')])
            raise RuntimeError("command '%s' failed (%d).\nOUTPUT:\n%s\n"
                               "ERRORS:\n%s" % (
                    ' '.join(cmd), result, indent(output), indent(errs)))

    def _setDataDir(self):
        if not self.data_dir.startswith('/'):
            if not os.path.isdir(self.data_dir):
                # try to prepend path to this script to basedir
                path = os.path.join(fileDir(), self.data_dir)
                if not os.path.isdir(path):
                    raise ValueError("Data directory '%s' not found in "
                                    "'%s' or current directory" % 
                                    (path, fileDir()))
                self.data_dir = path

class ParserTestCase:
    """Additional functionality for parser unit (etc.) tests.

    """
    def setParser(self, pclass):
        self.parser = pclass(StringIO.StringIO("FAKE"))

    def feed(self, line):
        try:
            result = self.parser.process(line.strip())
        except Exception,E:
            self.fail("Unexpected parse exception (%s) at line: %s" % (E, line))
        return result

    def feedRecord(self, record):
        """Feed a multiline record, one line at a time, to the parser
        and return the result after the last line is fed.
        Will assert failure if any line before the last returns a result
        or raises an error.
        """
        lines = record.split('\n')
        # remove extra empty line, if any
        if lines[-1] == '':
            lines = lines[:-1]
        # parse lines up to last
        for line in lines[:-1]:
            result = self.feed(line)
            self.failIf(result, "Unexpected result after line: %s" % line)
        return self.parser.process(lines[-1])


class BaseParserTestCase(BaseTestCase):
    """Additional shared functionality for parser-module unit tests
    """
    # Base directory for data files; usually a relative path
    basedir = BaseTestCase.data_dir
    # Base filename, typically parser class name, ending with a '.'
    basename = ''
    # Subclasses should set this to their parser class
    parser_class = None

    def setUp(self):
        # For internal use
        self._nlparser = NLFastParser()
        self._dynamic_parser = False

    def checkGood(self, filename='xyz', test=None, parser_kw={},
                  num_expected=-1):
        """Check if data in filename basename + 'filename' is good by running
        the function in 'test' against the results of the parser
        instantiated as 'parser_class(opened-file, **parser_kw)'.
        The function should do some assertions; its return value is
        ignored.

        If the value of test is a function, it is invoked
        for each returned value as test(event, num=number-of-event).
        If it is None, then it passes for any result at all (this really
        makes sense only when num_expected is set).
        """
        if test is None:
            test = self._trueTest
        elif not callable(test):
            self.fail("Test '%s' must be a function" % test) 
        parser = self._initParser(filename, parser_kw)
        n = 0
        for i, event in enumerate(parser):
            self.debug_("(%d) event: %s" % (i,event))
            if event is None:
                continue
            # test the event with the given function
            test(event, num=n)
            n += 1
        self._checkNum(n, num_expected)
        return parser

    def checkBad(self, filename=None, parser_kw={}, num_expected=-1):
        """Same as checkGood() except that there are no tests.

        The 'num_expected' parameter can be used to control how many
        good events are expected.
        """
        parser = self._initParser(filename, parser_kw)
        n = 0
        for event in parser:
            if event is not None:
                n += 1
        self._checkNum(n, num_expected)
        return parser

    def must_have(self, event, attrs, regex=False):
        """Utility function that asserts failure if dictionary 'event'
        is not a superset of dictionary 'attrs'.
        If 'regex' is true, evaluate values as regular expressions, otherwise
        check that the datatype and value of the values are the same.
        """
        for key, value in attrs.items():
            self.failUnless(event.has_key(key), 
                            "event '%s' is missing attribute named '%s'" % 
                            (event, key))
            if regex:
                self.failUnless(re.match(value, event[key]),
                                "event '%s' value for attribute named '%s' "
                                "is '%s', but expected to match the regex "
                                "'%s'" % (event, key, event[key], value))
            else:
                self.failUnless(type(event[key]) == type(value), 
                                "event '%s' type for attribute named '%s' " 
                                "is '%s', but expected '%s'" % 
                                (event, key, type(event[key]), type(value)))
                self.failUnless(event[key] == value, 
                                "event '%s' value for attribute named '%s' " 
                                "is '%s', but expected '%s'" % 
                                (event, key, event[key], value))

    def getFullPath(self, filename):
        self._setDataDir()
        self.basedir = self.data_dir
        return os.path.join(self.basedir, self.basename + filename)

    def setParseDynamic(self, enable, **kw):
        """Enable or disable the use of the 'dynamic' parser, which
        routes to a given parser based on a header. The assumption here
        is that the same parser_class will be the only possible parser,
        but that it will have access to header fields as well.
        This makes it easier to test how parsers behave if they are
        invoked via the dynamic parser module.
        If 'enable' is True, the keywords in '**kw' will be used when the
        dynamic parser gets instantiated (in _initParser).
        """
        if not enable:
            self._dynamic_parser = False
        else:
            self._dynamic_parser = True
            self._dynamic_parser_kw = kw

    def _initParser(self, filename, kw):
        """Open file and init parser class with it
        """
        self.assert_(self.parser_class,
                     "You forgot to set the 'parser_class' attribute in "
                     "the test-case class")
        path = self.getFullPath(filename)
        self.debug_("parsing data file: %s" % path)
        self.failUnless(os.path.isfile(path), "File not found: %s" % path)
        f = file(path)
        if self._dynamic_parser:
            #print "@@ make dynamic parser kw: %s" % self._dynamic_parser_kw
            instance = dynamic.Parser(f, **self._dynamic_parser_kw)
            sub_instance = self.parser_class(f, **kw)
            instance.add('foo', {}, sub_instance)
        else:
            instance = self.parser_class(f, **kw)
        return instance

    def _check(self, test, i, event):
        """Check one event
        """
        self.debug_("check event #%d: %s" % (i, event))
        test(event, num=i)

    def _checkNum(self, i, num_expected):
        if num_expected < 0:
            return
        self.assert_(i == num_expected, "Number of events found (%d) "
                     "does not match number expected (%d)" % 
                     (i, num_expected))

    def _trueTest(self, i, num=0):
        return True

def suite(clazz):
    suite = unittest.makeSuite(clazz,'test')
    return suite

def main():
    """Run either TestOOB or standard unittest"""
    try:
        import testoob
        testoob.main()
    except ImportError:
        unittest.main(defaultTest="suite")
