"""
Unittests for parsers/modules/globus_condor.py
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testParseGlobusCondor.py 23653 2009-04-15 17:46:25Z dang $'

## Imports

from StringIO import StringIO
import unittest
#
from netlogger.parsers.modules.globus_condor import Parser
from netlogger.tests import shared

## Constants and Global Variables

SUBMIT_EVENT = """<c>
    <a n="MyType"><s>SubmitEvent</s></a>
    <a n="EventTypeNumber"><i>0</i></a>
    <a n="MyType"><s>SubmitEvent</s></a>
    <a n="EventTime"><s>2008-02-06T16:14:24</s></a>
    <a n="Cluster"><i>2</i></a>
    <a n="Proc"><i>0</i></a>
    <a n="Subproc"><i>0</i></a>
    <a n="SubmitHost"><s>&lt;131.243.2.182:40152&gt;</s></a>
</c>"""
EXEC_EVENT = """<c>
    <a n="MyType"><s>ExecuteEvent</s></a>
    <a n="EventTypeNumber"><i>1</i></a>
    <a n="MyType"><s>ExecuteEvent</s></a>
    <a n="EventTime"><s>2008-02-06T16:14:27</s></a>
    <a n="Cluster"><i>2</i></a>
    <a n="Proc"><i>0</i></a>
    <a n="Subproc"><i>0</i></a>
    <a n="ExecuteHost"><s>&lt;131.243.2.130:47659&gt;</s></a>
</c>"""
TERM_EVENT = """<c>
    <a n="MyType"><s>JobTerminatedEvent</s></a>
    <a n="EventTypeNumber"><i>5</i></a>
    <a n="MyType"><s>JobTerminatedEvent</s></a>
    <a n="EventTime"><s>2008-02-06T16:14:27</s></a>
    <a n="Cluster"><i>2</i></a>
    <a n="Proc"><i>0</i></a>
    <a n="Subproc"><i>0</i></a>
    <a n="TerminatedNormally"><b v="t"/></a>
    <a n="ReturnValue"><i>0</i></a>
    <a n="RunLocalUsage"><s>Usr 0 00:00:00, Sys 0 00:00:00</s></a>
    <a n="RunRemoteUsage"><s>Usr 0 00:00:00, Sys 0 00:00:00</s></a>
    <a n="TotalLocalUsage"><s>Usr 0 00:00:00, Sys 0 00:00:00</s></a>
    <a n="TotalRemoteUsage"><s>Usr 0 00:00:00, Sys 0 00:00:00</s></a>
    <a n="SentBytes"><r>0.000000000000000E+00</r></a>
    <a n="ReceivedBytes"><r>0.000000000000000E+00</r></a>
    <a n="TotalSentBytes"><r>0.000000000000000E+00</r></a>
    <a n="TotalReceivedBytes"><r>0.000000000000000E+00</r></a>
</c>"""

PREMATURE = """<c>
    <a n="MyType"><s>SubmitEvent</s></a>
<c>"""

# leave off trailing </c> so we can
# easily add lines to it
NOPROC = """<c>
    <a n="MyType"><s>SubmitEvent</s></a>
    <a n="EventTypeNumber"><i>0</i></a>
    <a n="MyType"><s>SubmitEvent</s></a>
    <a n="EventTime"><s>2008-02-06T16:14:24</s></a>
    <a n="SubmitHost"><s>&lt;131.243.2.182:40152&gt;</s></a>"""

NOTS = """<c>
    <a n="MyType"><s>ExecuteEvent</s></a>
    <a n="EventTypeNumber"><i>1</i></a>
    <a n="MyType"><s>ExecuteEvent</s></a>
    <a n="Cluster"><i>2</i></a>
    <a n="Proc"><i>0</i></a>
    <a n="Subproc"><i>0</i></a>
    <a n="ExecuteHost"><s>&lt;131.243.2.130:47659&gt;</s></a>
</c>"""

## Classes

class TestCase(shared.BaseTestCase, shared.ParserTestCase):
    """Unit test cases.
    """

    def setUp(self):
        """Setup actions
        """
        self.setParser(Parser)
        
    def tearDown(self):
        """Any cleanup actions
        """
        pass

    def verifyResult(self, result, expected):
        n = len(result)
        self.failUnless(n == expected, "returned %d results instead of %d" % (n, expected))
        
    def testSubmitSimple(self):
        """Parse SubmitEvent and check that it returns a single result
        """
        self.debug_("Input:\n%s" % SUBMIT_EVENT)
        result = self.feedRecord(SUBMIT_EVENT)
        self.verifyResult(result, 1)

    def testExecSimple(self):
        """Parse ExecuteEvent and check that it returns a single result
        """
        result = self.feedRecord(EXEC_EVENT)
        self.verifyResult(result, 1)

    def testTermSimple(self):
        """Parse JobTerminatedEvent and check that it returns a single result
        """
        result = self.feedRecord(TERM_EVENT)
        self.verifyResult(result, 1)

    def testPremature(self):
        """Premature new record should abort old one"""
        self.assertRaises(ValueError, self.feedRecord, PREMATURE)
        result = self.feedRecord(SUBMIT_EVENT[4:])
        self.verifyResult(result, 1)

    def testMissingClusterProcOrSubproc(self):
        """If any one of cluster, proc, or subproc is missing, should raise a KeyError"""
        for add_fields in (("Cluster"), ("Cluster", "Proc"), ("Cluster", "Subproc"),
                      ("Proc"), ("Subproc"), ("Proc", "Subproc")):
            bad_event = NOPROC
            for field in add_fields:
                formatted_field = '\n<a n="%s"><i>0</i></a>' % field 
                bad_event += formatted_field
            bad_event += "\n</c>"
            self.assertRaises(KeyError, self.feedRecord, bad_event)
        # Make sure next event still parses OK
        result = self.feedRecord(SUBMIT_EVENT)
        self.verifyResult(result, 1)

    def testMissingTimestamp(self):
        """If timestamp is missing, should raise a ValueError"""
        self.assertRaises(KeyError, self.feedRecord, NOTS)
        # Make sure next event still parses OK
        result = self.feedRecord(SUBMIT_EVENT)
        self.verifyResult(result, 1)

    def testBlank(self):
        """Blank line should do nothing"""
        result = self.parser.process("")
        self.verifyResult(result, 0)

    def testNonXML(self):
        """Non-xml lines (garbage) should raise ValueError-s"""
        for line in ("A common mistake that people make", 
                     "when trying to design something completely foolproof",
                     "is to underestimate the ingenuity of complete fools."):
            self.assertRaises(ValueError, self.feedRecord, line)

    def testJobID(self):
        """Job id should be composed from cluster, proc and subproc"""
        result = self.feedRecord(SUBMIT_EVENT)
        self.verifyResult(result, 1)
        jobid = result[0]['job.id']
        expected = '002.000.000'
        self.failUnless(jobid == expected, "Found job.id '%s' != '%s' expected" % (
                jobid, expected))

    def testEventName1(self):
        """Submit event name should be in NetLogger-style in namespace globus.condor"""
        result = self.feedRecord(SUBMIT_EVENT)
        self.verifyResult(result, 1)
        expected = "globus.condor.submit"
        found = result[0]['event']
        self.failUnless(expected == found, "Found event name '%s' != '%s' expected" % (
                found, expected))

    def testEventName2(self):
        """Terminated event name should be in NetLogger-style in namespace globus.condor"""
        result = self.feedRecord(TERM_EVENT)
        self.verifyResult(result, 1)
        expected = "globus.condor.jobTerminated"
        found = result[0]['event']
        self.failUnless(expected == found, "Found event name '%s' != '%s' expected" % (
                found, expected))

# Boilerplate to run the tests
def suite(): 
    return shared.suite(TestCase)
if __name__ == '__main__':
    shared.main()
