#!/usr/bin/env python
"""
Unittests for gridfp "info" event parser
"""
__author__ = 'Dan Gunter dkgunter@lbl.gov'
__rcsid__ = '$Id: testParseGridftp.py 23974 2009-10-20 12:57:44Z dang $'

import logging
import os
import sys
import unittest
#
from netlogger.parsers.modules import gridftp
from netlogger.tests import shared

_opj = os.path.join

class TestCase(shared.BaseTestCase):

    def setUp(self):
        # logging

        log = logging.getLogger("netlogger.parsers.modules.gridftp")
        if self.DEBUG > 0:
            log.addHandler(logging.StreamHandler())
        else:
            log.addHandler(logging.StreamHandler(file("/dev/null", "w")))

        # input files

        self.crazy_us = _opj(self.data_dir, "gridftp.crazy_us")
        self.junky = _opj(self.data_dir, "gridftp.junky")

    def testCrazyMicroseconds(self):
        """Non-zero-padded microseconds in date
        """
        # open file and init parser

        infile = file(self.crazy_us)
        parser = gridftp.Parser(infile, one_event=True)

        # process file, checking that each duration is sane

        for line in infile:
            line = line.strip()
            result = parser.process(line)
            self.failUnless(result, "Could not parse: %s" % line)
            data = result[0]
            dur, nbytes = data['dur'], data['nbytes']
            self.failUnless(dur > 0, "Non-positive duration: %s" % line)
            mbits = (8*nbytes/1e6) / dur
            self.failUnless(mbits < 10000, "Duration too small (%lf): %s" % 
                            (dur, line))

    def testJunk(self):
        """Raise exceptions, don't die, on bad input
        """
        # open file and init parser

        infile = file(self.junky)
        parser = gridftp.Parser(infile)

        # process file, allowing for Key and Value errors from
        # the junky data

        for line in infile:
            line = line.strip()
            try:
                result = parser.process(line)
            except (ValueError, KeyError), E:
                self.debug_("Junk detected: %s" % line)

# Boilerplate to run the tests
def suite(): 
    return shared.suite(TestCase)
if __name__ == '__main__':
    shared.main()
