# This file is part of Neuroinfo Toolkit.
#
# Neuroinfo Toolkit is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Neuroinfo Toolkit is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Neuroinfo Toolkit.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
from neuro.system import exit
from neuro.common import Debugger
from neuro.common import Logger
from neuro.filesystem import dirname
from neuro.exceptions import IllegalArgumentException

__orghook = sys.excepthook

## --- disable stdout/stderr buffering
sys.stdout = os.fdopen(sys.stdout.fileno(), 'w', 0)
sys.stderr = os.fdopen(sys.stderr.fileno(), 'w', 0)

def exit(status=0):
	'''
	Exit the application

	:param status:
	:type status: int
	'''
	if(not isinstance(status, int)):
		raise IllegalArgumentException("Status must be an instance of int")

	exit(status)

def getLogger():
	'''
	Get a reference to the application logger

	:rtype: :class:`~neuro.common.Logger`
	'''
	return Logger.getAppLogger()

def log(message=None):
	'''
	Log an informational message using the application logger
	
	:param message:
	:type message: str
	'''
	Logger.getAppLogger().info(message)

def plain(message=None):
	'''
	Log a plain message using the application logger

	:param message:
	:type message: str
	'''
	Logger.getAppLogger().plain(message)

def info(message=None):
	'''
	Log an informational message using the application logger
	
	:param message:
	:type message: str
	'''
	Logger.getAppLogger().info(message)

def warn(message=None):
	'''
	Log a warning message using the application logger
	
	:param message:
	:type message: str
	'''
	Logger.getAppLogger().warn(message)
	
def error(message=None):
	'''
	Log an error message using the application logger
	
	:param message:
	:type message: str
	'''
	Logger.getAppLogger().error(message)

def fatal(message=None):
	'''
	Log a fatal message using the application logger (exits application)

	:param message:
	:type message: str
	'''
	Logger.getAppLogger().fatal(message)

def exceptionHandler(excType, excValue, strace):
	'''
	Default exception handler. Sends a fatal message to the application logger.
	'''
	debugger = Debugger()
	debugger.setTrace(strace)
	debugger.setException(excValue)

	Logger.getAppLogger().setDebugger(debugger)
	Logger.getAppLogger().fatal(excValue)

def unsetHandler():
	'''
	Unset the NIT exception handler
	'''
	sys.excepthook = __orghook

## --- register exception handler
sys.excepthook = exceptionHandler

## --- get the currently executing script
script = os.path.abspath(sys.argv[0])
scriptDir = dirname(script)