#!/usr/bin/env python

__version__ = '$Revision: 4791 $'.split()[1]
__date__ = '$Date: 2012-10-20 $'.split()[1]
__author__ = 'xmlbinmsg'

__doc__='''

Autogenerated python functions to serialize/deserialize binary messages.

Generated by: ../scripts/aisxmlbinmsg2py.py

Need to then wrap these functions with the outer AIS packet and then
convert the whole binary blob to a NMEA string.  Those functions are
not currently provided in this file.

serialize: python to ais binary
deserialize: ais binary to python

The generated code uses translators.py, binary.py, and aisstring.py
which should be packaged with the resulting files.


@requires: U{epydoc<http://epydoc.sourceforge.net/>} > 3.0alpha3
@requires: U{BitVector<http://cheeseshop.python.org/pypi/BitVector>}

@author: '''+__author__+'''
@version: ''' + __version__ +'''
@var __date__: Date of last svn commit
@undocumented: __version__ __author__ __doc__ parser
@status: under development
@license: Generated code has no license
@todo: FIX: put in a description of the message here with fields and types.
'''

import sys
from decimal import Decimal
from BitVector import BitVector

import binary, aisstring

# FIX: check to see if these will be needed
TrueBV  = BitVector(bitstring="1")
"Why always rebuild the True bit?  This should speed things up a bunch"
FalseBV = BitVector(bitstring="0")
"Why always rebuild the False bit?  This should speed things up a bunch"


fieldList = (
	'MessageID',
	'RepeatIndicator',
	'UserID',
	'Spare',
	'SOG',
	'PositionAccuracy',
	'longitude',
	'latitude',
	'COG',
	'TrueHeading',
	'TimeStamp',
	'Spare2',
	'name',
	'shipandcargo',
	'dimA',
	'dimB',
	'dimC',
	'dimD',
	'fixtype',
	'RAIM',
	'DTE',
	'Spare3',
)

fieldListPostgres = (
	'MessageID',
	'RepeatIndicator',
	'UserID',
	'Spare',
	'SOG',
	'PositionAccuracy',
	'Position',	# PostGIS data type
	'COG',
	'TrueHeading',
	'TimeStamp',
	'Spare2',
	'name',
	'shipandcargo',
	'dimA',
	'dimB',
	'dimC',
	'dimD',
	'fixtype',
	'RAIM',
	'DTE',
	'Spare3',
)

toPgFields = {
	'longitude':'Position',
	'latitude':'Position',
}
'''
Go to the Postgis field names from the straight field name
'''

fromPgFields = {
	'Position':('longitude','latitude',),
}
'''
Go from the Postgis field names to the straight field name
'''

pgTypes = {
	'Position':'POINT',
}
'''
Lookup table for each postgis field name to get its type.
'''

def encode(params, validate=False):
	'''Create a b_pos_and_shipdata binary message payload to pack into an AIS Msg b_pos_and_shipdata.

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 19 (field automatically set to "19")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): Unique ship identification number (MMSI)
	  - Spare(uint): Reseverd for definition by a compentent regional or local authority.  Should be set to zero. (field automatically set to "0")
	  - SOG(udecimal): Speed over ground
	  - PositionAccuracy(uint): Accuracy of positioning fixes
	  - longitude(decimal): Location of the vessel  East West location
	  - latitude(decimal): Location of the vessel  North South location
	  - COG(udecimal): Course over ground
	  - TrueHeading(uint): True heading (relative to true North)
	  - TimeStamp(uint): UTC second when the report was generated
	  - Spare2(uint): Not used.  Should be set to zero.  Researched for future use. (field automatically set to "0")
	  - name(aisstr6): Vessel name
	  - shipandcargo(uint): what
	  - dimA(uint): Distance from bow to reference position
	  - dimB(uint): Distance from reference position to stern
	  - dimC(uint): Distance from port side to reference position
	  - dimD(uint): Distance from reference position to starboard side
	  - fixtype(uint): Method used for positioning
	  - RAIM(bool): Receiver autonomous integrity monitoring flag
	  - DTE(uint): Data terminal ready
	  - Spare3(uint): Not used. Should be set to zero (field automatically set to "0")
	@param params: Dictionary of field names/values.  Throws a ValueError exception if required is missing
	@param validate: Set to true to cause checking to occur.  Runs slower.  FIX: not implemented.
	@rtype: BitVector
	@return: encoded binary message (for binary messages, this needs to be wrapped in a msg 8
	@note: The returned bits may not be 6 bit aligned.  It is up to you to pad out the bits.
	'''

	bvList = []
	bvList.append(binary.setBitVectorSize(BitVector(intVal=19),6))
	if 'RepeatIndicator' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['RepeatIndicator']),2))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),2))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['UserID']),30))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=0),8))
	if 'SOG' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['SOG'])*Decimal('10')))),10))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(1023)),10))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['PositionAccuracy']),1))
	if 'longitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['longitude'])*Decimal('600000')),28))
	else:
		bvList.append(binary.bvFromSignedInt(108600000,28))
	if 'latitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['latitude'])*Decimal('600000')),27))
	else:
		bvList.append(binary.bvFromSignedInt(54600000,27))
	if 'COG' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['COG'])*Decimal('10')))),12))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(3600)),12))
	if 'TrueHeading' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['TrueHeading']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'TimeStamp' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['TimeStamp']),6))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=60),6))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=0),4))
	if 'name' in params:
		bvList.append(aisstring.encode(params['name'],120))
	else:
		bvList.append(aisstring.encode('@@@@@@@@@@@@@@@@@@@@',120))
	if 'shipandcargo' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['shipandcargo']),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),8))
	if 'dimA' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['dimA']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),9))
	if 'dimB' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['dimB']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),9))
	if 'dimC' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['dimC']),6))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),6))
	if 'dimD' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['dimD']),6))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),6))
	if 'fixtype' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['fixtype']),4))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),4))
	if params["RAIM"]: bvList.append(TrueBV)
	else: bvList.append(FalseBV)
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['DTE']),1))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=0),5))

	return binary.joinBV(bvList)

def decode(bv, validate=False):
	'''Unpack a b_pos_and_shipdata message 

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 19 (field automatically set to "19")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): Unique ship identification number (MMSI)
	  - Spare(uint): Reseverd for definition by a compentent regional or local authority.  Should be set to zero. (field automatically set to "0")
	  - SOG(udecimal): Speed over ground
	  - PositionAccuracy(uint): Accuracy of positioning fixes
	  - longitude(decimal): Location of the vessel  East West location
	  - latitude(decimal): Location of the vessel  North South location
	  - COG(udecimal): Course over ground
	  - TrueHeading(uint): True heading (relative to true North)
	  - TimeStamp(uint): UTC second when the report was generated
	  - Spare2(uint): Not used.  Should be set to zero.  Researched for future use. (field automatically set to "0")
	  - name(aisstr6): Vessel name
	  - shipandcargo(uint): what
	  - dimA(uint): Distance from bow to reference position
	  - dimB(uint): Distance from reference position to stern
	  - dimC(uint): Distance from port side to reference position
	  - dimD(uint): Distance from reference position to starboard side
	  - fixtype(uint): Method used for positioning
	  - RAIM(bool): Receiver autonomous integrity monitoring flag
	  - DTE(uint): Data terminal ready
	  - Spare3(uint): Not used. Should be set to zero (field automatically set to "0")
	@type bv: BitVector
	@param bv: Bits defining a message
	@param validate: Set to true to cause checking to occur.  Runs slower.  FIX: not implemented.
	@rtype: dict
	@return: params
	'''

	#Would be nice to check the bit count here..
	#if validate:
	#	assert (len(bv)==FIX: SOME NUMBER)
	r = {}
	r['MessageID']=19
	r['RepeatIndicator']=int(bv[6:8])
	r['UserID']=int(bv[8:38])
	r['Spare']=0
	r['SOG']=Decimal(int(bv[46:56]))/Decimal('10')
	r['PositionAccuracy']=int(bv[56:57])
	r['longitude']=Decimal(binary.signedIntFromBV(bv[57:85]))/Decimal('600000')
	r['latitude']=Decimal(binary.signedIntFromBV(bv[85:112]))/Decimal('600000')
	r['COG']=Decimal(int(bv[112:124]))/Decimal('10')
	r['TrueHeading']=int(bv[124:133])
	r['TimeStamp']=int(bv[133:139])
	r['Spare2']=0
	r['name']=aisstring.decode(bv[143:263])
	r['shipandcargo']=int(bv[263:271])
	r['dimA']=int(bv[271:280])
	r['dimB']=int(bv[280:289])
	r['dimC']=int(bv[289:295])
	r['dimD']=int(bv[295:301])
	r['fixtype']=int(bv[301:305])
	r['RAIM']=bool(int(bv[305:306]))
	r['DTE']=int(bv[306:307])
	r['Spare3']=0
	return r

def decodeMessageID(bv, validate=False):
	return 19

def decodeRepeatIndicator(bv, validate=False):
	return int(bv[6:8])

def decodeUserID(bv, validate=False):
	return int(bv[8:38])

def decodeSpare(bv, validate=False):
	return 0

def decodeSOG(bv, validate=False):
	return Decimal(int(bv[46:56]))/Decimal('10')

def decodePositionAccuracy(bv, validate=False):
	return int(bv[56:57])

def decodelongitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[57:85]))/Decimal('600000')

def decodelatitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[85:112]))/Decimal('600000')

def decodeCOG(bv, validate=False):
	return Decimal(int(bv[112:124]))/Decimal('10')

def decodeTrueHeading(bv, validate=False):
	return int(bv[124:133])

def decodeTimeStamp(bv, validate=False):
	return int(bv[133:139])

def decodeSpare2(bv, validate=False):
	return 0

def decodename(bv, validate=False):
	return aisstring.decode(bv[143:263])

def decodeshipandcargo(bv, validate=False):
	return int(bv[263:271])

def decodedimA(bv, validate=False):
	return int(bv[271:280])

def decodedimB(bv, validate=False):
	return int(bv[280:289])

def decodedimC(bv, validate=False):
	return int(bv[289:295])

def decodedimD(bv, validate=False):
	return int(bv[295:301])

def decodefixtype(bv, validate=False):
	return int(bv[301:305])

def decodeRAIM(bv, validate=False):
	return bool(int(bv[305:306]))

def decodeDTE(bv, validate=False):
	return int(bv[306:307])

def decodeSpare3(bv, validate=False):
	return 0


def printHtml(params, out=sys.stdout):
		out.write("<h3>b_pos_and_shipdata</h3>\n")
		out.write("<table border=\"1\">\n")
		out.write("<tr bgcolor=\"orange\">\n")
		out.write("<th align=\"left\">Field Name</th>\n")
		out.write("<th align=\"left\">Type</th>\n")
		out.write("<th align=\"left\">Value</th>\n")
		out.write("<th align=\"left\">Value in Lookup Table</th>\n")
		out.write("<th align=\"left\">Units</th>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>MessageID</td>\n")
		out.write("<td>uint</td>\n")
		if 'MessageID' in params:
			out.write("	<td>"+str(params['MessageID'])+"</td>\n")
			out.write("	<td>"+str(params['MessageID'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>RepeatIndicator</td>\n")
		out.write("<td>uint</td>\n")
		if 'RepeatIndicator' in params:
			out.write("	<td>"+str(params['RepeatIndicator'])+"</td>\n")
			if str(params['RepeatIndicator']) in RepeatIndicatorDecodeLut:
				out.write("<td>"+RepeatIndicatorDecodeLut[str(params['RepeatIndicator'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>UserID</td>\n")
		out.write("<td>uint</td>\n")
		if 'UserID' in params:
			out.write("	<td>"+str(params['UserID'])+"</td>\n")
			out.write("	<td>"+str(params['UserID'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>Spare</td>\n")
		out.write("<td>uint</td>\n")
		if 'Spare' in params:
			out.write("	<td>"+str(params['Spare'])+"</td>\n")
			out.write("	<td>"+str(params['Spare'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>SOG</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'SOG' in params:
			out.write("	<td>"+str(params['SOG'])+"</td>\n")
			if str(params['SOG']) in SOGDecodeLut:
				out.write("<td>"+SOGDecodeLut[str(params['SOG'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>knots</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>PositionAccuracy</td>\n")
		out.write("<td>uint</td>\n")
		if 'PositionAccuracy' in params:
			out.write("	<td>"+str(params['PositionAccuracy'])+"</td>\n")
			if str(params['PositionAccuracy']) in PositionAccuracyDecodeLut:
				out.write("<td>"+PositionAccuracyDecodeLut[str(params['PositionAccuracy'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>longitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'longitude' in params:
			out.write("	<td>"+str(params['longitude'])+"</td>\n")
			out.write("	<td>"+str(params['longitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>latitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'latitude' in params:
			out.write("	<td>"+str(params['latitude'])+"</td>\n")
			out.write("	<td>"+str(params['latitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>COG</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'COG' in params:
			out.write("	<td>"+str(params['COG'])+"</td>\n")
			out.write("	<td>"+str(params['COG'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>TrueHeading</td>\n")
		out.write("<td>uint</td>\n")
		if 'TrueHeading' in params:
			out.write("	<td>"+str(params['TrueHeading'])+"</td>\n")
			out.write("	<td>"+str(params['TrueHeading'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>TimeStamp</td>\n")
		out.write("<td>uint</td>\n")
		if 'TimeStamp' in params:
			out.write("	<td>"+str(params['TimeStamp'])+"</td>\n")
			if str(params['TimeStamp']) in TimeStampDecodeLut:
				out.write("<td>"+TimeStampDecodeLut[str(params['TimeStamp'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>seconds</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>Spare2</td>\n")
		out.write("<td>uint</td>\n")
		if 'Spare2' in params:
			out.write("	<td>"+str(params['Spare2'])+"</td>\n")
			out.write("	<td>"+str(params['Spare2'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>name</td>\n")
		out.write("<td>aisstr6</td>\n")
		if 'name' in params:
			out.write("	<td>"+str(params['name'])+"</td>\n")
			out.write("	<td>"+str(params['name'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>shipandcargo</td>\n")
		out.write("<td>uint</td>\n")
		if 'shipandcargo' in params:
			out.write("	<td>"+str(params['shipandcargo'])+"</td>\n")
			if str(params['shipandcargo']) in shipandcargoDecodeLut:
				out.write("<td>"+shipandcargoDecodeLut[str(params['shipandcargo'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>dimA</td>\n")
		out.write("<td>uint</td>\n")
		if 'dimA' in params:
			out.write("	<td>"+str(params['dimA'])+"</td>\n")
			out.write("	<td>"+str(params['dimA'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>dimB</td>\n")
		out.write("<td>uint</td>\n")
		if 'dimB' in params:
			out.write("	<td>"+str(params['dimB'])+"</td>\n")
			out.write("	<td>"+str(params['dimB'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>dimC</td>\n")
		out.write("<td>uint</td>\n")
		if 'dimC' in params:
			out.write("	<td>"+str(params['dimC'])+"</td>\n")
			if str(params['dimC']) in dimCDecodeLut:
				out.write("<td>"+dimCDecodeLut[str(params['dimC'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>dimD</td>\n")
		out.write("<td>uint</td>\n")
		if 'dimD' in params:
			out.write("	<td>"+str(params['dimD'])+"</td>\n")
			if str(params['dimD']) in dimDDecodeLut:
				out.write("<td>"+dimDDecodeLut[str(params['dimD'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>fixtype</td>\n")
		out.write("<td>uint</td>\n")
		if 'fixtype' in params:
			out.write("	<td>"+str(params['fixtype'])+"</td>\n")
			if str(params['fixtype']) in fixtypeDecodeLut:
				out.write("<td>"+fixtypeDecodeLut[str(params['fixtype'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>RAIM</td>\n")
		out.write("<td>bool</td>\n")
		if 'RAIM' in params:
			out.write("	<td>"+str(params['RAIM'])+"</td>\n")
			if str(params['RAIM']) in RAIMDecodeLut:
				out.write("<td>"+RAIMDecodeLut[str(params['RAIM'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>DTE</td>\n")
		out.write("<td>uint</td>\n")
		if 'DTE' in params:
			out.write("	<td>"+str(params['DTE'])+"</td>\n")
			if str(params['DTE']) in DTEDecodeLut:
				out.write("<td>"+DTEDecodeLut[str(params['DTE'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>Spare3</td>\n")
		out.write("<td>uint</td>\n")
		if 'Spare3' in params:
			out.write("	<td>"+str(params['Spare3'])+"</td>\n")
			out.write("	<td>"+str(params['Spare3'])+"</td>\n")
		out.write("</tr>\n")
		out.write("</table>\n")


def printKml(params, out=sys.stdout):
	'''KML (Keyhole Markup Language) for Google Earth, but without the header/footer'''
	out.write("\	<Placemark>\n")
	out.write("\t	<name>"+str(params['UserID'])+"</name>\n")
	out.write("\t\t<description>\n")
	import StringIO
	buf = StringIO.StringIO()
	printHtml(params,buf)
	import cgi
	out.write(cgi.escape(buf.getvalue()))
	out.write("\t\t</description>\n")
	out.write("\t\t<styleUrl>#m_ylw-pushpin_copy0</styleUrl>\n")
	out.write("\t\t<Point>\n")
	out.write("\t\t\t<coordinates>")
	out.write(str(params['longitude']))
	out.write(',')
	out.write(str(params['latitude']))
	out.write(",0</coordinates>\n")
	out.write("\t\t</Point>\n")
	out.write("\t</Placemark>\n")

def printFields(params, out=sys.stdout, format='std', fieldList=None, dbType='postgres'):
	'''Print a b_pos_and_shipdata message to stdout.

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 19 (field automatically set to "19")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): Unique ship identification number (MMSI)
	  - Spare(uint): Reseverd for definition by a compentent regional or local authority.  Should be set to zero. (field automatically set to "0")
	  - SOG(udecimal): Speed over ground
	  - PositionAccuracy(uint): Accuracy of positioning fixes
	  - longitude(decimal): Location of the vessel  East West location
	  - latitude(decimal): Location of the vessel  North South location
	  - COG(udecimal): Course over ground
	  - TrueHeading(uint): True heading (relative to true North)
	  - TimeStamp(uint): UTC second when the report was generated
	  - Spare2(uint): Not used.  Should be set to zero.  Researched for future use. (field automatically set to "0")
	  - name(aisstr6): Vessel name
	  - shipandcargo(uint): what
	  - dimA(uint): Distance from bow to reference position
	  - dimB(uint): Distance from reference position to stern
	  - dimC(uint): Distance from port side to reference position
	  - dimD(uint): Distance from reference position to starboard side
	  - fixtype(uint): Method used for positioning
	  - RAIM(bool): Receiver autonomous integrity monitoring flag
	  - DTE(uint): Data terminal ready
	  - Spare3(uint): Not used. Should be set to zero (field automatically set to "0")
	@param params: Dictionary of field names/values.  
	@param out: File like object to write to
	@rtype: stdout
	@return: text to out
	'''

	if 'std'==format:
		out.write("b_pos_and_shipdata:\n")
		if 'MessageID' in params: out.write("	MessageID:         "+str(params['MessageID'])+"\n")
		if 'RepeatIndicator' in params: out.write("	RepeatIndicator:   "+str(params['RepeatIndicator'])+"\n")
		if 'UserID' in params: out.write("	UserID:            "+str(params['UserID'])+"\n")
		if 'Spare' in params: out.write("	Spare:             "+str(params['Spare'])+"\n")
		if 'SOG' in params: out.write("	SOG:               "+str(params['SOG'])+"\n")
		if 'PositionAccuracy' in params: out.write("	PositionAccuracy:  "+str(params['PositionAccuracy'])+"\n")
		if 'longitude' in params: out.write("	longitude:         "+str(params['longitude'])+"\n")
		if 'latitude' in params: out.write("	latitude:          "+str(params['latitude'])+"\n")
		if 'COG' in params: out.write("	COG:               "+str(params['COG'])+"\n")
		if 'TrueHeading' in params: out.write("	TrueHeading:       "+str(params['TrueHeading'])+"\n")
		if 'TimeStamp' in params: out.write("	TimeStamp:         "+str(params['TimeStamp'])+"\n")
		if 'Spare2' in params: out.write("	Spare2:            "+str(params['Spare2'])+"\n")
		if 'name' in params: out.write("	name:              "+str(params['name'])+"\n")
		if 'shipandcargo' in params: out.write("	shipandcargo:      "+str(params['shipandcargo'])+"\n")
		if 'dimA' in params: out.write("	dimA:              "+str(params['dimA'])+"\n")
		if 'dimB' in params: out.write("	dimB:              "+str(params['dimB'])+"\n")
		if 'dimC' in params: out.write("	dimC:              "+str(params['dimC'])+"\n")
		if 'dimD' in params: out.write("	dimD:              "+str(params['dimD'])+"\n")
		if 'fixtype' in params: out.write("	fixtype:           "+str(params['fixtype'])+"\n")
		if 'RAIM' in params: out.write("	RAIM:              "+str(params['RAIM'])+"\n")
		if 'DTE' in params: out.write("	DTE:               "+str(params['DTE'])+"\n")
		if 'Spare3' in params: out.write("	Spare3:            "+str(params['Spare3'])+"\n")
	elif 'csv'==format:
		if None == options.fieldList:
			options.fieldList = fieldList
		needComma = False;
		for field in fieldList:
			if needComma: out.write(',')
			needComma = True
			if field in params:
				out.write(str(params[field]))
			# else: leave it empty
		out.write("\n")
	elif 'html'==format:
		printHtml(params,out)
	elif 'sql'==format:
		sqlInsertStr(params,out,dbType=dbType)
	elif 'kml'==format:
		printKml(params,out)
	elif 'kml-full'==format:
		out.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n")
		out.write("<kml xmlns=\"http://earth.google.com/kml/2.1\">\n")
		out.write("<Document>\n")
		out.write("	<name>b_pos_and_shipdata</name>\n")
		printKml(params,out)
		out.write("</Document>\n")
		out.write("</kml>\n")
	else: 
		print "ERROR: unknown format:",format
		assert False

	return # Nothing to return

RepeatIndicatorEncodeLut = {
	'default':'0',
	'do not repeat any more':'3',
	} #RepeatIndicatorEncodeLut

RepeatIndicatorDecodeLut = {
	'0':'default',
	'3':'do not repeat any more',
	} # RepeatIndicatorEncodeLut

SOGEncodeLut = {
	'102.2 knots or higher':'102.2',
	} #SOGEncodeLut

SOGDecodeLut = {
	'102.2':'102.2 knots or higher',
	} # SOGEncodeLut

PositionAccuracyEncodeLut = {
	'low (greater than 10 m)':'0',
	'high (less than 10 m)':'1',
	} #PositionAccuracyEncodeLut

PositionAccuracyDecodeLut = {
	'0':'low (greater than 10 m)',
	'1':'high (less than 10 m)',
	} # PositionAccuracyEncodeLut

TimeStampEncodeLut = {
	'not available/default':'60',
	'manual input':'61',
	'dead reckoning':'62',
	'inoperative':'63',
	} #TimeStampEncodeLut

TimeStampDecodeLut = {
	'60':'not available/default',
	'61':'manual input',
	'62':'dead reckoning',
	'63':'inoperative',
	} # TimeStampEncodeLut

shipandcargoEncodeLut = {
	'Wing in ground (WIG), all ships of this type':'20',
	'Wing in ground (WIG), Hazardous catagory A':'21',
	'Wing in ground (WIG), Hazardous catagory B':'22',
	'Wing in ground (WIG), Hazardous catagory C':'23',
	'Wing in ground (WIG), Hazardous catagory D':'24',
	'Wing in ground (WIG), Reserved for future use':'25',
	'Wing in ground (WIG), Reserved for future use':'26',
	'Wing in ground (WIG), Reserved for future use':'27',
	'Wing in ground (WIG), Reserved for future use':'28',
	'Wing in ground (WIG), No additional information':'29',
	'fishing':'30',
	'towing':'31',
	'towing length exceeds 200m or breadth exceeds 25m':'32',
	'dredging or underwater ops':'33',
	'diving ops':'34',
	'military ops':'35',
	'sailing':'36',
	'pleasure craft':'37',
	'reserved':'38',
	'reserved':'39',
	'High speed craft (HSC), all ships of this type':'40',
	'High speed craft (HSC), Hazardous catagory A':'41',
	'High speed craft (HSC), Hazardous catagory B':'42',
	'High speed craft (HSC), Hazardous catagory C':'43',
	'High speed craft (HSC), Hazardous catagory D':'44',
	'High speed craft (HSC), Reserved for future use':'45',
	'High speed craft (HSC), Reserved for future use':'46',
	'High speed craft (HSC), Reserved for future use':'47',
	'High speed craft (HSC), Reserved for future use':'48',
	'High speed craft (HSC), No additional information':'49',
	'pilot vessel':'50',
	'search and rescue vessel':'51',
	'tug':'52',
	'port tender':'53',
	'anti-polution equipment':'54',
	'law enforcement':'55',
	'spare - local vessel':'56',
	'spare - local vessel':'57',
	'medical transport':'58',
	'ship according to RR Resolution No. 18':'59',
	'passenger, all ships of this type':'60',
	'passenger, Hazardous catagory A':'61',
	'passenger, Hazardous catagory B':'62',
	'passenger, Hazardous catagory C':'63',
	'passenger, Hazardous catagory D':'64',
	'passenger, Reserved for future use':'65',
	'passenger, Reserved for future use':'66',
	'passenger, Reserved for future use':'67',
	'passenger, Reserved for future use':'68',
	'passenger, No additional information':'69',
	'cargo, all ships of this type':'70',
	'cargo, Hazardous catagory A':'71',
	'cargo, Hazardous catagory B':'72',
	'cargo, Hazardous catagory C':'73',
	'cargo, Hazardous catagory D':'74',
	'cargo, Reserved for future use':'75',
	'cargo, Reserved for future use':'76',
	'cargo, Reserved for future use':'77',
	'cargo, Reserved for future use':'78',
	'cargo, No additional information':'79',
	'tanker, all ships of this type':'80',
	'tanker, Hazardous catagory A':'81',
	'tanker, Hazardous catagory B':'82',
	'tanker, Hazardous catagory C':'83',
	'tanker, Hazardous catagory D':'84',
	'tanker, Reserved for future use':'85',
	'tanker, Reserved for future use':'86',
	'tanker, Reserved for future use':'87',
	'tanker, Reserved for future use':'88',
	'tanker, No additional information':'89',
	'other type, all ships of this type':'90',
	'other type, Hazardous catagory A':'91',
	'other type, Hazardous catagory B':'92',
	'other type, Hazardous catagory C':'93',
	'other type, Hazardous catagory D':'94',
	'other type, Reserved for future use':'95',
	'other type, Reserved for future use':'96',
	'other type, Reserved for future use':'97',
	'other type, Reserved for future use':'98',
	'other type, No additional information':'99',
	} #shipandcargoEncodeLut

shipandcargoDecodeLut = {
	'20':'Wing in ground (WIG), all ships of this type',
	'21':'Wing in ground (WIG), Hazardous catagory A',
	'22':'Wing in ground (WIG), Hazardous catagory B',
	'23':'Wing in ground (WIG), Hazardous catagory C',
	'24':'Wing in ground (WIG), Hazardous catagory D',
	'25':'Wing in ground (WIG), Reserved for future use',
	'26':'Wing in ground (WIG), Reserved for future use',
	'27':'Wing in ground (WIG), Reserved for future use',
	'28':'Wing in ground (WIG), Reserved for future use',
	'29':'Wing in ground (WIG), No additional information',
	'30':'fishing',
	'31':'towing',
	'32':'towing length exceeds 200m or breadth exceeds 25m',
	'33':'dredging or underwater ops',
	'34':'diving ops',
	'35':'military ops',
	'36':'sailing',
	'37':'pleasure craft',
	'38':'reserved',
	'39':'reserved',
	'40':'High speed craft (HSC), all ships of this type',
	'41':'High speed craft (HSC), Hazardous catagory A',
	'42':'High speed craft (HSC), Hazardous catagory B',
	'43':'High speed craft (HSC), Hazardous catagory C',
	'44':'High speed craft (HSC), Hazardous catagory D',
	'45':'High speed craft (HSC), Reserved for future use',
	'46':'High speed craft (HSC), Reserved for future use',
	'47':'High speed craft (HSC), Reserved for future use',
	'48':'High speed craft (HSC), Reserved for future use',
	'49':'High speed craft (HSC), No additional information',
	'50':'pilot vessel',
	'51':'search and rescue vessel',
	'52':'tug',
	'53':'port tender',
	'54':'anti-polution equipment',
	'55':'law enforcement',
	'56':'spare - local vessel',
	'57':'spare - local vessel',
	'58':'medical transport',
	'59':'ship according to RR Resolution No. 18',
	'60':'passenger, all ships of this type',
	'61':'passenger, Hazardous catagory A',
	'62':'passenger, Hazardous catagory B',
	'63':'passenger, Hazardous catagory C',
	'64':'passenger, Hazardous catagory D',
	'65':'passenger, Reserved for future use',
	'66':'passenger, Reserved for future use',
	'67':'passenger, Reserved for future use',
	'68':'passenger, Reserved for future use',
	'69':'passenger, No additional information',
	'70':'cargo, all ships of this type',
	'71':'cargo, Hazardous catagory A',
	'72':'cargo, Hazardous catagory B',
	'73':'cargo, Hazardous catagory C',
	'74':'cargo, Hazardous catagory D',
	'75':'cargo, Reserved for future use',
	'76':'cargo, Reserved for future use',
	'77':'cargo, Reserved for future use',
	'78':'cargo, Reserved for future use',
	'79':'cargo, No additional information',
	'80':'tanker, all ships of this type',
	'81':'tanker, Hazardous catagory A',
	'82':'tanker, Hazardous catagory B',
	'83':'tanker, Hazardous catagory C',
	'84':'tanker, Hazardous catagory D',
	'85':'tanker, Reserved for future use',
	'86':'tanker, Reserved for future use',
	'87':'tanker, Reserved for future use',
	'88':'tanker, Reserved for future use',
	'89':'tanker, No additional information',
	'90':'other type, all ships of this type',
	'91':'other type, Hazardous catagory A',
	'92':'other type, Hazardous catagory B',
	'93':'other type, Hazardous catagory C',
	'94':'other type, Hazardous catagory D',
	'95':'other type, Reserved for future use',
	'96':'other type, Reserved for future use',
	'97':'other type, Reserved for future use',
	'98':'other type, Reserved for future use',
	'99':'other type, No additional information',
	} # shipandcargoEncodeLut

dimCEncodeLut = {
	'63 m or greater':'63',
	} #dimCEncodeLut

dimCDecodeLut = {
	'63':'63 m or greater',
	} # dimCEncodeLut

dimDEncodeLut = {
	'63 m or greater':'63',
	} #dimDEncodeLut

dimDDecodeLut = {
	'63':'63 m or greater',
	} # dimDEncodeLut

fixtypeEncodeLut = {
	'undefined':'0',
	'GPS':'1',
	'GLONASS':'2',
	'combined GPS/GLONASS':'3',
	'Loran-C':'4',
	'Chayka':'5',
	'integrated navigation system':'6',
	'surveyed':'7',
	} #fixtypeEncodeLut

fixtypeDecodeLut = {
	'0':'undefined',
	'1':'GPS',
	'2':'GLONASS',
	'3':'combined GPS/GLONASS',
	'4':'Loran-C',
	'5':'Chayka',
	'6':'integrated navigation system',
	'7':'surveyed',
	} # fixtypeEncodeLut

RAIMEncodeLut = {
	'not in use':'False',
	'in use':'True',
	} #RAIMEncodeLut

RAIMDecodeLut = {
	'False':'not in use',
	'True':'in use',
	} # RAIMEncodeLut

DTEEncodeLut = {
	'available':'0',
	'not available':'1',
	} #DTEEncodeLut

DTEDecodeLut = {
	'0':'available',
	'1':'not available',
	} # DTEEncodeLut

######################################################################
# SQL SUPPORT
######################################################################

dbTableName='b_pos_and_shipdata'
'Database table name'

def sqlCreateStr(outfile=sys.stdout, fields=None, extraFields=None
		,addCoastGuardFields=True
		,dbType='postgres'
		):
	'''
	Return the SQL CREATE command for this message type
	@param outfile: file like object to print to.
	@param fields: which fields to put in the create.  Defaults to all.
	@param extraFields: A sequence of tuples containing (name,sql type) for additional fields
	@param addCoastGuardFields: Add the extra fields that come after the NMEA check some from the USCG N-AIS format
	@param dbType: Which flavor of database we are using so that the create is tailored ('sqlite' or 'postgres')
	@type addCoastGuardFields: bool
	@return: sql create string
	@rtype: str

	@see: sqlCreate
	'''
	# FIX: should this sqlCreate be the same as in LaTeX (createFuncName) rather than hard coded?
	outfile.write(str(sqlCreate(fields,extraFields,addCoastGuardFields,dbType=dbType)))

def sqlCreate(fields=None, extraFields=None, addCoastGuardFields=True, dbType='postgres'):
	'''
	Return the sqlhelp object to create the table.

	@param fields: which fields to put in the create.  Defaults to all.
	@param extraFields: A sequence of tuples containing (name,sql type) for additional fields
	@param addCoastGuardFields: Add the extra fields that come after the NMEA check some from the USCG N-AIS format
	@type addCoastGuardFields: bool
	@param dbType: Which flavor of database we are using so that the create is tailored ('sqlite' or 'postgres')
	@return: An object that can be used to generate a return
	@rtype: sqlhelp.create
	'''
	if None == fields: fields = fieldList
	import sqlhelp
	c = sqlhelp.create('b_pos_and_shipdata',dbType=dbType)
	c.addPrimaryKey()
	if 'MessageID' in fields: c.addInt ('MessageID')
	if 'RepeatIndicator' in fields: c.addInt ('RepeatIndicator')
	if 'UserID' in fields: c.addInt ('UserID')
	if 'Spare' in fields: c.addInt ('Spare')
	if 'SOG' in fields: c.addDecimal('SOG',4,1)
	if 'PositionAccuracy' in fields: c.addInt ('PositionAccuracy')
	if dbType != 'postgres':
		if 'longitude' in fields: c.addDecimal('longitude',8,5)
	if dbType != 'postgres':
		if 'latitude' in fields: c.addDecimal('latitude',8,5)
	if 'COG' in fields: c.addDecimal('COG',4,1)
	if 'TrueHeading' in fields: c.addInt ('TrueHeading')
	if 'TimeStamp' in fields: c.addInt ('TimeStamp')
	if 'Spare2' in fields: c.addInt ('Spare2')
	if 'name' in fields: c.addVarChar('name',20)
	if 'shipandcargo' in fields: c.addInt ('shipandcargo')
	if 'dimA' in fields: c.addInt ('dimA')
	if 'dimB' in fields: c.addInt ('dimB')
	if 'dimC' in fields: c.addInt ('dimC')
	if 'dimD' in fields: c.addInt ('dimD')
	if 'fixtype' in fields: c.addInt ('fixtype')
	if 'RAIM' in fields: c.addBool('RAIM')
	if 'DTE' in fields: c.addInt ('DTE')
	if 'Spare3' in fields: c.addInt ('Spare3')

	if addCoastGuardFields:
		# c.addInt('cg_s_rssi')     # Relative signal strength indicator
		# c.addInt('cg_d_strength')        # dBm receive strength
		# c.addVarChar('cg_x',10) # Idonno
		c.addInt('cg_t_arrival')        # Receive timestamp from the AIS equipment 'T'
		c.addInt('cg_s_slotnum')        # Slot received in
		c.addVarChar('cg_r',15)   # Receiver station ID  -  should usually be an MMSI, but sometimes is a string
		c.addInt('cg_sec')        # UTC seconds since the epoch

		c.addTimestamp('cg_timestamp') # UTC decoded cg_sec - not actually in the data stream

	if dbType == 'postgres':
		#--- EPSG 4326 : WGS 84
		#INSERT INTO "spatial_ref_sys" ("srid","auth_name","auth_srid","srtext","proj4text") VALUES (4326,'EPSG',4326,'GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.01745329251994328,AUTHORITY["EPSG","9122"]],AUTHORITY["EPSG","4326"]]','+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs ');
		c.addPostGIS('Position','POINT',2,SRID=4326);

	return c

def sqlInsertStr(params, outfile=sys.stdout, extraParams=None, dbType='postgres'):
	'''
	Return the SQL INSERT command for this message type
	@param params: dictionary of values keyed by field name
	@param outfile: file like object to print to.
	@param extraParams: A sequence of tuples containing (name,sql type) for additional fields
	@return: sql create string
	@rtype: str

	@see: sqlCreate
	'''
	outfile.write(str(sqlInsert(params,extraParams,dbType=dbType)))


def sqlInsert(params,extraParams=None,dbType='postgres'):
	'''
	Give the SQL INSERT statement
	@param params: dict keyed by field name of values
	@param extraParams: any extra fields that you have created beyond the normal ais message fields
	@rtype: sqlhelp.insert
	@return: insert class instance
	@todo: allow optional type checking of params?
	@warning: this will take invalid keys happily and do what???
	'''
	import sqlhelp
	i = sqlhelp.insert('b_pos_and_shipdata',dbType=dbType)

	if dbType=='postgres':
		finished = []
		for key in params:
			if key in finished: 
				continue

			if key not in toPgFields and key not in fromPgFields:
				if type(params[key])==Decimal: i.add(key,float(params[key]))
				else: i.add(key,params[key])
			else:
				if key in fromPgFields:
					val = params[key]
				        # Had better be a WKT type like POINT(-88.1 30.321)
					i.addPostGIS(key,val)
					finished.append(key)
				else:
					# Need to construct the type.
					pgName = toPgFields[key]
					#valStr='GeomFromText(\''+pgTypes[pgName]+'('
					valStr=pgTypes[pgName]+'('
					vals = []
					for nonPgKey in fromPgFields[pgName]:
						vals.append(str(params[nonPgKey]))
						finished.append(nonPgKey)
					valStr+=' '.join(vals)+')'
					i.addPostGIS(pgName,valStr)
	else:
		for key in params: 
			if type(params[key])==Decimal: i.add(key,float(params[key]))
			else: i.add(key,params[key])

	if None != extraParams:
		for key in extraParams: 
			i.add(key,extraParams[key])

	return i

######################################################################
# LATEX SUPPORT
######################################################################

def latexDefinitionTable(outfile=sys.stdout
		):
	'''
	Return the LaTeX definition table for this message type
	@param outfile: file like object to print to.
	@type outfile: file obj
	@return: LaTeX table string via the outfile
	@rtype: str

	'''
	o = outfile

	o.write('''
\\begin{table}%[htb]
\\centering
\\begin{tabular}{|l|c|l|}
\\hline
Parameter & Number of bits & Description 
\\\\  \\hline\\hline
MessageID & 6 & AIS message number.  Must be 19 \\\\ \hline 
RepeatIndicator & 2 & Indicated how many times a message has been repeated \\\\ \hline 
UserID & 30 & Unique ship identification number (MMSI) \\\\ \hline 
Spare & 8 & Reseverd for definition by a compentent regional or local authority.  Should be set to zero. \\\\ \hline 
SOG & 10 & Speed over ground \\\\ \hline 
PositionAccuracy & 1 & Accuracy of positioning fixes \\\\ \hline 
longitude & 28 & Location of the vessel  East West location \\\\ \hline 
latitude & 27 & Location of the vessel  North South location \\\\ \hline 
COG & 12 & Course over ground \\\\ \hline 
TrueHeading & 9 & True heading (relative to true North) \\\\ \hline 
TimeStamp & 6 & UTC second when the report was generated \\\\ \hline 
Spare2 & 4 & Not used.  Should be set to zero.  Researched for future use. \\\\ \hline 
name & 120 & Vessel name \\\\ \hline 
shipandcargo & 8 & Type of ship and cargo type \\\\ \hline 
dimA & 9 & Distance from bow to reference position \\\\ \hline 
dimB & 9 & Distance from reference position to stern \\\\ \hline 
dimC & 6 & Distance from port side to reference position \\\\ \hline 
dimD & 6 & Distance from reference position to starboard side \\\\ \hline 
fixtype & 4 & Method used for positioning \\\\ \hline 
RAIM & 1 & Receiver autonomous integrity monitoring flag \\\\ \hline 
DTE & 1 & Data terminal ready \\\\ \hline 
Spare3 & 5 & Not used. Should be set to zero\\\\ \\hline \\hline
Total bits & 312 & Appears to take 2 slots with 112 pad bits to fill the last slot \\\\ \\hline
\\end{tabular}
\\caption{AIS message number 19: Extended Class B equipment position report - NOT WORKING IN NOAADATA}
\\label{tab:b_pos_and_shipdata}
\\end{table}
''')

######################################################################
# Text Definition
######################################################################

def textDefinitionTable(outfile=sys.stdout
		,delim='\t'
		):
	'''
	Return the text definition table for this message type
	@param outfile: file like object to print to.
	@type outfile: file obj
	@return: text table string via the outfile
	@rtype: str

	'''
	o = outfile
	o.write('''Parameter'''+delim+'Number of bits'''+delim+'''Description 
MessageID'''+delim+'''6'''+delim+'''AIS message number.  Must be 19
RepeatIndicator'''+delim+'''2'''+delim+'''Indicated how many times a message has been repeated
UserID'''+delim+'''30'''+delim+'''Unique ship identification number (MMSI)
Spare'''+delim+'''8'''+delim+'''Reseverd for definition by a compentent regional or local authority.  Should be set to zero.
SOG'''+delim+'''10'''+delim+'''Speed over ground
PositionAccuracy'''+delim+'''1'''+delim+'''Accuracy of positioning fixes
longitude'''+delim+'''28'''+delim+'''Location of the vessel  East West location
latitude'''+delim+'''27'''+delim+'''Location of the vessel  North South location
COG'''+delim+'''12'''+delim+'''Course over ground
TrueHeading'''+delim+'''9'''+delim+'''True heading (relative to true North)
TimeStamp'''+delim+'''6'''+delim+'''UTC second when the report was generated
Spare2'''+delim+'''4'''+delim+'''Not used.  Should be set to zero.  Researched for future use.
name'''+delim+'''120'''+delim+'''Vessel name
shipandcargo'''+delim+'''8'''+delim+'''Type of ship and cargo type
dimA'''+delim+'''9'''+delim+'''Distance from bow to reference position
dimB'''+delim+'''9'''+delim+'''Distance from reference position to stern
dimC'''+delim+'''6'''+delim+'''Distance from port side to reference position
dimD'''+delim+'''6'''+delim+'''Distance from reference position to starboard side
fixtype'''+delim+'''4'''+delim+'''Method used for positioning
RAIM'''+delim+'''1'''+delim+'''Receiver autonomous integrity monitoring flag
DTE'''+delim+'''1'''+delim+'''Data terminal ready
Spare3'''+delim+'''5'''+delim+'''Not used. Should be set to zero
Total bits'''+delim+'''312'''+delim+'''Appears to take 2 slots with 112 pad bits to fill the last slot''')


######################################################################
# UNIT TESTING
######################################################################
import unittest
def testParams():
	'''Return a params file base on the testvalue tags.
	@rtype: dict
	@return: params based on testvalue tags
	'''
	params = {}
	params['MessageID'] = 19
	params['RepeatIndicator'] = 1
	params['UserID'] = 1193046
	params['Spare'] = 0
	params['SOG'] = Decimal('101.9')
	params['PositionAccuracy'] = 1
	params['longitude'] = Decimal('-122.16328055555556')
	params['latitude'] = Decimal('37.424458333333334')
	params['COG'] = Decimal('34.5')
	params['TrueHeading'] = 41
	params['TimeStamp'] = 35
	params['Spare2'] = 0
	params['name'] = 'BLACK PEARL@@@@@@@@@'
	params['shipandcargo'] = 55
	params['dimA'] = 10
	params['dimB'] = 11
	params['dimC'] = 12
	params['dimD'] = 13
	params['fixtype'] = 1
	params['RAIM'] = False
	params['DTE'] = 0
	params['Spare3'] = 0

	return params

class Testb_pos_and_shipdata(unittest.TestCase):
	'''Use testvalue tag text from each type to build test case the b_pos_and_shipdata message'''
	def testEncodeDecode(self):

		params = testParams()
		bits   = encode(params)
		r      = decode(bits)

		# Check that each parameter came through ok.
		self.failUnlessEqual(r['MessageID'],params['MessageID'])
		self.failUnlessEqual(r['RepeatIndicator'],params['RepeatIndicator'])
		self.failUnlessEqual(r['UserID'],params['UserID'])
		self.failUnlessEqual(r['Spare'],params['Spare'])
		self.failUnlessAlmostEqual(r['SOG'],params['SOG'],1)
		self.failUnlessEqual(r['PositionAccuracy'],params['PositionAccuracy'])
		self.failUnlessAlmostEqual(r['longitude'],params['longitude'],5)
		self.failUnlessAlmostEqual(r['latitude'],params['latitude'],5)
		self.failUnlessAlmostEqual(r['COG'],params['COG'],1)
		self.failUnlessEqual(r['TrueHeading'],params['TrueHeading'])
		self.failUnlessEqual(r['TimeStamp'],params['TimeStamp'])
		self.failUnlessEqual(r['Spare2'],params['Spare2'])
		self.failUnlessEqual(r['name'],params['name'])
		self.failUnlessEqual(r['shipandcargo'],params['shipandcargo'])
		self.failUnlessEqual(r['dimA'],params['dimA'])
		self.failUnlessEqual(r['dimB'],params['dimB'])
		self.failUnlessEqual(r['dimC'],params['dimC'])
		self.failUnlessEqual(r['dimD'],params['dimD'])
		self.failUnlessEqual(r['fixtype'],params['fixtype'])
		self.failUnlessEqual(r['RAIM'],params['RAIM'])
		self.failUnlessEqual(r['DTE'],params['DTE'])
		self.failUnlessEqual(r['Spare3'],params['Spare3'])

def addMsgOptions(parser):
	parser.add_option('-d','--decode',dest='doDecode',default=False,action='store_true',
		help='decode a "b_pos_and_shipdata" AIS message')
	parser.add_option('-e','--encode',dest='doEncode',default=False,action='store_true',
		help='encode a "b_pos_and_shipdata" AIS message')
	parser.add_option('--RepeatIndicator-field', dest='RepeatIndicatorField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--UserID-field', dest='UserIDField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--SOG-field', dest='SOGField',default=Decimal('102.3'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--PositionAccuracy-field', dest='PositionAccuracyField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--longitude-field', dest='longitudeField',default=Decimal('181'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--latitude-field', dest='latitudeField',default=Decimal('91'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--COG-field', dest='COGField',default=Decimal('360'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--TrueHeading-field', dest='TrueHeadingField',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--TimeStamp-field', dest='TimeStampField',default=60,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--name-field', dest='nameField',default='@@@@@@@@@@@@@@@@@@@@',metavar='aisstr6',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--shipandcargo-field', dest='shipandcargoField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--dimA-field', dest='dimAField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--dimB-field', dest='dimBField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--dimC-field', dest='dimCField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--dimD-field', dest='dimDField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--fixtype-field', dest='fixtypeField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--RAIM-field', dest='RAIMField',metavar='bool',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--DTE-field', dest='DTEField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')

def main():
	from optparse import OptionParser
	parser = OptionParser(usage="%prog [options]",
		version="%prog "+__version__)

	parser.add_option('--doc-test',dest='doctest',default=False,action='store_true',
		help='run the documentation tests')
	parser.add_option('--unit-test',dest='unittest',default=False,action='store_true',
		help='run the unit tests')
	parser.add_option('-v','--verbose',dest='verbose',default=False,action='store_true',
		help='Make the test output verbose')

	# FIX: remove nmea from binary messages.  No way to build the whole packet?
	# FIX: or build the surrounding msg 8 for a broadcast?
	typeChoices = ('binary','nmeapayload','nmea') # FIX: what about a USCG type message?
	parser.add_option('-t','--type',choices=typeChoices,type='choice',dest='ioType'
		,default='nmeapayload'
		,help='What kind of string to write for encoding ('+', '.join(typeChoices)+') [default: %default]')


	outputChoices = ('std','html','csv','sql' , 'kml','kml-full')
	parser.add_option('-T','--output-type',choices=outputChoices,type='choice',dest='outputType'
		,default='std'
		,help='What kind of string to output ('+', '.join(outputChoices)+') [default: %default]')

	parser.add_option('-o','--output',dest='outputFileName',default=None,
			  help='Name of the python file to write [default: stdout]')

	parser.add_option('-f','--fields',dest='fieldList',default=None, action='append',
			  choices=fieldList,
			  help='Which fields to include in the output.  Currently only for csv output [default: all]')

	parser.add_option('-p','--print-csv-field-list',dest='printCsvfieldList',default=False,action='store_true',
			  help='Print the field name for csv')

	parser.add_option('-c','--sql-create',dest='sqlCreate',default=False,action='store_true',
			  help='Print out an sql create command for the table.')

	parser.add_option('--latex-table',dest='latexDefinitionTable',default=False,action='store_true',
			  help='Print a LaTeX table of the type')

	parser.add_option('--text-table',dest='textDefinitionTable',default=False,action='store_true',
			  help='Print delimited table of the type (for Word table importing)')
	parser.add_option('--delimt-text-table',dest='delimTextDefinitionTable',default='\t'
			  ,help='Delimiter for text table [default: \'%default\'](for Word table importing)')


	dbChoices = ('sqlite','postgres')
	parser.add_option('-D','--db-type',dest='dbType',default='postgres'
			  ,choices=dbChoices,type='choice'
			  ,help='What kind of database ('+', '.join(dbChoices)+') [default: %default]')

	addMsgOptions(parser)

	(options,args) = parser.parse_args()
	success=True

	if options.doctest:
		import os; print os.path.basename(sys.argv[0]), 'doctests ...',
		sys.argv= [sys.argv[0]]
		if options.verbose: sys.argv.append('-v')
		import doctest
		numfail,numtests=doctest.testmod()
		if numfail==0: print 'ok'
		else: 
			print 'FAILED'
			success=False

	if not success: sys.exit('Something Failed')
	del success # Hide success from epydoc

	if options.unittest:
		sys.argv = [sys.argv[0]]
		if options.verbose: sys.argv.append('-v')
		unittest.main()

	outfile = sys.stdout
	if None!=options.outputFileName:
		outfile = file(options.outputFileName,'w')


	if options.doEncode:
		# First make sure all non required options are specified
		if None==options.RepeatIndicatorField: parser.error("missing value for RepeatIndicatorField")
		if None==options.UserIDField: parser.error("missing value for UserIDField")
		if None==options.SOGField: parser.error("missing value for SOGField")
		if None==options.PositionAccuracyField: parser.error("missing value for PositionAccuracyField")
		if None==options.longitudeField: parser.error("missing value for longitudeField")
		if None==options.latitudeField: parser.error("missing value for latitudeField")
		if None==options.COGField: parser.error("missing value for COGField")
		if None==options.TrueHeadingField: parser.error("missing value for TrueHeadingField")
		if None==options.TimeStampField: parser.error("missing value for TimeStampField")
		if None==options.nameField: parser.error("missing value for nameField")
		if None==options.shipandcargoField: parser.error("missing value for shipandcargoField")
		if None==options.dimAField: parser.error("missing value for dimAField")
		if None==options.dimBField: parser.error("missing value for dimBField")
		if None==options.dimCField: parser.error("missing value for dimCField")
		if None==options.dimDField: parser.error("missing value for dimDField")
		if None==options.fixtypeField: parser.error("missing value for fixtypeField")
		if None==options.RAIMField: parser.error("missing value for RAIMField")
		if None==options.DTEField: parser.error("missing value for DTEField")
		msgDict={
			'MessageID': '19',
			'RepeatIndicator': options.RepeatIndicatorField,
			'UserID': options.UserIDField,
			'Spare': '0',
			'SOG': options.SOGField,
			'PositionAccuracy': options.PositionAccuracyField,
			'longitude': options.longitudeField,
			'latitude': options.latitudeField,
			'COG': options.COGField,
			'TrueHeading': options.TrueHeadingField,
			'TimeStamp': options.TimeStampField,
			'Spare2': '0',
			'name': options.nameField,
			'shipandcargo': options.shipandcargoField,
			'dimA': options.dimAField,
			'dimB': options.dimBField,
			'dimC': options.dimCField,
			'dimD': options.dimDField,
			'fixtype': options.fixtypeField,
			'RAIM': options.RAIMField,
			'DTE': options.DTEField,
			'Spare3': '0',
		}

		bits = encode(msgDict)
		if 'binary'==options.ioType: print str(bits)
		elif 'nmeapayload'==options.ioType:
		    # FIX: figure out if this might be necessary at compile time
		    #print "bitLen",len(bits)
		    bitLen=len(bits)
		    if bitLen%6!=0:
			bits = bits + BitVector(size=(6 - (bitLen%6)))  # Pad out to multiple of 6
		    #print "result:",binary.bitvectoais6(bits)[0]
		    print binary.bitvectoais6(bits)[0]


		# FIX: Do not emit this option for the binary message payloads.  Does not make sense.
		elif 'nmea'==options.ioType: 
		    #bitLen=len(bits)
                    #if bitLen%6!=0:
		    #	bits = bits + BitVector(size=(6 - (bitLen%6)))  # Pad out to multiple of 6
                    import aisutils.uscg as uscg
                    nmea = uscg.create_nmea(bits)
                    print nmea
                    #
                    #


                    #sys.exit("FIX: need to implement creating nmea capability")
		else: sys.exit('ERROR: unknown ioType.  Help!')


	if options.sqlCreate:
		sqlCreateStr(outfile,options.fieldList,dbType=options.dbType)

	if options.latexDefinitionTable:
		latexDefinitionTable(outfile)

	# For conversion to word tables
	if options.textDefinitionTable:
		textDefinitionTable(outfile,options.delimTextDefinitionTable)

	if options.printCsvfieldList:
		# Make a csv separated list of fields that will be displayed for csv
		if None == options.fieldList: options.fieldList = fieldList
		import StringIO
		buf = StringIO.StringIO()
		for field in options.fieldList:
			buf.write(field+',')
		result = buf.getvalue()
		if result[-1] == ',': print result[:-1]
		else: print result

	if options.doDecode:
		if len(args)==0: args = sys.stdin
		for msg in args:
			bv = None

			if msg[0] in ('$','!') and msg[3:6] in ('VDM','VDO'):
				# Found nmea
				# FIX: do checksum
				bv = binary.ais6tobitvec(msg.split(',')[5])
			else: # either binary or nmeapayload... expect mostly nmeapayloads
				# assumes that an all 0 and 1 string can not be a nmeapayload
				binaryMsg=True
				for c in msg:
					if c not in ('0','1'):
						binaryMsg=False
						break
				if binaryMsg:
					bv = BitVector(bitstring=msg)
				else: # nmeapayload
					bv = binary.ais6tobitvec(msg)

			printFields(decode(bv)
				    ,out=outfile
				    ,format=options.outputType
				    ,fieldList=options.fieldList
				    ,dbType=options.dbType
				    )

############################################################
if __name__=='__main__':
    main()
