#!/usr/bin/env python
__author__    = 'Kurt Schwehr'
__version__   = '$Revision: 4799 $'.split()[1]
__revision__  = __version__ # For pylint
__date__ = '$Date: 2006-09-25 11:09:02 -0400 (Mon, 25 Sep 2006) $'.split()[1]
__copyright__ = '2008'
__license__   = 'GPL v3'
__contact__   = 'kurt at ccom.unh.edu'

__doc__ ='''
Convert AIS messages from AIVDM binary to AI AIT: Automatic
Indentification system - Automatic Indentification system Text.  These
packages break out AIS messages in an easier to parse format.

Note: I think this was for work with IVS / Fledermaus

This violates the number of charcters per line rule in the NMEA
standard.  I refuse to do multi-line output.

!AIVDM,1,1,,B,15MvrUPP1DG?wfBE`;;IB?vT0<0o,0*03,x144329,b003669712,1224508882
!AIVDM,1,1,,B,181:Jf002AIGoB8?w2TUPTHV0@;Q,0*18,s23690,d-113,T00.60063558,r993669942,1224508880

@requires: U{Python<http://python.org/>} >= 2.5
@requires: U{epydoc<http://epydoc.sourceforge.net/>} >= 3.0.1

@undocumented: __doc__
@since: 2008-Oct-20
@status: under development
@organization: U{CCOM<http://ccom.unh.edu/>}

@todo: Eventually this code should be autogenerated from a XML specification file

'''
import nmea.checksum
import uscg
import ais
from math import *


ait_1_fields = [
    'nmea_type',
    'message_type',
    'seq_id',
    'chan',
    'message_id', # 1, 2, or 3
    'repeat_indicator',
    'user_id',
    'navigation_status',
    'ROT', # FIX: take this out when ROT is done in the main code
    'SOG',
    'position_accuracy',
    'longitude',
    'latitude',
    'cog',
    'True_Heading',
    'time_stamp',
    'special_maneuver_indicator', # FIX: special manoevre indicator
    'spare',
    'raim',
    'sync_state',
    'slot_timeout',
    'receive_stations',
    'slot_num',
    'utc_hr',
    'utc_min',
    'slot_offset',
    'station',
    'unix_time_stamp'
    ]

def msg_1_to_ait(nmea_str):
    match_obj = uscg.uscg_ais_nmea_regex.search(nmea_str.strip())
    if match_obj is None:
        # throw exception
        print 'no match!',nmea_str
        return None
    grp = match_obj.group
    r = '$AIAIT,pos_a,%s,%s,' % (grp('seqId'),grp('chan'))
    bv = ais.binary.ais6tobitvec(grp('body'))
    body = ais.ais_msg_1.decode(bv)

    #ais.ais_msg_1.printFields(body)

    #print 'ROT:',body['ROT'],'->',pow(body['ROT']/4.733,2)

    body_list = [
        str(body['MessageID']),
        str(body['RepeatIndicator']),
	str(body['UserID']),
	str(body['NavigationStatus']),
	str(pow(body['ROT']/4.733,2)), # FIX: take this out when ROT is done in the main code
	str(body['SOG']),
	str(body['PositionAccuracy']),
	str(float(body['longitude'])),
	str(float(body['latitude'])),
	str(float(body['COG'])),
	str(body['TrueHeading']),
	str(body['TimeStamp']),
	str(body['RegionalReserved']), # FIX: special manoevre indicator
	str(body['Spare']),
	str(body['RAIM']),
	str(body['state_syncstate']),
	str(body['state_slottimeout']),
	#str(body['state_slotoffset'])
        # 3.3.7.2.3 sub messages
        '', # receive_stations
        '', # slot_num
        '', # utc_hr
        '', # utc_min
        '', # slot_offset
       ]
    r += ','.join(body_list)
    r += ',' + grp('station')
    r += ',' + grp('timeStamp')
    checksum = nmea.checksum.checksumStr(r)
    #print 'checksum',checksum
    return r+'*'+checksum

def ait_1_to_dict(ait_str):
    d = {}
    fields = ait_str.split(',')
    for i in range(len(ait_1_fields)):
        #print i,ait_1_fields[i],fields[i]
        d[ait_1_fields[i]] = fields[i]

    d['unix_time_stamp'] = fields[-1].split('*')[0] # Drop checksum
    return d


def test():
    # Undef ROT: print msg_1_to_ait('!AIVDM,1,1,,B,15MvrUPP1DG?wfBE`;;IB?vT0<0o,0*03,x144329,b003669712,1224508882')
    # ROT 127: print msg_1_to_ait('!AIVDM,1,1,,B,15PlLL0Oiern4dfGVUlnimGF2<09,0*00,x49862,r003669929,1224516044')
    # ROT -128: print msg_1_to_ait('!AIVDM,1,1,,A,15MvdBPP00G<knjL7iV75gwD08Hv,0*0F,x371073,b003669704,1224516042')
    # ROT: 0 print msg_1_to_ait('!AIVDM,1,1,,A,35Q3t`5000G<a<PL=UHPgRcD0000,0*0B,x371074,b003669704,1224516042')
    lines='''!AIVDM,1,1,,B,15N4Mj0P00IHlBV@5Emh0?wl05q`,0*50,s24093,d-112,T00.81986647,r993669942,1224516421
!AIVDM,1,1,,A,15M6ad0000G?j>FK@9Ds2mV22<1f,0*51,x261293,b003669707,1224516421
!AIVDM,1,1,,B,14eGG10P01G<MOJL=opVuwwl0<1l,0*56,x184456,b003669976,1224516421
!AIVDM,1,1,,A,3819<vEP01JqPWtGgKtW>gv2P000,0*06,x112467,s24969,d-116,T59.30972529,r003669959,1224516423
!AIVDM,1,1,,B,15MvHmPP1FG?>PlKepidEgv200RJ,0*1F,x252824,b003669701,1224516421
!AIVDM,1,1,,A,14eGpd@000G<MR4L=o1LUBv40D2:,0*7F,x539741,b003669705,1224516421
!AIVDM,1,1,,A,153AE<0002o:rmbKRa4;GPl40H0q,0*02,x184461,b003669976,1224516422
!AIVDM,1,1,,A,15OuPl0P00I?0Dp@0=`00?wl00Rt,0*19,s30596,d-093,T00.99011319,r993669945,1224516421
!AIVDM,1,1,,A,85PH5sQKf43Q4uCKOM2fD0Rs9e>GN4`oHGcmreNI3oc<K0AW8u`5,0*22,x517704,b003669984,1224516377
'''
    lines='''!AIVDM,1,1,,B,15Mwq1WP01rB2crBh5G:6?v200Rj,0*59,s28057,d-095,T49.46179499,x91028,rRDSULI1,1224516422
!AIVDM,1,1,,B,1000000P01Jt;pDHaP>78gvt0<02,0*57,rnhcml,1213228823.96
'''

    import StringIO
    str_file = StringIO.StringIO(lines)
    for line in str_file.readlines():
        print 'LINE:',line
        aiait_str = msg_1_to_ait(line)
        print 'aiait_str:   ',aiait_str
        d = ait_1_to_dict(aiait_str)
        for key in d:
            print key,'\t',d[key]
