#!/usr/bin/env python

__version__ = '$Revision: 4791 $'.split()[1]
__date__ = '$Date: 2012-10-07 $'.split()[1]
__author__ = 'xmlbinmsg'

__doc__='''

Autogenerated python functions to serialize/deserialize binary messages.

Generated by: ../scripts/aisxmlbinmsg2py.py

Need to then wrap these functions with the outer AIS packet and then
convert the whole binary blob to a NMEA string.  Those functions are
not currently provided in this file.

serialize: python to ais binary
deserialize: ais binary to python

The generated code uses translators.py, binary.py, and aisstring.py
which should be packaged with the resulting files.


@requires: U{epydoc<http://epydoc.sourceforge.net/>} > 3.0alpha3
@requires: U{BitVector<http://cheeseshop.python.org/pypi/BitVector>}

@author: '''+__author__+'''
@version: ''' + __version__ +'''
@var __date__: Date of last svn commit
@undocumented: __version__ __author__ __doc__ parser
@status: under development
@license: Generated code has no license
@todo: FIX: put in a description of the message here with fields and types.
'''

import sys
from decimal import Decimal
from BitVector import BitVector

import binary, aisstring

# FIX: check to see if these will be needed
TrueBV  = BitVector(bitstring="1")
"Why always rebuild the True bit?  This should speed things up a bunch"
FalseBV = BitVector(bitstring="0")
"Why always rebuild the False bit?  This should speed things up a bunch"


fieldList = (
	'MessageID',
	'RepeatIndicator',
	'UserID',
	'Spare',
	'dac',
	'fid',
	'latitude',
	'longitude',
	'day',
	'hour',
	'min',
	'avewind',
	'windgust',
	'winddir',
	'windgustdir',
	'airtemp',
	'relhumid',
	'dewpoint',
	'airpressure',
	'airpressuretrend',
	'horizvis',
	'waterlevel',
	'waterleveltrend',
	'surfcurspeed',
	'surfcurdir',
	'curspeed2',
	'curdir2',
	'curlevel2',
	'curspeed3',
	'curdir3',
	'curlevel3',
	'sigwaveheight',
	'waveperiod',
	'wavedir',
	'swellheight',
	'swellperiod',
	'swelldir',
	'seastate',
	'watertemp',
	'preciptype',
	'salinity',
	'ice',
	'Spare2',
)

fieldListPostgres = (
	'MessageID',
	'RepeatIndicator',
	'UserID',
	'Spare',
	'dac',
	'fid',
	'Position',	# PostGIS data type
	'day',
	'hour',
	'min',
	'avewind',
	'windgust',
	'winddir',
	'windgustdir',
	'airtemp',
	'relhumid',
	'dewpoint',
	'airpressure',
	'airpressuretrend',
	'horizvis',
	'waterlevel',
	'waterleveltrend',
	'surfcurspeed',
	'surfcurdir',
	'curspeed2',
	'curdir2',
	'curlevel2',
	'curspeed3',
	'curdir3',
	'curlevel3',
	'sigwaveheight',
	'waveperiod',
	'wavedir',
	'swellheight',
	'swellperiod',
	'swelldir',
	'seastate',
	'watertemp',
	'preciptype',
	'salinity',
	'ice',
	'Spare2',
)

toPgFields = {
	'latitude':'Position',
	'longitude':'Position',
}
'''
Go to the Postgis field names from the straight field name
'''

fromPgFields = {
	'Position':('latitude','longitude',),
}
'''
Go from the Postgis field names to the straight field name
'''

pgTypes = {
	'Position':'POINT',
}
'''
Lookup table for each postgis field name to get its type.
'''

def encode(params, validate=False):
	'''Create a imo_met_hydro binary message payload to pack into an AIS Msg imo_met_hydro.

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 8 (field automatically set to "8")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): MMSI number of transmitter broadcasting the message
	  - Spare(uint): Reserved for definition by a regional authority. (field automatically set to "0")
	  - dac(uint): Designated Area Code - part 1 of the IAI (field automatically set to "1")
	  - fid(uint): Functional Identifier - part 2 of the IAI (field automatically set to "11")
	  - latitude(decimal): Location of the vessel.  North South location
	  - longitude(decimal): Location of the vessel.  East West location
	  - day(uint): Day 0..31
	  - hour(uint): Hour 0..23
	  - min(uint): Min
	  - avewind(uint): Average wind speed values for the last 10 minutes.
	  - windgust(uint): Wind gust is the max wind speed value reading  during the last 10 minutes.
	  - winddir(uint): Wind direction
	  - windgustdir(uint): Wind direction for the gust.
	  - airtemp(decimal): Dry bulb temperature
	  - relhumid(uint): Relative humidity
	  - dewpoint(decimal): Dew Point
	  - airpressure(udecimal): Air pressure
	  - airpressuretrend(uint): Air pressure trend
	  - horizvis(udecimal): Horizontal visibility
	  - waterlevel(decimal): Water level (incl. tide)
	  - waterleveltrend(uint): Water level trend
	  - surfcurspeed(udecimal): Surface current speed
	  - surfcurdir(uint): Surface current direction
	  - curspeed2(udecimal): Level 2 current speed
	  - curdir2(uint): Level 2 current direction
	  - curlevel2(uint): Measuring level below sea surface for level 2
	  - curspeed3(udecimal): Level 3 current speed
	  - curdir3(uint): Level 3 current direction
	  - curlevel3(uint): Measuring level below sea surface for level 3
	  - sigwaveheight(udecimal): Significant wave height
	  - waveperiod(uint): Wave period
	  - wavedir(uint): Wave direction
	  - swellheight(udecimal): Swell height
	  - swellperiod(uint): Swell period
	  - swelldir(uint): Swell direction
	  - seastate(uint): Sea state according to the Beaufort scale
	  - watertemp(udecimal): Water temperature
	  - preciptype(uint): According to WMO
	  - salinity(decimal): Salinity
	  - ice(uint): Yes or no for the presence of ice
	  - Spare2(uint): Must be zero (field automatically set to "0")
	@param params: Dictionary of field names/values.  Throws a ValueError exception if required is missing
	@param validate: Set to true to cause checking to occur.  Runs slower.  FIX: not implemented.
	@rtype: BitVector
	@return: encoded binary message (for binary messages, this needs to be wrapped in a msg 8
	@note: The returned bits may not be 6 bit aligned.  It is up to you to pad out the bits.
	'''

	bvList = []
	bvList.append(binary.setBitVectorSize(BitVector(intVal=8),6))
	if 'RepeatIndicator' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['RepeatIndicator']),2))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),2))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['UserID']),30))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=0),2))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=1),10))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=11),6))
	if 'latitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['latitude'])*Decimal('60000')),24))
	else:
		bvList.append(binary.bvFromSignedInt(5460000,24))
	if 'longitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['longitude'])*Decimal('60000')),25))
	else:
		bvList.append(binary.bvFromSignedInt(10860000,25))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['day']),5))
	if 'hour' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['hour']),5))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=31),5))
	if 'min' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['min']),6))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=63),6))
	if 'avewind' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['avewind']),7))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=127),7))
	if 'windgust' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['windgust']),7))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=127),7))
	if 'winddir' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['winddir']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'windgustdir' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['windgustdir']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'airtemp' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['airtemp'])*Decimal('10')),11))
	else:
		bvList.append(binary.bvFromSignedInt(1023,11))
	if 'relhumid' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['relhumid']),7))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=127),7))
	if 'dewpoint' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['dewpoint'])*Decimal('10')),10))
	else:
		bvList.append(binary.bvFromSignedInt(511,10))
	if 'airpressure' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['airpressure']-(800))*Decimal('1')))),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(1311)),9))
	if 'airpressuretrend' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['airpressuretrend']),2))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=3),2))
	if 'horizvis' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['horizvis'])*Decimal('10')))),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(255)),8))
	bvList.append(binary.bvFromSignedInt(int(Decimal(params['waterlevel'])*Decimal('10')),9))
	if 'waterleveltrend' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['waterleveltrend']),2))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=3),2))
	if 'surfcurspeed' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['surfcurspeed'])*Decimal('10')))),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(255)),8))
	if 'surfcurdir' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['surfcurdir']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'curspeed2' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['curspeed2'])*Decimal('10')))),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(255)),8))
	if 'curdir2' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['curdir2']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'curlevel2' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['curlevel2']),5))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=31),5))
	if 'curspeed3' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['curspeed3'])*Decimal('10')))),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(255)),8))
	if 'curdir3' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['curdir3']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'curlevel3' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['curlevel3']),5))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=31),5))
	if 'sigwaveheight' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['sigwaveheight'])*Decimal('10')))),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(255)),8))
	if 'waveperiod' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['waveperiod']),6))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=63),6))
	if 'wavedir' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['wavedir']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'swellheight' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['swellheight'])*Decimal('10')))),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(255)),8))
	if 'swellperiod' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['swellperiod']),6))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=63),6))
	if 'swelldir' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['swelldir']),9))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=511),9))
	if 'seastate' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['seastate']),4))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=15),4))
	if 'watertemp' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int((Decimal(params['watertemp']-(-10))*Decimal('10')))),10))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=int(923)),10))
	if 'preciptype' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['preciptype']),3))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=7),3))
	if 'salinity' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['salinity'])*Decimal('10')),9))
	else:
		bvList.append(binary.bvFromSignedInt(923,9))
	if 'ice' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['ice']),2))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=3),2))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=0),6))

	return binary.joinBV(bvList)

def decode(bv, validate=False):
	'''Unpack a imo_met_hydro message 

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 8 (field automatically set to "8")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): MMSI number of transmitter broadcasting the message
	  - Spare(uint): Reserved for definition by a regional authority. (field automatically set to "0")
	  - dac(uint): Designated Area Code - part 1 of the IAI (field automatically set to "1")
	  - fid(uint): Functional Identifier - part 2 of the IAI (field automatically set to "11")
	  - latitude(decimal): Location of the vessel.  North South location
	  - longitude(decimal): Location of the vessel.  East West location
	  - day(uint): Day 0..31
	  - hour(uint): Hour 0..23
	  - min(uint): Min
	  - avewind(uint): Average wind speed values for the last 10 minutes.
	  - windgust(uint): Wind gust is the max wind speed value reading  during the last 10 minutes.
	  - winddir(uint): Wind direction
	  - windgustdir(uint): Wind direction for the gust.
	  - airtemp(decimal): Dry bulb temperature
	  - relhumid(uint): Relative humidity
	  - dewpoint(decimal): Dew Point
	  - airpressure(udecimal): Air pressure
	  - airpressuretrend(uint): Air pressure trend
	  - horizvis(udecimal): Horizontal visibility
	  - waterlevel(decimal): Water level (incl. tide)
	  - waterleveltrend(uint): Water level trend
	  - surfcurspeed(udecimal): Surface current speed
	  - surfcurdir(uint): Surface current direction
	  - curspeed2(udecimal): Level 2 current speed
	  - curdir2(uint): Level 2 current direction
	  - curlevel2(uint): Measuring level below sea surface for level 2
	  - curspeed3(udecimal): Level 3 current speed
	  - curdir3(uint): Level 3 current direction
	  - curlevel3(uint): Measuring level below sea surface for level 3
	  - sigwaveheight(udecimal): Significant wave height
	  - waveperiod(uint): Wave period
	  - wavedir(uint): Wave direction
	  - swellheight(udecimal): Swell height
	  - swellperiod(uint): Swell period
	  - swelldir(uint): Swell direction
	  - seastate(uint): Sea state according to the Beaufort scale
	  - watertemp(udecimal): Water temperature
	  - preciptype(uint): According to WMO
	  - salinity(decimal): Salinity
	  - ice(uint): Yes or no for the presence of ice
	  - Spare2(uint): Must be zero (field automatically set to "0")
	@type bv: BitVector
	@param bv: Bits defining a message
	@param validate: Set to true to cause checking to occur.  Runs slower.  FIX: not implemented.
	@rtype: dict
	@return: params
	'''

	#Would be nice to check the bit count here..
	#if validate:
	#	assert (len(bv)==FIX: SOME NUMBER)
	r = {}
	r['MessageID']=8
	r['RepeatIndicator']=int(bv[6:8])
	r['UserID']=int(bv[8:38])
	r['Spare']=0
	r['dac']=1
	r['fid']=11
	r['latitude']=Decimal(binary.signedIntFromBV(bv[56:80]))/Decimal('60000')
	r['longitude']=Decimal(binary.signedIntFromBV(bv[80:105]))/Decimal('60000')
	r['day']=int(bv[105:110])
	r['hour']=int(bv[110:115])
	r['min']=int(bv[115:121])
	r['avewind']=int(bv[121:128])
	r['windgust']=int(bv[128:135])
	r['winddir']=int(bv[135:144])
	r['windgustdir']=int(bv[144:153])
	r['airtemp']=Decimal(binary.signedIntFromBV(bv[153:164]))/Decimal('10')
	r['relhumid']=int(bv[164:171])
	r['dewpoint']=Decimal(binary.signedIntFromBV(bv[171:181]))/Decimal('10')
	r['airpressure']=Decimal(int(bv[181:190]))/Decimal('1')+Decimal('800')
	r['airpressuretrend']=int(bv[190:192])
	r['horizvis']=Decimal(int(bv[192:200]))/Decimal('10')
	r['waterlevel']=Decimal(binary.signedIntFromBV(bv[200:209]))/Decimal('10')
	r['waterleveltrend']=int(bv[209:211])
	r['surfcurspeed']=Decimal(int(bv[211:219]))/Decimal('10')
	r['surfcurdir']=int(bv[219:228])
	r['curspeed2']=Decimal(int(bv[228:236]))/Decimal('10')
	r['curdir2']=int(bv[236:245])
	r['curlevel2']=int(bv[245:250])
	r['curspeed3']=Decimal(int(bv[250:258]))/Decimal('10')
	r['curdir3']=int(bv[258:267])
	r['curlevel3']=int(bv[267:272])
	r['sigwaveheight']=Decimal(int(bv[272:280]))/Decimal('10')
	r['waveperiod']=int(bv[280:286])
	r['wavedir']=int(bv[286:295])
	r['swellheight']=Decimal(int(bv[295:303]))/Decimal('10')
	r['swellperiod']=int(bv[303:309])
	r['swelldir']=int(bv[309:318])
	r['seastate']=int(bv[318:322])
	r['watertemp']=Decimal(int(bv[322:332]))/Decimal('10')+Decimal('-10')
	r['preciptype']=int(bv[332:335])
	r['salinity']=Decimal(binary.signedIntFromBV(bv[335:344]))/Decimal('10')
	r['ice']=int(bv[344:346])
	r['Spare2']=0
	return r

def decodeMessageID(bv, validate=False):
	return 8

def decodeRepeatIndicator(bv, validate=False):
	return int(bv[6:8])

def decodeUserID(bv, validate=False):
	return int(bv[8:38])

def decodeSpare(bv, validate=False):
	return 0

def decodedac(bv, validate=False):
	return 1

def decodefid(bv, validate=False):
	return 11

def decodelatitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[56:80]))/Decimal('60000')

def decodelongitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[80:105]))/Decimal('60000')

def decodeday(bv, validate=False):
	return int(bv[105:110])

def decodehour(bv, validate=False):
	return int(bv[110:115])

def decodemin(bv, validate=False):
	return int(bv[115:121])

def decodeavewind(bv, validate=False):
	return int(bv[121:128])

def decodewindgust(bv, validate=False):
	return int(bv[128:135])

def decodewinddir(bv, validate=False):
	return int(bv[135:144])

def decodewindgustdir(bv, validate=False):
	return int(bv[144:153])

def decodeairtemp(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[153:164]))/Decimal('10')

def decoderelhumid(bv, validate=False):
	return int(bv[164:171])

def decodedewpoint(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[171:181]))/Decimal('10')

def decodeairpressure(bv, validate=False):
	return Decimal(int(bv[181:190]))/Decimal('1')+Decimal('800')

def decodeairpressuretrend(bv, validate=False):
	return int(bv[190:192])

def decodehorizvis(bv, validate=False):
	return Decimal(int(bv[192:200]))/Decimal('10')

def decodewaterlevel(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[200:209]))/Decimal('10')

def decodewaterleveltrend(bv, validate=False):
	return int(bv[209:211])

def decodesurfcurspeed(bv, validate=False):
	return Decimal(int(bv[211:219]))/Decimal('10')

def decodesurfcurdir(bv, validate=False):
	return int(bv[219:228])

def decodecurspeed2(bv, validate=False):
	return Decimal(int(bv[228:236]))/Decimal('10')

def decodecurdir2(bv, validate=False):
	return int(bv[236:245])

def decodecurlevel2(bv, validate=False):
	return int(bv[245:250])

def decodecurspeed3(bv, validate=False):
	return Decimal(int(bv[250:258]))/Decimal('10')

def decodecurdir3(bv, validate=False):
	return int(bv[258:267])

def decodecurlevel3(bv, validate=False):
	return int(bv[267:272])

def decodesigwaveheight(bv, validate=False):
	return Decimal(int(bv[272:280]))/Decimal('10')

def decodewaveperiod(bv, validate=False):
	return int(bv[280:286])

def decodewavedir(bv, validate=False):
	return int(bv[286:295])

def decodeswellheight(bv, validate=False):
	return Decimal(int(bv[295:303]))/Decimal('10')

def decodeswellperiod(bv, validate=False):
	return int(bv[303:309])

def decodeswelldir(bv, validate=False):
	return int(bv[309:318])

def decodeseastate(bv, validate=False):
	return int(bv[318:322])

def decodewatertemp(bv, validate=False):
	return Decimal(int(bv[322:332]))/Decimal('10')+Decimal('-10')

def decodepreciptype(bv, validate=False):
	return int(bv[332:335])

def decodesalinity(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[335:344]))/Decimal('10')

def decodeice(bv, validate=False):
	return int(bv[344:346])

def decodeSpare2(bv, validate=False):
	return 0


def printHtml(params, out=sys.stdout):
		out.write("<h3>imo_met_hydro</h3>\n")
		out.write("<table border=\"1\">\n")
		out.write("<tr bgcolor=\"orange\">\n")
		out.write("<th align=\"left\">Field Name</th>\n")
		out.write("<th align=\"left\">Type</th>\n")
		out.write("<th align=\"left\">Value</th>\n")
		out.write("<th align=\"left\">Value in Lookup Table</th>\n")
		out.write("<th align=\"left\">Units</th>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>MessageID</td>\n")
		out.write("<td>uint</td>\n")
		if 'MessageID' in params:
			out.write("	<td>"+str(params['MessageID'])+"</td>\n")
			out.write("	<td>"+str(params['MessageID'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>RepeatIndicator</td>\n")
		out.write("<td>uint</td>\n")
		if 'RepeatIndicator' in params:
			out.write("	<td>"+str(params['RepeatIndicator'])+"</td>\n")
			if str(params['RepeatIndicator']) in RepeatIndicatorDecodeLut:
				out.write("<td>"+RepeatIndicatorDecodeLut[str(params['RepeatIndicator'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>UserID</td>\n")
		out.write("<td>uint</td>\n")
		if 'UserID' in params:
			out.write("	<td>"+str(params['UserID'])+"</td>\n")
			out.write("	<td>"+str(params['UserID'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>Spare</td>\n")
		out.write("<td>uint</td>\n")
		if 'Spare' in params:
			out.write("	<td>"+str(params['Spare'])+"</td>\n")
			out.write("	<td>"+str(params['Spare'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>dac</td>\n")
		out.write("<td>uint</td>\n")
		if 'dac' in params:
			out.write("	<td>"+str(params['dac'])+"</td>\n")
			out.write("	<td>"+str(params['dac'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>fid</td>\n")
		out.write("<td>uint</td>\n")
		if 'fid' in params:
			out.write("	<td>"+str(params['fid'])+"</td>\n")
			out.write("	<td>"+str(params['fid'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>latitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'latitude' in params:
			out.write("	<td>"+str(params['latitude'])+"</td>\n")
			out.write("	<td>"+str(params['latitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>longitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'longitude' in params:
			out.write("	<td>"+str(params['longitude'])+"</td>\n")
			out.write("	<td>"+str(params['longitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>day</td>\n")
		out.write("<td>uint</td>\n")
		if 'day' in params:
			out.write("	<td>"+str(params['day'])+"</td>\n")
			out.write("	<td>"+str(params['day'])+"</td>\n")
		out.write("<td>days</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>hour</td>\n")
		out.write("<td>uint</td>\n")
		if 'hour' in params:
			out.write("	<td>"+str(params['hour'])+"</td>\n")
			out.write("	<td>"+str(params['hour'])+"</td>\n")
		out.write("<td>hours</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>min</td>\n")
		out.write("<td>uint</td>\n")
		if 'min' in params:
			out.write("	<td>"+str(params['min'])+"</td>\n")
			out.write("	<td>"+str(params['min'])+"</td>\n")
		out.write("<td>minutes</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>avewind</td>\n")
		out.write("<td>uint</td>\n")
		if 'avewind' in params:
			out.write("	<td>"+str(params['avewind'])+"</td>\n")
			out.write("	<td>"+str(params['avewind'])+"</td>\n")
		out.write("<td>knots</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>windgust</td>\n")
		out.write("<td>uint</td>\n")
		if 'windgust' in params:
			out.write("	<td>"+str(params['windgust'])+"</td>\n")
			out.write("	<td>"+str(params['windgust'])+"</td>\n")
		out.write("<td>knots</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>winddir</td>\n")
		out.write("<td>uint</td>\n")
		if 'winddir' in params:
			out.write("	<td>"+str(params['winddir'])+"</td>\n")
			out.write("	<td>"+str(params['winddir'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>windgustdir</td>\n")
		out.write("<td>uint</td>\n")
		if 'windgustdir' in params:
			out.write("	<td>"+str(params['windgustdir'])+"</td>\n")
			out.write("	<td>"+str(params['windgustdir'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>airtemp</td>\n")
		out.write("<td>decimal</td>\n")
		if 'airtemp' in params:
			out.write("	<td>"+str(params['airtemp'])+"</td>\n")
			out.write("	<td>"+str(params['airtemp'])+"</td>\n")
		out.write("<td>degrees Celsius</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>relhumid</td>\n")
		out.write("<td>uint</td>\n")
		if 'relhumid' in params:
			out.write("	<td>"+str(params['relhumid'])+"</td>\n")
			out.write("	<td>"+str(params['relhumid'])+"</td>\n")
		out.write("<td>percent</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>dewpoint</td>\n")
		out.write("<td>decimal</td>\n")
		if 'dewpoint' in params:
			out.write("	<td>"+str(params['dewpoint'])+"</td>\n")
			out.write("	<td>"+str(params['dewpoint'])+"</td>\n")
		out.write("<td>degrees Celsius</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>airpressure</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'airpressure' in params:
			out.write("	<td>"+str(params['airpressure'])+"</td>\n")
			out.write("	<td>"+str(params['airpressure'])+"</td>\n")
		out.write("<td>hPa</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>airpressuretrend</td>\n")
		out.write("<td>uint</td>\n")
		if 'airpressuretrend' in params:
			out.write("	<td>"+str(params['airpressuretrend'])+"</td>\n")
			if str(params['airpressuretrend']) in airpressuretrendDecodeLut:
				out.write("<td>"+airpressuretrendDecodeLut[str(params['airpressuretrend'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>horizvis</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'horizvis' in params:
			out.write("	<td>"+str(params['horizvis'])+"</td>\n")
			out.write("	<td>"+str(params['horizvis'])+"</td>\n")
		out.write("<td>nm</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>waterlevel</td>\n")
		out.write("<td>decimal</td>\n")
		if 'waterlevel' in params:
			out.write("	<td>"+str(params['waterlevel'])+"</td>\n")
			out.write("	<td>"+str(params['waterlevel'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>waterleveltrend</td>\n")
		out.write("<td>uint</td>\n")
		if 'waterleveltrend' in params:
			out.write("	<td>"+str(params['waterleveltrend'])+"</td>\n")
			if str(params['waterleveltrend']) in waterleveltrendDecodeLut:
				out.write("<td>"+waterleveltrendDecodeLut[str(params['waterleveltrend'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>surfcurspeed</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'surfcurspeed' in params:
			out.write("	<td>"+str(params['surfcurspeed'])+"</td>\n")
			out.write("	<td>"+str(params['surfcurspeed'])+"</td>\n")
		out.write("<td>knots</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>surfcurdir</td>\n")
		out.write("<td>uint</td>\n")
		if 'surfcurdir' in params:
			out.write("	<td>"+str(params['surfcurdir'])+"</td>\n")
			out.write("	<td>"+str(params['surfcurdir'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>curspeed2</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'curspeed2' in params:
			out.write("	<td>"+str(params['curspeed2'])+"</td>\n")
			out.write("	<td>"+str(params['curspeed2'])+"</td>\n")
		out.write("<td>knots</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>curdir2</td>\n")
		out.write("<td>uint</td>\n")
		if 'curdir2' in params:
			out.write("	<td>"+str(params['curdir2'])+"</td>\n")
			out.write("	<td>"+str(params['curdir2'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>curlevel2</td>\n")
		out.write("<td>uint</td>\n")
		if 'curlevel2' in params:
			out.write("	<td>"+str(params['curlevel2'])+"</td>\n")
			out.write("	<td>"+str(params['curlevel2'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>curspeed3</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'curspeed3' in params:
			out.write("	<td>"+str(params['curspeed3'])+"</td>\n")
			out.write("	<td>"+str(params['curspeed3'])+"</td>\n")
		out.write("<td>knots</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>curdir3</td>\n")
		out.write("<td>uint</td>\n")
		if 'curdir3' in params:
			out.write("	<td>"+str(params['curdir3'])+"</td>\n")
			out.write("	<td>"+str(params['curdir3'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>curlevel3</td>\n")
		out.write("<td>uint</td>\n")
		if 'curlevel3' in params:
			out.write("	<td>"+str(params['curlevel3'])+"</td>\n")
			out.write("	<td>"+str(params['curlevel3'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>sigwaveheight</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'sigwaveheight' in params:
			out.write("	<td>"+str(params['sigwaveheight'])+"</td>\n")
			out.write("	<td>"+str(params['sigwaveheight'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>waveperiod</td>\n")
		out.write("<td>uint</td>\n")
		if 'waveperiod' in params:
			out.write("	<td>"+str(params['waveperiod'])+"</td>\n")
			out.write("	<td>"+str(params['waveperiod'])+"</td>\n")
		out.write("<td>sec</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>wavedir</td>\n")
		out.write("<td>uint</td>\n")
		if 'wavedir' in params:
			out.write("	<td>"+str(params['wavedir'])+"</td>\n")
			out.write("	<td>"+str(params['wavedir'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>swellheight</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'swellheight' in params:
			out.write("	<td>"+str(params['swellheight'])+"</td>\n")
			out.write("	<td>"+str(params['swellheight'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>swellperiod</td>\n")
		out.write("<td>uint</td>\n")
		if 'swellperiod' in params:
			out.write("	<td>"+str(params['swellperiod'])+"</td>\n")
			out.write("	<td>"+str(params['swellperiod'])+"</td>\n")
		out.write("<td>sec</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>swelldir</td>\n")
		out.write("<td>uint</td>\n")
		if 'swelldir' in params:
			out.write("	<td>"+str(params['swelldir'])+"</td>\n")
			out.write("	<td>"+str(params['swelldir'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>seastate</td>\n")
		out.write("<td>uint</td>\n")
		if 'seastate' in params:
			out.write("	<td>"+str(params['seastate'])+"</td>\n")
			if str(params['seastate']) in seastateDecodeLut:
				out.write("<td>"+seastateDecodeLut[str(params['seastate'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>Beaufort scale</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>watertemp</td>\n")
		out.write("<td>udecimal</td>\n")
		if 'watertemp' in params:
			out.write("	<td>"+str(params['watertemp'])+"</td>\n")
			out.write("	<td>"+str(params['watertemp'])+"</td>\n")
		out.write("<td>degrees Celsius</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>preciptype</td>\n")
		out.write("<td>uint</td>\n")
		if 'preciptype' in params:
			out.write("	<td>"+str(params['preciptype'])+"</td>\n")
			if str(params['preciptype']) in preciptypeDecodeLut:
				out.write("<td>"+preciptypeDecodeLut[str(params['preciptype'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>WMO scale index</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>salinity</td>\n")
		out.write("<td>decimal</td>\n")
		if 'salinity' in params:
			out.write("	<td>"+str(params['salinity'])+"</td>\n")
			out.write("	<td>"+str(params['salinity'])+"</td>\n")
		out.write("<td>0/00</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>ice</td>\n")
		out.write("<td>uint</td>\n")
		if 'ice' in params:
			out.write("	<td>"+str(params['ice'])+"</td>\n")
			if str(params['ice']) in iceDecodeLut:
				out.write("<td>"+iceDecodeLut[str(params['ice'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>Spare2</td>\n")
		out.write("<td>uint</td>\n")
		if 'Spare2' in params:
			out.write("	<td>"+str(params['Spare2'])+"</td>\n")
			out.write("	<td>"+str(params['Spare2'])+"</td>\n")
		out.write("</tr>\n")
		out.write("</table>\n")


def printKml(params, out=sys.stdout):
	'''KML (Keyhole Markup Language) for Google Earth, but without the header/footer'''
	out.write("\	<Placemark>\n")
	out.write("\t	<name>"+str(params['UserID'])+"</name>\n")
	out.write("\t\t<description>\n")
	import StringIO
	buf = StringIO.StringIO()
	printHtml(params,buf)
	import cgi
	out.write(cgi.escape(buf.getvalue()))
	out.write("\t\t</description>\n")
	out.write("\t\t<styleUrl>#m_ylw-pushpin_copy0</styleUrl>\n")
	out.write("\t\t<Point>\n")
	out.write("\t\t\t<coordinates>")
	out.write(str(params['longitude']))
	out.write(',')
	out.write(str(params['latitude']))
	out.write(",0</coordinates>\n")
	out.write("\t\t</Point>\n")
	out.write("\t</Placemark>\n")

def printFields(params, out=sys.stdout, format='std', fieldList=None, dbType='postgres'):
	'''Print a imo_met_hydro message to stdout.

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 8 (field automatically set to "8")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): MMSI number of transmitter broadcasting the message
	  - Spare(uint): Reserved for definition by a regional authority. (field automatically set to "0")
	  - dac(uint): Designated Area Code - part 1 of the IAI (field automatically set to "1")
	  - fid(uint): Functional Identifier - part 2 of the IAI (field automatically set to "11")
	  - latitude(decimal): Location of the vessel.  North South location
	  - longitude(decimal): Location of the vessel.  East West location
	  - day(uint): Day 0..31
	  - hour(uint): Hour 0..23
	  - min(uint): Min
	  - avewind(uint): Average wind speed values for the last 10 minutes.
	  - windgust(uint): Wind gust is the max wind speed value reading  during the last 10 minutes.
	  - winddir(uint): Wind direction
	  - windgustdir(uint): Wind direction for the gust.
	  - airtemp(decimal): Dry bulb temperature
	  - relhumid(uint): Relative humidity
	  - dewpoint(decimal): Dew Point
	  - airpressure(udecimal): Air pressure
	  - airpressuretrend(uint): Air pressure trend
	  - horizvis(udecimal): Horizontal visibility
	  - waterlevel(decimal): Water level (incl. tide)
	  - waterleveltrend(uint): Water level trend
	  - surfcurspeed(udecimal): Surface current speed
	  - surfcurdir(uint): Surface current direction
	  - curspeed2(udecimal): Level 2 current speed
	  - curdir2(uint): Level 2 current direction
	  - curlevel2(uint): Measuring level below sea surface for level 2
	  - curspeed3(udecimal): Level 3 current speed
	  - curdir3(uint): Level 3 current direction
	  - curlevel3(uint): Measuring level below sea surface for level 3
	  - sigwaveheight(udecimal): Significant wave height
	  - waveperiod(uint): Wave period
	  - wavedir(uint): Wave direction
	  - swellheight(udecimal): Swell height
	  - swellperiod(uint): Swell period
	  - swelldir(uint): Swell direction
	  - seastate(uint): Sea state according to the Beaufort scale
	  - watertemp(udecimal): Water temperature
	  - preciptype(uint): According to WMO
	  - salinity(decimal): Salinity
	  - ice(uint): Yes or no for the presence of ice
	  - Spare2(uint): Must be zero (field automatically set to "0")
	@param params: Dictionary of field names/values.  
	@param out: File like object to write to
	@rtype: stdout
	@return: text to out
	'''

	if 'std'==format:
		out.write("imo_met_hydro:\n")
		if 'MessageID' in params: out.write("	MessageID:         "+str(params['MessageID'])+"\n")
		if 'RepeatIndicator' in params: out.write("	RepeatIndicator:   "+str(params['RepeatIndicator'])+"\n")
		if 'UserID' in params: out.write("	UserID:            "+str(params['UserID'])+"\n")
		if 'Spare' in params: out.write("	Spare:             "+str(params['Spare'])+"\n")
		if 'dac' in params: out.write("	dac:               "+str(params['dac'])+"\n")
		if 'fid' in params: out.write("	fid:               "+str(params['fid'])+"\n")
		if 'latitude' in params: out.write("	latitude:          "+str(params['latitude'])+"\n")
		if 'longitude' in params: out.write("	longitude:         "+str(params['longitude'])+"\n")
		if 'day' in params: out.write("	day:               "+str(params['day'])+"\n")
		if 'hour' in params: out.write("	hour:              "+str(params['hour'])+"\n")
		if 'min' in params: out.write("	min:               "+str(params['min'])+"\n")
		if 'avewind' in params: out.write("	avewind:           "+str(params['avewind'])+"\n")
		if 'windgust' in params: out.write("	windgust:          "+str(params['windgust'])+"\n")
		if 'winddir' in params: out.write("	winddir:           "+str(params['winddir'])+"\n")
		if 'windgustdir' in params: out.write("	windgustdir:       "+str(params['windgustdir'])+"\n")
		if 'airtemp' in params: out.write("	airtemp:           "+str(params['airtemp'])+"\n")
		if 'relhumid' in params: out.write("	relhumid:          "+str(params['relhumid'])+"\n")
		if 'dewpoint' in params: out.write("	dewpoint:          "+str(params['dewpoint'])+"\n")
		if 'airpressure' in params: out.write("	airpressure:       "+str(params['airpressure'])+"\n")
		if 'airpressuretrend' in params: out.write("	airpressuretrend:  "+str(params['airpressuretrend'])+"\n")
		if 'horizvis' in params: out.write("	horizvis:          "+str(params['horizvis'])+"\n")
		if 'waterlevel' in params: out.write("	waterlevel:        "+str(params['waterlevel'])+"\n")
		if 'waterleveltrend' in params: out.write("	waterleveltrend:   "+str(params['waterleveltrend'])+"\n")
		if 'surfcurspeed' in params: out.write("	surfcurspeed:      "+str(params['surfcurspeed'])+"\n")
		if 'surfcurdir' in params: out.write("	surfcurdir:        "+str(params['surfcurdir'])+"\n")
		if 'curspeed2' in params: out.write("	curspeed2:         "+str(params['curspeed2'])+"\n")
		if 'curdir2' in params: out.write("	curdir2:           "+str(params['curdir2'])+"\n")
		if 'curlevel2' in params: out.write("	curlevel2:         "+str(params['curlevel2'])+"\n")
		if 'curspeed3' in params: out.write("	curspeed3:         "+str(params['curspeed3'])+"\n")
		if 'curdir3' in params: out.write("	curdir3:           "+str(params['curdir3'])+"\n")
		if 'curlevel3' in params: out.write("	curlevel3:         "+str(params['curlevel3'])+"\n")
		if 'sigwaveheight' in params: out.write("	sigwaveheight:     "+str(params['sigwaveheight'])+"\n")
		if 'waveperiod' in params: out.write("	waveperiod:        "+str(params['waveperiod'])+"\n")
		if 'wavedir' in params: out.write("	wavedir:           "+str(params['wavedir'])+"\n")
		if 'swellheight' in params: out.write("	swellheight:       "+str(params['swellheight'])+"\n")
		if 'swellperiod' in params: out.write("	swellperiod:       "+str(params['swellperiod'])+"\n")
		if 'swelldir' in params: out.write("	swelldir:          "+str(params['swelldir'])+"\n")
		if 'seastate' in params: out.write("	seastate:          "+str(params['seastate'])+"\n")
		if 'watertemp' in params: out.write("	watertemp:         "+str(params['watertemp'])+"\n")
		if 'preciptype' in params: out.write("	preciptype:        "+str(params['preciptype'])+"\n")
		if 'salinity' in params: out.write("	salinity:          "+str(params['salinity'])+"\n")
		if 'ice' in params: out.write("	ice:               "+str(params['ice'])+"\n")
		if 'Spare2' in params: out.write("	Spare2:            "+str(params['Spare2'])+"\n")
	elif 'csv'==format:
		if None == options.fieldList:
			options.fieldList = fieldList
		needComma = False;
		for field in fieldList:
			if needComma: out.write(',')
			needComma = True
			if field in params:
				out.write(str(params[field]))
			# else: leave it empty
		out.write("\n")
	elif 'html'==format:
		printHtml(params,out)
	elif 'sql'==format:
		sqlInsertStr(params,out,dbType=dbType)
	elif 'kml'==format:
		printKml(params,out)
	elif 'kml-full'==format:
		out.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n")
		out.write("<kml xmlns=\"http://earth.google.com/kml/2.1\">\n")
		out.write("<Document>\n")
		out.write("	<name>imo_met_hydro</name>\n")
		printKml(params,out)
		out.write("</Document>\n")
		out.write("</kml>\n")
	else: 
		print "ERROR: unknown format:",format
		assert False

	return # Nothing to return

RepeatIndicatorEncodeLut = {
	'default':'0',
	'do not repeat any more':'3',
	} #RepeatIndicatorEncodeLut

RepeatIndicatorDecodeLut = {
	'0':'default',
	'3':'do not repeat any more',
	} # RepeatIndicatorEncodeLut

airpressuretrendEncodeLut = {
	'steady':'0',
	'decreasing':'1',
	'increasing':'2',
	'unavailable':'3',
	} #airpressuretrendEncodeLut

airpressuretrendDecodeLut = {
	'0':'steady',
	'1':'decreasing',
	'2':'increasing',
	'3':'unavailable',
	} # airpressuretrendEncodeLut

waterleveltrendEncodeLut = {
	'steady':'0',
	'decreasing':'1',
	'increasing':'2',
	'unavailable':'3',
	} #waterleveltrendEncodeLut

waterleveltrendDecodeLut = {
	'0':'steady',
	'1':'decreasing',
	'2':'increasing',
	'3':'unavailable',
	} # waterleveltrendEncodeLut

seastateEncodeLut = {
	'Calm':'0',
	'Light air':'1',
	'Light breeze':'2',
	'Gentle breeze':'3',
	'Moderate breeze':'4',
	'Fresh breeze':'5',
	'Strong breeze':'6',
	'Near gale':'7',
	'Gale':'8',
	'Strong gale':'9',
	'Storm':'10',
	'Violent storm':'11',
	'Hurricane':'12',
	'unavailable':'15',
	} #seastateEncodeLut

seastateDecodeLut = {
	'0':'Calm',
	'1':'Light air',
	'2':'Light breeze',
	'3':'Gentle breeze',
	'4':'Moderate breeze',
	'5':'Fresh breeze',
	'6':'Strong breeze',
	'7':'Near gale',
	'8':'Gale',
	'9':'Strong gale',
	'10':'Storm',
	'11':'Violent storm',
	'12':'Hurricane',
	'15':'unavailable',
	} # seastateEncodeLut

preciptypeEncodeLut = {
	'FIX: find the WMO list of types':'0',
	'unavailable':'7',
	} #preciptypeEncodeLut

preciptypeDecodeLut = {
	'0':'FIX: find the WMO list of types',
	'7':'unavailable',
	} # preciptypeEncodeLut

iceEncodeLut = {
	'No ice':'0',
	'Ice present':'1',
	'Reserved':'2',
	'Unknown if there is ice present':'3',
	} #iceEncodeLut

iceDecodeLut = {
	'0':'No ice',
	'1':'Ice present',
	'2':'Reserved',
	'3':'Unknown if there is ice present',
	} # iceEncodeLut

######################################################################
# SQL SUPPORT
######################################################################

dbTableName='imo_met_hydro'
'Database table name'

def sqlCreateStr(outfile=sys.stdout, fields=None, extraFields=None
		,addCoastGuardFields=True
		,dbType='postgres'
		):
	'''
	Return the SQL CREATE command for this message type
	@param outfile: file like object to print to.
	@param fields: which fields to put in the create.  Defaults to all.
	@param extraFields: A sequence of tuples containing (name,sql type) for additional fields
	@param addCoastGuardFields: Add the extra fields that come after the NMEA check some from the USCG N-AIS format
	@param dbType: Which flavor of database we are using so that the create is tailored ('sqlite' or 'postgres')
	@type addCoastGuardFields: bool
	@return: sql create string
	@rtype: str

	@see: sqlCreate
	'''
	# FIX: should this sqlCreate be the same as in LaTeX (createFuncName) rather than hard coded?
	outfile.write(str(sqlCreate(fields,extraFields,addCoastGuardFields,dbType=dbType)))

def sqlCreate(fields=None, extraFields=None, addCoastGuardFields=True, dbType='postgres'):
	'''
	Return the sqlhelp object to create the table.

	@param fields: which fields to put in the create.  Defaults to all.
	@param extraFields: A sequence of tuples containing (name,sql type) for additional fields
	@param addCoastGuardFields: Add the extra fields that come after the NMEA check some from the USCG N-AIS format
	@type addCoastGuardFields: bool
	@param dbType: Which flavor of database we are using so that the create is tailored ('sqlite' or 'postgres')
	@return: An object that can be used to generate a return
	@rtype: sqlhelp.create
	'''
	if None == fields: fields = fieldList
	import sqlhelp
	c = sqlhelp.create('imo_met_hydro',dbType=dbType)
	c.addPrimaryKey()
	if 'MessageID' in fields: c.addInt ('MessageID')
	if 'RepeatIndicator' in fields: c.addInt ('RepeatIndicator')
	if 'UserID' in fields: c.addInt ('UserID')
	if 'Spare' in fields: c.addInt ('Spare')
	if 'dac' in fields: c.addInt ('dac')
	if 'fid' in fields: c.addInt ('fid')
	if dbType != 'postgres':
		if 'latitude' in fields: c.addDecimal('latitude',7,4)
	if dbType != 'postgres':
		if 'longitude' in fields: c.addDecimal('longitude',7,4)
	if 'day' in fields: c.addInt ('day')
	if 'hour' in fields: c.addInt ('hour')
	if 'min' in fields: c.addInt ('min')
	if 'avewind' in fields: c.addInt ('avewind')
	if 'windgust' in fields: c.addInt ('windgust')
	if 'winddir' in fields: c.addInt ('winddir')
	if 'windgustdir' in fields: c.addInt ('windgustdir')
	if 'airtemp' in fields: c.addDecimal('airtemp',4,1)
	if 'relhumid' in fields: c.addInt ('relhumid')
	if 'dewpoint' in fields: c.addDecimal('dewpoint',4,1)
	if 'airpressure' in fields: c.addDecimal('airpressure',3,0)
	if 'airpressuretrend' in fields: c.addInt ('airpressuretrend')
	if 'horizvis' in fields: c.addDecimal('horizvis',3,1)
	if 'waterlevel' in fields: c.addDecimal('waterlevel',3,1)
	if 'waterleveltrend' in fields: c.addInt ('waterleveltrend')
	if 'surfcurspeed' in fields: c.addDecimal('surfcurspeed',3,1)
	if 'surfcurdir' in fields: c.addInt ('surfcurdir')
	if 'curspeed2' in fields: c.addDecimal('curspeed2',3,1)
	if 'curdir2' in fields: c.addInt ('curdir2')
	if 'curlevel2' in fields: c.addInt ('curlevel2')
	if 'curspeed3' in fields: c.addDecimal('curspeed3',3,1)
	if 'curdir3' in fields: c.addInt ('curdir3')
	if 'curlevel3' in fields: c.addInt ('curlevel3')
	if 'sigwaveheight' in fields: c.addDecimal('sigwaveheight',3,1)
	if 'waveperiod' in fields: c.addInt ('waveperiod')
	if 'wavedir' in fields: c.addInt ('wavedir')
	if 'swellheight' in fields: c.addDecimal('swellheight',3,1)
	if 'swellperiod' in fields: c.addInt ('swellperiod')
	if 'swelldir' in fields: c.addInt ('swelldir')
	if 'seastate' in fields: c.addInt ('seastate')
	if 'watertemp' in fields: c.addDecimal('watertemp',4,1)
	if 'preciptype' in fields: c.addInt ('preciptype')
	if 'salinity' in fields: c.addDecimal('salinity',3,1)
	if 'ice' in fields: c.addInt ('ice')
	if 'Spare2' in fields: c.addInt ('Spare2')

	if addCoastGuardFields:
		# c.addInt('cg_s_rssi')     # Relative signal strength indicator
		# c.addInt('cg_d_strength')        # dBm receive strength
		# c.addVarChar('cg_x',10) # Idonno
		c.addInt('cg_t_arrival')        # Receive timestamp from the AIS equipment 'T'
		c.addInt('cg_s_slotnum')        # Slot received in
		c.addVarChar('cg_r',15)   # Receiver station ID  -  should usually be an MMSI, but sometimes is a string
		c.addInt('cg_sec')        # UTC seconds since the epoch

		c.addTimestamp('cg_timestamp') # UTC decoded cg_sec - not actually in the data stream

	if dbType == 'postgres':
		#--- EPSG 4326 : WGS 84
		#INSERT INTO "spatial_ref_sys" ("srid","auth_name","auth_srid","srtext","proj4text") VALUES (4326,'EPSG',4326,'GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.01745329251994328,AUTHORITY["EPSG","9122"]],AUTHORITY["EPSG","4326"]]','+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs ');
		c.addPostGIS('Position','POINT',2,SRID=4326);

	return c

def sqlInsertStr(params, outfile=sys.stdout, extraParams=None, dbType='postgres'):
	'''
	Return the SQL INSERT command for this message type
	@param params: dictionary of values keyed by field name
	@param outfile: file like object to print to.
	@param extraParams: A sequence of tuples containing (name,sql type) for additional fields
	@return: sql create string
	@rtype: str

	@see: sqlCreate
	'''
	outfile.write(str(sqlInsert(params,extraParams,dbType=dbType)))


def sqlInsert(params,extraParams=None,dbType='postgres'):
	'''
	Give the SQL INSERT statement
	@param params: dict keyed by field name of values
	@param extraParams: any extra fields that you have created beyond the normal ais message fields
	@rtype: sqlhelp.insert
	@return: insert class instance
	@todo: allow optional type checking of params?
	@warning: this will take invalid keys happily and do what???
	'''
	import sqlhelp
	i = sqlhelp.insert('imo_met_hydro',dbType=dbType)

	if dbType=='postgres':
		finished = []
		for key in params:
			if key in finished: 
				continue

			if key not in toPgFields and key not in fromPgFields:
				if type(params[key])==Decimal: i.add(key,float(params[key]))
				else: i.add(key,params[key])
			else:
				if key in fromPgFields:
					val = params[key]
				        # Had better be a WKT type like POINT(-88.1 30.321)
					i.addPostGIS(key,val)
					finished.append(key)
				else:
					# Need to construct the type.
					pgName = toPgFields[key]
					#valStr='GeomFromText(\''+pgTypes[pgName]+'('
					valStr=pgTypes[pgName]+'('
					vals = []
					for nonPgKey in fromPgFields[pgName]:
						vals.append(str(params[nonPgKey]))
						finished.append(nonPgKey)
					valStr+=' '.join(vals)+')'
					i.addPostGIS(pgName,valStr)
	else:
		for key in params: 
			if type(params[key])==Decimal: i.add(key,float(params[key]))
			else: i.add(key,params[key])

	if None != extraParams:
		for key in extraParams: 
			i.add(key,extraParams[key])

	return i

######################################################################
# LATEX SUPPORT
######################################################################

def latexDefinitionTable(outfile=sys.stdout
		):
	'''
	Return the LaTeX definition table for this message type
	@param outfile: file like object to print to.
	@type outfile: file obj
	@return: LaTeX table string via the outfile
	@rtype: str

	'''
	o = outfile

	o.write('''
\\begin{table}%[htb]
\\centering
\\begin{tabular}{|l|c|l|}
\\hline
Parameter & Number of bits & Description 
\\\\  \\hline\\hline
MessageID & 6 & AIS message number.  Must be 8 \\\\ \hline 
RepeatIndicator & 2 & Indicated how many times a message has been repeated \\\\ \hline 
UserID & 30 & MMSI number of transmitter broadcasting the message \\\\ \hline 
Spare & 2 & Reserved for definition by a regional authority. \\\\ \hline 
dac & 10 & Designated Area Code - part 1 of the IAI \\\\ \hline 
fid & 6 & Functional Identifier - part 2 of the IAI \\\\ \hline 
latitude & 24 & Location of the vessel.  North South location \\\\ \hline 
longitude & 25 & Location of the vessel.  East West location \\\\ \hline 
day & 5 & Day 0..31 \\\\ \hline 
hour & 5 & Hour 0..23 \\\\ \hline 
min & 6 & Min \\\\ \hline 
avewind & 7 & Average wind speed values for the last 10 minutes. \\\\ \hline 
windgust & 7 & Wind gust is the max wind speed value reading  during the last 10 minutes. \\\\ \hline 
winddir & 9 & Wind direction \\\\ \hline 
windgustdir & 9 & Wind direction for the gust. \\\\ \hline 
airtemp & 11 & Dry bulb temperature \\\\ \hline 
relhumid & 7 & Relative humidity \\\\ \hline 
dewpoint & 10 & Dew Point \\\\ \hline 
airpressure & 9 & Air pressure \\\\ \hline 
airpressuretrend & 2 & Air pressure trend \\\\ \hline 
horizvis & 8 & Horizontal visibility \\\\ \hline 
waterlevel & 9 & Water level (incl. tide) \\\\ \hline 
waterleveltrend & 2 & Water level trend \\\\ \hline 
surfcurspeed & 8 & Surface current speed \\\\ \hline 
surfcurdir & 9 & Surface current direction \\\\ \hline 
curspeed2 & 8 & Level 2 current speed \\\\ \hline 
curdir2 & 9 & Level 2 current direction \\\\ \hline 
curlevel2 & 5 & Measuring level below sea surface for level 2 \\\\ \hline 
curspeed3 & 8 & Level 3 current speed \\\\ \hline 
curdir3 & 9 & Level 3 current direction \\\\ \hline 
curlevel3 & 5 & Measuring level below sea surface for level 3 \\\\ \hline 
sigwaveheight & 8 & Significant wave height \\\\ \hline 
waveperiod & 6 & Wave period \\\\ \hline 
wavedir & 9 & Wave direction \\\\ \hline 
swellheight & 8 & Swell height \\\\ \hline 
swellperiod & 6 & Swell period \\\\ \hline 
swelldir & 9 & Swell direction \\\\ \hline 
seastate & 4 & Sea state according to the Beaufort scale \\\\ \hline 
watertemp & 10 & Water temperature \\\\ \hline 
preciptype & 3 & According to WMO \\\\ \hline 
salinity & 9 & Salinity \\\\ \hline 
ice & 2 & Yes or no for the presence of ice \\\\ \hline 
Spare2 & 6 & Must be zero\\\\ \\hline \\hline
Total bits & 352 & Appears to take 2 slots with 72 pad bits to fill the last slot \\\\ \\hline
\\end{tabular}
\\caption{AIS message number 8: IMO meteorological and hydroglogical data.  Specified       in SN\\Circ.236 Annex 2.  Also defined in IALA Guidelines on AIS,       Vol 1, Part 1, Ed. 1.3. Guildeline No 1028.     }
\\label{tab:imo_met_hydro}
\\end{table}
''')

######################################################################
# Text Definition
######################################################################

def textDefinitionTable(outfile=sys.stdout
		,delim='\t'
		):
	'''
	Return the text definition table for this message type
	@param outfile: file like object to print to.
	@type outfile: file obj
	@return: text table string via the outfile
	@rtype: str

	'''
	o = outfile
	o.write('''Parameter'''+delim+'Number of bits'''+delim+'''Description 
MessageID'''+delim+'''6'''+delim+'''AIS message number.  Must be 8
RepeatIndicator'''+delim+'''2'''+delim+'''Indicated how many times a message has been repeated
UserID'''+delim+'''30'''+delim+'''MMSI number of transmitter broadcasting the message
Spare'''+delim+'''2'''+delim+'''Reserved for definition by a regional authority.
dac'''+delim+'''10'''+delim+'''Designated Area Code - part 1 of the IAI
fid'''+delim+'''6'''+delim+'''Functional Identifier - part 2 of the IAI
latitude'''+delim+'''24'''+delim+'''Location of the vessel.  North South location
longitude'''+delim+'''25'''+delim+'''Location of the vessel.  East West location
day'''+delim+'''5'''+delim+'''Day 0..31
hour'''+delim+'''5'''+delim+'''Hour 0..23
min'''+delim+'''6'''+delim+'''Min
avewind'''+delim+'''7'''+delim+'''Average wind speed values for the last 10 minutes.
windgust'''+delim+'''7'''+delim+'''Wind gust is the max wind speed value reading   during the last 10 minutes.
winddir'''+delim+'''9'''+delim+'''Wind direction
windgustdir'''+delim+'''9'''+delim+'''Wind direction for the gust.
airtemp'''+delim+'''11'''+delim+'''Dry bulb temperature
relhumid'''+delim+'''7'''+delim+'''Relative humidity
dewpoint'''+delim+'''10'''+delim+'''Dew Point
airpressure'''+delim+'''9'''+delim+'''Air pressure
airpressuretrend'''+delim+'''2'''+delim+'''Air pressure trend
horizvis'''+delim+'''8'''+delim+'''Horizontal visibility
waterlevel'''+delim+'''9'''+delim+'''Water level (incl. tide)
waterleveltrend'''+delim+'''2'''+delim+'''Water level trend
surfcurspeed'''+delim+'''8'''+delim+'''Surface current speed
surfcurdir'''+delim+'''9'''+delim+'''Surface current direction
curspeed2'''+delim+'''8'''+delim+'''Level 2 current speed
curdir2'''+delim+'''9'''+delim+'''Level 2 current direction
curlevel2'''+delim+'''5'''+delim+'''Measuring level below sea surface for level 2
curspeed3'''+delim+'''8'''+delim+'''Level 3 current speed
curdir3'''+delim+'''9'''+delim+'''Level 3 current direction
curlevel3'''+delim+'''5'''+delim+'''Measuring level below sea surface for level 3
sigwaveheight'''+delim+'''8'''+delim+'''Significant wave height
waveperiod'''+delim+'''6'''+delim+'''Wave period
wavedir'''+delim+'''9'''+delim+'''Wave direction
swellheight'''+delim+'''8'''+delim+'''Swell height
swellperiod'''+delim+'''6'''+delim+'''Swell period
swelldir'''+delim+'''9'''+delim+'''Swell direction
seastate'''+delim+'''4'''+delim+'''Sea state according to the Beaufort scale
watertemp'''+delim+'''10'''+delim+'''Water temperature
preciptype'''+delim+'''3'''+delim+'''According to WMO
salinity'''+delim+'''9'''+delim+'''Salinity
ice'''+delim+'''2'''+delim+'''Yes or no for the presence of ice
Spare2'''+delim+'''6'''+delim+'''Must be zero
Total bits'''+delim+'''352'''+delim+'''Appears to take 2 slots with 72 pad bits to fill the last slot''')


######################################################################
# UNIT TESTING
######################################################################
import unittest
def testParams():
	'''Return a params file base on the testvalue tags.
	@rtype: dict
	@return: params based on testvalue tags
	'''
	params = {}
	params['MessageID'] = 8
	params['RepeatIndicator'] = 1
	params['UserID'] = 1193046
	params['Spare'] = 0
	params['dac'] = 1
	params['fid'] = 11
	params['latitude'] = Decimal('37.42446')
	params['longitude'] = Decimal('-122.16328')
	params['day'] = 3
	params['hour'] = 21
	params['min'] = 58
	params['avewind'] = 23
	params['windgust'] = 35
	params['winddir'] = 329
	params['windgustdir'] = 293
	params['airtemp'] = Decimal('-40.1')
	params['relhumid'] = 99
	params['dewpoint'] = Decimal('-19.2')
	params['airpressure'] = Decimal('1150')
	params['airpressuretrend'] = 2
	params['horizvis'] = Decimal('11.9')
	params['waterlevel'] = Decimal('-8.9')
	params['waterleveltrend'] = 0
	params['surfcurspeed'] = Decimal('22.3')
	params['surfcurdir'] = 321
	params['curspeed2'] = Decimal('12.7')
	params['curdir2'] = 122
	params['curlevel2'] = 29
	params['curspeed3'] = Decimal('19.2')
	params['curdir3'] = 93
	params['curlevel3'] = 28
	params['sigwaveheight'] = Decimal('22.8')
	params['waveperiod'] = 2
	params['wavedir'] = 187
	params['swellheight'] = Decimal('0.2')
	params['swellperiod'] = 59
	params['swelldir'] = 1
	params['seastate'] = 12
	params['watertemp'] = Decimal('48.8')
	params['preciptype'] = 2
	params['salinity'] = Decimal('0.9')
	params['ice'] = 1
	params['Spare2'] = 0

	return params

class Testimo_met_hydro(unittest.TestCase):
	'''Use testvalue tag text from each type to build test case the imo_met_hydro message'''
	def testEncodeDecode(self):

		params = testParams()
		bits   = encode(params)
		r      = decode(bits)

		# Check that each parameter came through ok.
		self.failUnlessEqual(r['MessageID'],params['MessageID'])
		self.failUnlessEqual(r['RepeatIndicator'],params['RepeatIndicator'])
		self.failUnlessEqual(r['UserID'],params['UserID'])
		self.failUnlessEqual(r['Spare'],params['Spare'])
		self.failUnlessEqual(r['dac'],params['dac'])
		self.failUnlessEqual(r['fid'],params['fid'])
		self.failUnlessAlmostEqual(r['latitude'],params['latitude'],4)
		self.failUnlessAlmostEqual(r['longitude'],params['longitude'],4)
		self.failUnlessEqual(r['day'],params['day'])
		self.failUnlessEqual(r['hour'],params['hour'])
		self.failUnlessEqual(r['min'],params['min'])
		self.failUnlessEqual(r['avewind'],params['avewind'])
		self.failUnlessEqual(r['windgust'],params['windgust'])
		self.failUnlessEqual(r['winddir'],params['winddir'])
		self.failUnlessEqual(r['windgustdir'],params['windgustdir'])
		self.failUnlessAlmostEqual(r['airtemp'],params['airtemp'],1)
		self.failUnlessEqual(r['relhumid'],params['relhumid'])
		self.failUnlessAlmostEqual(r['dewpoint'],params['dewpoint'],1)
		self.failUnlessAlmostEqual(r['airpressure'],params['airpressure'],0)
		self.failUnlessEqual(r['airpressuretrend'],params['airpressuretrend'])
		self.failUnlessAlmostEqual(r['horizvis'],params['horizvis'],1)
		self.failUnlessAlmostEqual(r['waterlevel'],params['waterlevel'],1)
		self.failUnlessEqual(r['waterleveltrend'],params['waterleveltrend'])
		self.failUnlessAlmostEqual(r['surfcurspeed'],params['surfcurspeed'],1)
		self.failUnlessEqual(r['surfcurdir'],params['surfcurdir'])
		self.failUnlessAlmostEqual(r['curspeed2'],params['curspeed2'],1)
		self.failUnlessEqual(r['curdir2'],params['curdir2'])
		self.failUnlessEqual(r['curlevel2'],params['curlevel2'])
		self.failUnlessAlmostEqual(r['curspeed3'],params['curspeed3'],1)
		self.failUnlessEqual(r['curdir3'],params['curdir3'])
		self.failUnlessEqual(r['curlevel3'],params['curlevel3'])
		self.failUnlessAlmostEqual(r['sigwaveheight'],params['sigwaveheight'],1)
		self.failUnlessEqual(r['waveperiod'],params['waveperiod'])
		self.failUnlessEqual(r['wavedir'],params['wavedir'])
		self.failUnlessAlmostEqual(r['swellheight'],params['swellheight'],1)
		self.failUnlessEqual(r['swellperiod'],params['swellperiod'])
		self.failUnlessEqual(r['swelldir'],params['swelldir'])
		self.failUnlessEqual(r['seastate'],params['seastate'])
		self.failUnlessAlmostEqual(r['watertemp'],params['watertemp'],1)
		self.failUnlessEqual(r['preciptype'],params['preciptype'])
		self.failUnlessAlmostEqual(r['salinity'],params['salinity'],1)
		self.failUnlessEqual(r['ice'],params['ice'])
		self.failUnlessEqual(r['Spare2'],params['Spare2'])

def addMsgOptions(parser):
	parser.add_option('-d','--decode',dest='doDecode',default=False,action='store_true',
		help='decode a "imo_met_hydro" AIS message')
	parser.add_option('-e','--encode',dest='doEncode',default=False,action='store_true',
		help='encode a "imo_met_hydro" AIS message')
	parser.add_option('--RepeatIndicator-field', dest='RepeatIndicatorField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--UserID-field', dest='UserIDField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--latitude-field', dest='latitudeField',default=Decimal('91'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--longitude-field', dest='longitudeField',default=Decimal('181'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--day-field', dest='dayField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--hour-field', dest='hourField',default=31,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--min-field', dest='minField',default=63,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--avewind-field', dest='avewindField',default=127,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--windgust-field', dest='windgustField',default=127,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--winddir-field', dest='winddirField',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--windgustdir-field', dest='windgustdirField',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--airtemp-field', dest='airtempField',default=Decimal('102.3'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--relhumid-field', dest='relhumidField',default=127,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--dewpoint-field', dest='dewpointField',default=Decimal('51.1'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--airpressure-field', dest='airpressureField',default=Decimal('1311'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--airpressuretrend-field', dest='airpressuretrendField',default=3,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--horizvis-field', dest='horizvisField',default=Decimal('25.5'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--waterlevel-field', dest='waterlevelField',metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--waterleveltrend-field', dest='waterleveltrendField',default=3,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--surfcurspeed-field', dest='surfcurspeedField',default=Decimal('25.5'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--surfcurdir-field', dest='surfcurdirField',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--curspeed2-field', dest='curspeed2Field',default=Decimal('25.5'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--curdir2-field', dest='curdir2Field',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--curlevel2-field', dest='curlevel2Field',default=31,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--curspeed3-field', dest='curspeed3Field',default=Decimal('25.5'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--curdir3-field', dest='curdir3Field',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--curlevel3-field', dest='curlevel3Field',default=31,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--sigwaveheight-field', dest='sigwaveheightField',default=Decimal('25.5'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--waveperiod-field', dest='waveperiodField',default=63,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--wavedir-field', dest='wavedirField',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--swellheight-field', dest='swellheightField',default=Decimal('25.5'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--swellperiod-field', dest='swellperiodField',default=63,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--swelldir-field', dest='swelldirField',default=511,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--seastate-field', dest='seastateField',default=15,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--watertemp-field', dest='watertempField',default=Decimal('92.3'),metavar='udecimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--preciptype-field', dest='preciptypeField',default=7,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--salinity-field', dest='salinityField',default=Decimal('92.3'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--ice-field', dest='iceField',default=3,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')

def main():
	from optparse import OptionParser
	parser = OptionParser(usage="%prog [options]",
		version="%prog "+__version__)

	parser.add_option('--doc-test',dest='doctest',default=False,action='store_true',
		help='run the documentation tests')
	parser.add_option('--unit-test',dest='unittest',default=False,action='store_true',
		help='run the unit tests')
	parser.add_option('-v','--verbose',dest='verbose',default=False,action='store_true',
		help='Make the test output verbose')

	# FIX: remove nmea from binary messages.  No way to build the whole packet?
	# FIX: or build the surrounding msg 8 for a broadcast?
	typeChoices = ('binary','nmeapayload','nmea') # FIX: what about a USCG type message?
	parser.add_option('-t','--type',choices=typeChoices,type='choice',dest='ioType'
		,default='nmeapayload'
		,help='What kind of string to write for encoding ('+', '.join(typeChoices)+') [default: %default]')


	outputChoices = ('std','html','csv','sql' , 'kml','kml-full')
	parser.add_option('-T','--output-type',choices=outputChoices,type='choice',dest='outputType'
		,default='std'
		,help='What kind of string to output ('+', '.join(outputChoices)+') [default: %default]')

	parser.add_option('-o','--output',dest='outputFileName',default=None,
			  help='Name of the python file to write [default: stdout]')

	parser.add_option('-f','--fields',dest='fieldList',default=None, action='append',
			  choices=fieldList,
			  help='Which fields to include in the output.  Currently only for csv output [default: all]')

	parser.add_option('-p','--print-csv-field-list',dest='printCsvfieldList',default=False,action='store_true',
			  help='Print the field name for csv')

	parser.add_option('-c','--sql-create',dest='sqlCreate',default=False,action='store_true',
			  help='Print out an sql create command for the table.')

	parser.add_option('--latex-table',dest='latexDefinitionTable',default=False,action='store_true',
			  help='Print a LaTeX table of the type')

	parser.add_option('--text-table',dest='textDefinitionTable',default=False,action='store_true',
			  help='Print delimited table of the type (for Word table importing)')
	parser.add_option('--delimt-text-table',dest='delimTextDefinitionTable',default='\t'
			  ,help='Delimiter for text table [default: \'%default\'](for Word table importing)')


	dbChoices = ('sqlite','postgres')
	parser.add_option('-D','--db-type',dest='dbType',default='postgres'
			  ,choices=dbChoices,type='choice'
			  ,help='What kind of database ('+', '.join(dbChoices)+') [default: %default]')

	addMsgOptions(parser)

	(options,args) = parser.parse_args()
	success=True

	if options.doctest:
		import os; print os.path.basename(sys.argv[0]), 'doctests ...',
		sys.argv= [sys.argv[0]]
		if options.verbose: sys.argv.append('-v')
		import doctest
		numfail,numtests=doctest.testmod()
		if numfail==0: print 'ok'
		else: 
			print 'FAILED'
			success=False

	if not success: sys.exit('Something Failed')
	del success # Hide success from epydoc

	if options.unittest:
		sys.argv = [sys.argv[0]]
		if options.verbose: sys.argv.append('-v')
		unittest.main()

	outfile = sys.stdout
	if None!=options.outputFileName:
		outfile = file(options.outputFileName,'w')


	if options.doEncode:
		# First make sure all non required options are specified
		if None==options.RepeatIndicatorField: parser.error("missing value for RepeatIndicatorField")
		if None==options.UserIDField: parser.error("missing value for UserIDField")
		if None==options.latitudeField: parser.error("missing value for latitudeField")
		if None==options.longitudeField: parser.error("missing value for longitudeField")
		if None==options.dayField: parser.error("missing value for dayField")
		if None==options.hourField: parser.error("missing value for hourField")
		if None==options.minField: parser.error("missing value for minField")
		if None==options.avewindField: parser.error("missing value for avewindField")
		if None==options.windgustField: parser.error("missing value for windgustField")
		if None==options.winddirField: parser.error("missing value for winddirField")
		if None==options.windgustdirField: parser.error("missing value for windgustdirField")
		if None==options.airtempField: parser.error("missing value for airtempField")
		if None==options.relhumidField: parser.error("missing value for relhumidField")
		if None==options.dewpointField: parser.error("missing value for dewpointField")
		if None==options.airpressureField: parser.error("missing value for airpressureField")
		if None==options.airpressuretrendField: parser.error("missing value for airpressuretrendField")
		if None==options.horizvisField: parser.error("missing value for horizvisField")
		if None==options.waterlevelField: parser.error("missing value for waterlevelField")
		if None==options.waterleveltrendField: parser.error("missing value for waterleveltrendField")
		if None==options.surfcurspeedField: parser.error("missing value for surfcurspeedField")
		if None==options.surfcurdirField: parser.error("missing value for surfcurdirField")
		if None==options.curspeed2Field: parser.error("missing value for curspeed2Field")
		if None==options.curdir2Field: parser.error("missing value for curdir2Field")
		if None==options.curlevel2Field: parser.error("missing value for curlevel2Field")
		if None==options.curspeed3Field: parser.error("missing value for curspeed3Field")
		if None==options.curdir3Field: parser.error("missing value for curdir3Field")
		if None==options.curlevel3Field: parser.error("missing value for curlevel3Field")
		if None==options.sigwaveheightField: parser.error("missing value for sigwaveheightField")
		if None==options.waveperiodField: parser.error("missing value for waveperiodField")
		if None==options.wavedirField: parser.error("missing value for wavedirField")
		if None==options.swellheightField: parser.error("missing value for swellheightField")
		if None==options.swellperiodField: parser.error("missing value for swellperiodField")
		if None==options.swelldirField: parser.error("missing value for swelldirField")
		if None==options.seastateField: parser.error("missing value for seastateField")
		if None==options.watertempField: parser.error("missing value for watertempField")
		if None==options.preciptypeField: parser.error("missing value for preciptypeField")
		if None==options.salinityField: parser.error("missing value for salinityField")
		if None==options.iceField: parser.error("missing value for iceField")
		msgDict={
			'MessageID': '8',
			'RepeatIndicator': options.RepeatIndicatorField,
			'UserID': options.UserIDField,
			'Spare': '0',
			'dac': '1',
			'fid': '11',
			'latitude': options.latitudeField,
			'longitude': options.longitudeField,
			'day': options.dayField,
			'hour': options.hourField,
			'min': options.minField,
			'avewind': options.avewindField,
			'windgust': options.windgustField,
			'winddir': options.winddirField,
			'windgustdir': options.windgustdirField,
			'airtemp': options.airtempField,
			'relhumid': options.relhumidField,
			'dewpoint': options.dewpointField,
			'airpressure': options.airpressureField,
			'airpressuretrend': options.airpressuretrendField,
			'horizvis': options.horizvisField,
			'waterlevel': options.waterlevelField,
			'waterleveltrend': options.waterleveltrendField,
			'surfcurspeed': options.surfcurspeedField,
			'surfcurdir': options.surfcurdirField,
			'curspeed2': options.curspeed2Field,
			'curdir2': options.curdir2Field,
			'curlevel2': options.curlevel2Field,
			'curspeed3': options.curspeed3Field,
			'curdir3': options.curdir3Field,
			'curlevel3': options.curlevel3Field,
			'sigwaveheight': options.sigwaveheightField,
			'waveperiod': options.waveperiodField,
			'wavedir': options.wavedirField,
			'swellheight': options.swellheightField,
			'swellperiod': options.swellperiodField,
			'swelldir': options.swelldirField,
			'seastate': options.seastateField,
			'watertemp': options.watertempField,
			'preciptype': options.preciptypeField,
			'salinity': options.salinityField,
			'ice': options.iceField,
			'Spare2': '0',
		}

		bits = encode(msgDict)
		if 'binary'==options.ioType: print str(bits)
		elif 'nmeapayload'==options.ioType:
		    # FIX: figure out if this might be necessary at compile time
		    #print "bitLen",len(bits)
		    bitLen=len(bits)
		    if bitLen%6!=0:
			bits = bits + BitVector(size=(6 - (bitLen%6)))  # Pad out to multiple of 6
		    #print "result:",binary.bitvectoais6(bits)[0]
		    print binary.bitvectoais6(bits)[0]


		# FIX: Do not emit this option for the binary message payloads.  Does not make sense.
		elif 'nmea'==options.ioType: 
		    #bitLen=len(bits)
                    #if bitLen%6!=0:
		    #	bits = bits + BitVector(size=(6 - (bitLen%6)))  # Pad out to multiple of 6
                    import aisutils.uscg as uscg
                    nmea = uscg.create_nmea(bits)
                    print nmea
                    #
                    #


                    #sys.exit("FIX: need to implement creating nmea capability")
		else: sys.exit('ERROR: unknown ioType.  Help!')


	if options.sqlCreate:
		sqlCreateStr(outfile,options.fieldList,dbType=options.dbType)

	if options.latexDefinitionTable:
		latexDefinitionTable(outfile)

	# For conversion to word tables
	if options.textDefinitionTable:
		textDefinitionTable(outfile,options.delimTextDefinitionTable)

	if options.printCsvfieldList:
		# Make a csv separated list of fields that will be displayed for csv
		if None == options.fieldList: options.fieldList = fieldList
		import StringIO
		buf = StringIO.StringIO()
		for field in options.fieldList:
			buf.write(field+',')
		result = buf.getvalue()
		if result[-1] == ',': print result[:-1]
		else: print result

	if options.doDecode:
		if len(args)==0: args = sys.stdin
		for msg in args:
			bv = None

			if msg[0] in ('$','!') and msg[3:6] in ('VDM','VDO'):
				# Found nmea
				# FIX: do checksum
				bv = binary.ais6tobitvec(msg.split(',')[5])
			else: # either binary or nmeapayload... expect mostly nmeapayloads
				# assumes that an all 0 and 1 string can not be a nmeapayload
				binaryMsg=True
				for c in msg:
					if c not in ('0','1'):
						binaryMsg=False
						break
				if binaryMsg:
					bv = BitVector(bitstring=msg)
				else: # nmeapayload
					bv = binary.ais6tobitvec(msg)

			printFields(decode(bv)
				    ,out=outfile
				    ,format=options.outputType
				    ,fieldList=options.fieldList
				    ,dbType=options.dbType
				    )

############################################################
if __name__=='__main__':
    main()
