#!/usr/bin/env python

__version__ = '$Revision: 4791 $'.split()[1]
__date__ = '$Date: 2012-10-07 $'.split()[1]
__author__ = 'xmlbinmsg'

__doc__='''

Autogenerated python functions to serialize/deserialize binary messages.

Generated by: ../scripts/aisxmlbinmsg2py.py

Need to then wrap these functions with the outer AIS packet and then
convert the whole binary blob to a NMEA string.  Those functions are
not currently provided in this file.

serialize: python to ais binary
deserialize: ais binary to python

The generated code uses translators.py, binary.py, and aisstring.py
which should be packaged with the resulting files.


@requires: U{epydoc<http://epydoc.sourceforge.net/>} > 3.0alpha3
@requires: U{BitVector<http://cheeseshop.python.org/pypi/BitVector>}

@author: '''+__author__+'''
@version: ''' + __version__ +'''
@var __date__: Date of last svn commit
@undocumented: __version__ __author__ __doc__ parser
@status: under development
@license: Generated code has no license
@todo: FIX: put in a description of the message here with fields and types.
'''

import sys
from decimal import Decimal
from BitVector import BitVector

import binary, aisstring

# FIX: check to see if these will be needed
TrueBV  = BitVector(bitstring="1")
"Why always rebuild the True bit?  This should speed things up a bunch"
FalseBV = BitVector(bitstring="0")
"Why always rebuild the False bit?  This should speed things up a bunch"


fieldList = (
	'MessageID',
	'RepeatIndicator',
	'UserID',
	'Spare',
	'dac',
	'fid',
	'efid',
	'numreports',
	'stationid1',
	'time1_day',
	'time1_hour',
	'time1_min',
	'center1_longitude',
	'center1_latitude',
	'timetoexpire1',
	'radius1',
	'stationid2',
	'time2_day',
	'time2_hour',
	'time2_min',
	'center2_longitude',
	'center2_latitude',
	'timetoexpire2',
	'radius2',
	'stationid3',
	'time3_day',
	'time3_hour',
	'time3_min',
	'center3_longitude',
	'center3_latitude',
	'timetoexpire3',
	'radius3',
	'Spare2',
)

fieldListPostgres = (
	'MessageID',
	'RepeatIndicator',
	'UserID',
	'Spare',
	'dac',
	'fid',
	'efid',
	'numreports',
	'stationid1',
	'time1_day',
	'time1_hour',
	'time1_min',
	'center1',	# PostGIS data type
	'timetoexpire1',
	'radius1',
	'stationid2',
	'time2_day',
	'time2_hour',
	'time2_min',
	'center2',	# PostGIS data type
	'timetoexpire2',
	'radius2',
	'stationid3',
	'time3_day',
	'time3_hour',
	'time3_min',
	'center3',	# PostGIS data type
	'timetoexpire3',
	'radius3',
	'Spare2',
)

toPgFields = {
	'center1_longitude':'center1',
	'center1_latitude':'center1',
	'center2_longitude':'center2',
	'center2_latitude':'center2',
	'center3_longitude':'center3',
	'center3_latitude':'center3',
}
'''
Go to the Postgis field names from the straight field name
'''

fromPgFields = {
	'center1':('center1_longitude','center1_latitude',),
	'center2':('center2_longitude','center2_latitude',),
	'center3':('center3_longitude','center3_latitude',),
}
'''
Go from the Postgis field names to the straight field name
'''

pgTypes = {
	'center1':'POINT',
	'center2':'POINT',
	'center3':'POINT',
}
'''
Lookup table for each postgis field name to get its type.
'''

def encode(params, validate=False):
	'''Create a whalenotice binary message payload to pack into an AIS Msg whalenotice.

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 8 (field automatically set to "8")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): Unique ship identification number (MMSI)
	  - Spare(uint): Reserved for definition by a regional authority. (field automatically set to "0")
	  - dac(uint): Designated Area Code - 366 for the United States (field automatically set to "366")
	  - fid(uint): Functional IDentifier - 63 for the Whale Notice (field automatically set to "63")
	  - efid(uint): Extended Functional IDentifier.  1 for the Whale Notice (dac+fid+efid defines the exact message type) (field automatically set to "1")
	  - numreports(uint): Number of detection reports filled out in this message
	  - stationid1(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time1_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time1_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time1_min(uint): Time of most recent whale detection.  UTC minutes
	  - center1_longitude(decimal): Center of the detection zone.  East West location
	  - center1_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire1(uint): Seconds from the detection time until the notice expires
	  - radius1(uint): Distance from center of detection zone (lat/lon above)
	  - stationid2(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time2_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time2_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time2_min(uint): Time of most recent whale detection.  UTC minutes
	  - center2_longitude(decimal): Center of the detection zone.  East West location
	  - center2_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire2(uint): Seconds from the detection time until the notice expires
	  - radius2(uint): Distance from center of detection zone (lat/lon above)
	  - stationid3(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time3_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time3_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time3_min(uint): Time of most recent whale detection.  UTC minutes
	  - center3_longitude(decimal): Center of the detection zone.  East West location
	  - center3_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire3(uint): Seconds from the detection time until the notice expires
	  - radius3(uint): Distance from center of detection zone (lat/lon above)
	  - Spare2(uint): Not used.  Should be set to zero. (field automatically set to "0")
	@param params: Dictionary of field names/values.  Throws a ValueError exception if required is missing
	@param validate: Set to true to cause checking to occur.  Runs slower.  FIX: not implemented.
	@rtype: BitVector
	@return: encoded binary message (for binary messages, this needs to be wrapped in a msg 8
	@note: The returned bits may not be 6 bit aligned.  It is up to you to pad out the bits.
	'''

	bvList = []
	bvList.append(binary.setBitVectorSize(BitVector(intVal=8),6))
	if 'RepeatIndicator' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['RepeatIndicator']),2))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),2))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['UserID']),30))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=0),2))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=366),10))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=63),6))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=1),12))
	if 'numreports' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['numreports']),2))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),2))
	if 'stationid1' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['stationid1']),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),8))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time1_day']),5))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time1_hour']),5))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time1_min']),6))
	if 'center1_longitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['center1_longitude'])*Decimal('600000')),28))
	else:
		bvList.append(binary.bvFromSignedInt(108600000,28))
	if 'center1_latitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['center1_latitude'])*Decimal('600000')),27))
	else:
		bvList.append(binary.bvFromSignedInt(54600000,27))
	if 'timetoexpire1' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['timetoexpire1']),16))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=65535),16))
	if 'radius1' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['radius1']),16))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=65534),16))
	if 'stationid2' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['stationid2']),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),8))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time2_day']),5))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time2_hour']),5))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time2_min']),6))
	if 'center2_longitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['center2_longitude'])*Decimal('600000')),28))
	else:
		bvList.append(binary.bvFromSignedInt(108600000,28))
	if 'center2_latitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['center2_latitude'])*Decimal('600000')),27))
	else:
		bvList.append(binary.bvFromSignedInt(54600000,27))
	if 'timetoexpire2' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['timetoexpire2']),16))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=65535),16))
	if 'radius2' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['radius2']),16))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=65534),16))
	if 'stationid3' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['stationid3']),8))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=0),8))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time3_day']),5))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time3_hour']),5))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=params['time3_min']),6))
	if 'center3_longitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['center3_longitude'])*Decimal('600000')),28))
	else:
		bvList.append(binary.bvFromSignedInt(108600000,28))
	if 'center3_latitude' in params:
		bvList.append(binary.bvFromSignedInt(int(Decimal(params['center3_latitude'])*Decimal('600000')),27))
	else:
		bvList.append(binary.bvFromSignedInt(54600000,27))
	if 'timetoexpire3' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['timetoexpire3']),16))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=65535),16))
	if 'radius3' in params:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=params['radius3']),16))
	else:
		bvList.append(binary.setBitVectorSize(BitVector(intVal=65534),16))
	bvList.append(binary.setBitVectorSize(BitVector(intVal=0),21))

	return binary.joinBV(bvList)

def decode(bv, validate=False):
	'''Unpack a whalenotice message 

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 8 (field automatically set to "8")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): Unique ship identification number (MMSI)
	  - Spare(uint): Reserved for definition by a regional authority. (field automatically set to "0")
	  - dac(uint): Designated Area Code - 366 for the United States (field automatically set to "366")
	  - fid(uint): Functional IDentifier - 63 for the Whale Notice (field automatically set to "63")
	  - efid(uint): Extended Functional IDentifier.  1 for the Whale Notice (dac+fid+efid defines the exact message type) (field automatically set to "1")
	  - numreports(uint): Number of detection reports filled out in this message
	  - stationid1(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time1_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time1_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time1_min(uint): Time of most recent whale detection.  UTC minutes
	  - center1_longitude(decimal): Center of the detection zone.  East West location
	  - center1_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire1(uint): Seconds from the detection time until the notice expires
	  - radius1(uint): Distance from center of detection zone (lat/lon above)
	  - stationid2(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time2_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time2_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time2_min(uint): Time of most recent whale detection.  UTC minutes
	  - center2_longitude(decimal): Center of the detection zone.  East West location
	  - center2_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire2(uint): Seconds from the detection time until the notice expires
	  - radius2(uint): Distance from center of detection zone (lat/lon above)
	  - stationid3(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time3_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time3_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time3_min(uint): Time of most recent whale detection.  UTC minutes
	  - center3_longitude(decimal): Center of the detection zone.  East West location
	  - center3_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire3(uint): Seconds from the detection time until the notice expires
	  - radius3(uint): Distance from center of detection zone (lat/lon above)
	  - Spare2(uint): Not used.  Should be set to zero. (field automatically set to "0")
	@type bv: BitVector
	@param bv: Bits defining a message
	@param validate: Set to true to cause checking to occur.  Runs slower.  FIX: not implemented.
	@rtype: dict
	@return: params
	'''

	#Would be nice to check the bit count here..
	#if validate:
	#	assert (len(bv)==FIX: SOME NUMBER)
	r = {}
	r['MessageID']=8
	r['RepeatIndicator']=int(bv[6:8])
	r['UserID']=int(bv[8:38])
	r['Spare']=0
	r['dac']=366
	r['fid']=63
	r['efid']=1
	r['numreports']=int(bv[68:70])
	r['stationid1']=int(bv[70:78])
	r['time1_day']=int(bv[78:83])
	r['time1_hour']=int(bv[83:88])
	r['time1_min']=int(bv[88:94])
	r['center1_longitude']=Decimal(binary.signedIntFromBV(bv[94:122]))/Decimal('600000')
	r['center1_latitude']=Decimal(binary.signedIntFromBV(bv[122:149]))/Decimal('600000')
	r['timetoexpire1']=int(bv[149:165])
	r['radius1']=int(bv[165:181])
	r['stationid2']=int(bv[181:189])
	r['time2_day']=int(bv[189:194])
	r['time2_hour']=int(bv[194:199])
	r['time2_min']=int(bv[199:205])
	r['center2_longitude']=Decimal(binary.signedIntFromBV(bv[205:233]))/Decimal('600000')
	r['center2_latitude']=Decimal(binary.signedIntFromBV(bv[233:260]))/Decimal('600000')
	r['timetoexpire2']=int(bv[260:276])
	r['radius2']=int(bv[276:292])
	r['stationid3']=int(bv[292:300])
	r['time3_day']=int(bv[300:305])
	r['time3_hour']=int(bv[305:310])
	r['time3_min']=int(bv[310:316])
	r['center3_longitude']=Decimal(binary.signedIntFromBV(bv[316:344]))/Decimal('600000')
	r['center3_latitude']=Decimal(binary.signedIntFromBV(bv[344:371]))/Decimal('600000')
	r['timetoexpire3']=int(bv[371:387])
	r['radius3']=int(bv[387:403])
	r['Spare2']=0
	return r

def decodeMessageID(bv, validate=False):
	return 8

def decodeRepeatIndicator(bv, validate=False):
	return int(bv[6:8])

def decodeUserID(bv, validate=False):
	return int(bv[8:38])

def decodeSpare(bv, validate=False):
	return 0

def decodedac(bv, validate=False):
	return 366

def decodefid(bv, validate=False):
	return 63

def decodeefid(bv, validate=False):
	return 1

def decodenumreports(bv, validate=False):
	return int(bv[68:70])

def decodestationid1(bv, validate=False):
	return int(bv[70:78])

def decodetime1_day(bv, validate=False):
	return int(bv[78:83])

def decodetime1_hour(bv, validate=False):
	return int(bv[83:88])

def decodetime1_min(bv, validate=False):
	return int(bv[88:94])

def decodecenter1_longitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[94:122]))/Decimal('600000')

def decodecenter1_latitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[122:149]))/Decimal('600000')

def decodetimetoexpire1(bv, validate=False):
	return int(bv[149:165])

def decoderadius1(bv, validate=False):
	return int(bv[165:181])

def decodestationid2(bv, validate=False):
	return int(bv[181:189])

def decodetime2_day(bv, validate=False):
	return int(bv[189:194])

def decodetime2_hour(bv, validate=False):
	return int(bv[194:199])

def decodetime2_min(bv, validate=False):
	return int(bv[199:205])

def decodecenter2_longitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[205:233]))/Decimal('600000')

def decodecenter2_latitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[233:260]))/Decimal('600000')

def decodetimetoexpire2(bv, validate=False):
	return int(bv[260:276])

def decoderadius2(bv, validate=False):
	return int(bv[276:292])

def decodestationid3(bv, validate=False):
	return int(bv[292:300])

def decodetime3_day(bv, validate=False):
	return int(bv[300:305])

def decodetime3_hour(bv, validate=False):
	return int(bv[305:310])

def decodetime3_min(bv, validate=False):
	return int(bv[310:316])

def decodecenter3_longitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[316:344]))/Decimal('600000')

def decodecenter3_latitude(bv, validate=False):
	return Decimal(binary.signedIntFromBV(bv[344:371]))/Decimal('600000')

def decodetimetoexpire3(bv, validate=False):
	return int(bv[371:387])

def decoderadius3(bv, validate=False):
	return int(bv[387:403])

def decodeSpare2(bv, validate=False):
	return 0


def printHtml(params, out=sys.stdout):
		out.write("<h3>whalenotice</h3>\n")
		out.write("<table border=\"1\">\n")
		out.write("<tr bgcolor=\"orange\">\n")
		out.write("<th align=\"left\">Field Name</th>\n")
		out.write("<th align=\"left\">Type</th>\n")
		out.write("<th align=\"left\">Value</th>\n")
		out.write("<th align=\"left\">Value in Lookup Table</th>\n")
		out.write("<th align=\"left\">Units</th>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>MessageID</td>\n")
		out.write("<td>uint</td>\n")
		if 'MessageID' in params:
			out.write("	<td>"+str(params['MessageID'])+"</td>\n")
			out.write("	<td>"+str(params['MessageID'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>RepeatIndicator</td>\n")
		out.write("<td>uint</td>\n")
		if 'RepeatIndicator' in params:
			out.write("	<td>"+str(params['RepeatIndicator'])+"</td>\n")
			if str(params['RepeatIndicator']) in RepeatIndicatorDecodeLut:
				out.write("<td>"+RepeatIndicatorDecodeLut[str(params['RepeatIndicator'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>UserID</td>\n")
		out.write("<td>uint</td>\n")
		if 'UserID' in params:
			out.write("	<td>"+str(params['UserID'])+"</td>\n")
			out.write("	<td>"+str(params['UserID'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>Spare</td>\n")
		out.write("<td>uint</td>\n")
		if 'Spare' in params:
			out.write("	<td>"+str(params['Spare'])+"</td>\n")
			out.write("	<td>"+str(params['Spare'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>dac</td>\n")
		out.write("<td>uint</td>\n")
		if 'dac' in params:
			out.write("	<td>"+str(params['dac'])+"</td>\n")
			out.write("	<td>"+str(params['dac'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>fid</td>\n")
		out.write("<td>uint</td>\n")
		if 'fid' in params:
			out.write("	<td>"+str(params['fid'])+"</td>\n")
			out.write("	<td>"+str(params['fid'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>efid</td>\n")
		out.write("<td>uint</td>\n")
		if 'efid' in params:
			out.write("	<td>"+str(params['efid'])+"</td>\n")
			out.write("	<td>"+str(params['efid'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>numreports</td>\n")
		out.write("<td>uint</td>\n")
		if 'numreports' in params:
			out.write("	<td>"+str(params['numreports'])+"</td>\n")
			out.write("	<td>"+str(params['numreports'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>stationid1</td>\n")
		out.write("<td>uint</td>\n")
		if 'stationid1' in params:
			out.write("	<td>"+str(params['stationid1'])+"</td>\n")
			out.write("	<td>"+str(params['stationid1'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time1_day</td>\n")
		out.write("<td>uint</td>\n")
		if 'time1_day' in params:
			out.write("	<td>"+str(params['time1_day'])+"</td>\n")
			out.write("	<td>"+str(params['time1_day'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time1_hour</td>\n")
		out.write("<td>uint</td>\n")
		if 'time1_hour' in params:
			out.write("	<td>"+str(params['time1_hour'])+"</td>\n")
			out.write("	<td>"+str(params['time1_hour'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time1_min</td>\n")
		out.write("<td>uint</td>\n")
		if 'time1_min' in params:
			out.write("	<td>"+str(params['time1_min'])+"</td>\n")
			out.write("	<td>"+str(params['time1_min'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>center1_longitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'center1_longitude' in params:
			out.write("	<td>"+str(params['center1_longitude'])+"</td>\n")
			out.write("	<td>"+str(params['center1_longitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>center1_latitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'center1_latitude' in params:
			out.write("	<td>"+str(params['center1_latitude'])+"</td>\n")
			out.write("	<td>"+str(params['center1_latitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>timetoexpire1</td>\n")
		out.write("<td>uint</td>\n")
		if 'timetoexpire1' in params:
			out.write("	<td>"+str(params['timetoexpire1'])+"</td>\n")
			if str(params['timetoexpire1']) in timetoexpire1DecodeLut:
				out.write("<td>"+timetoexpire1DecodeLut[str(params['timetoexpire1'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>seconds</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>radius1</td>\n")
		out.write("<td>uint</td>\n")
		if 'radius1' in params:
			out.write("	<td>"+str(params['radius1'])+"</td>\n")
			out.write("	<td>"+str(params['radius1'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>stationid2</td>\n")
		out.write("<td>uint</td>\n")
		if 'stationid2' in params:
			out.write("	<td>"+str(params['stationid2'])+"</td>\n")
			out.write("	<td>"+str(params['stationid2'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time2_day</td>\n")
		out.write("<td>uint</td>\n")
		if 'time2_day' in params:
			out.write("	<td>"+str(params['time2_day'])+"</td>\n")
			out.write("	<td>"+str(params['time2_day'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time2_hour</td>\n")
		out.write("<td>uint</td>\n")
		if 'time2_hour' in params:
			out.write("	<td>"+str(params['time2_hour'])+"</td>\n")
			out.write("	<td>"+str(params['time2_hour'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time2_min</td>\n")
		out.write("<td>uint</td>\n")
		if 'time2_min' in params:
			out.write("	<td>"+str(params['time2_min'])+"</td>\n")
			out.write("	<td>"+str(params['time2_min'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>center2_longitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'center2_longitude' in params:
			out.write("	<td>"+str(params['center2_longitude'])+"</td>\n")
			out.write("	<td>"+str(params['center2_longitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>center2_latitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'center2_latitude' in params:
			out.write("	<td>"+str(params['center2_latitude'])+"</td>\n")
			out.write("	<td>"+str(params['center2_latitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>timetoexpire2</td>\n")
		out.write("<td>uint</td>\n")
		if 'timetoexpire2' in params:
			out.write("	<td>"+str(params['timetoexpire2'])+"</td>\n")
			if str(params['timetoexpire2']) in timetoexpire2DecodeLut:
				out.write("<td>"+timetoexpire2DecodeLut[str(params['timetoexpire2'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>seconds</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>radius2</td>\n")
		out.write("<td>uint</td>\n")
		if 'radius2' in params:
			out.write("	<td>"+str(params['radius2'])+"</td>\n")
			out.write("	<td>"+str(params['radius2'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>stationid3</td>\n")
		out.write("<td>uint</td>\n")
		if 'stationid3' in params:
			out.write("	<td>"+str(params['stationid3'])+"</td>\n")
			out.write("	<td>"+str(params['stationid3'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time3_day</td>\n")
		out.write("<td>uint</td>\n")
		if 'time3_day' in params:
			out.write("	<td>"+str(params['time3_day'])+"</td>\n")
			out.write("	<td>"+str(params['time3_day'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time3_hour</td>\n")
		out.write("<td>uint</td>\n")
		if 'time3_hour' in params:
			out.write("	<td>"+str(params['time3_hour'])+"</td>\n")
			out.write("	<td>"+str(params['time3_hour'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>time3_min</td>\n")
		out.write("<td>uint</td>\n")
		if 'time3_min' in params:
			out.write("	<td>"+str(params['time3_min'])+"</td>\n")
			out.write("	<td>"+str(params['time3_min'])+"</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>center3_longitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'center3_longitude' in params:
			out.write("	<td>"+str(params['center3_longitude'])+"</td>\n")
			out.write("	<td>"+str(params['center3_longitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>center3_latitude</td>\n")
		out.write("<td>decimal</td>\n")
		if 'center3_latitude' in params:
			out.write("	<td>"+str(params['center3_latitude'])+"</td>\n")
			out.write("	<td>"+str(params['center3_latitude'])+"</td>\n")
		out.write("<td>degrees</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>timetoexpire3</td>\n")
		out.write("<td>uint</td>\n")
		if 'timetoexpire3' in params:
			out.write("	<td>"+str(params['timetoexpire3'])+"</td>\n")
			if str(params['timetoexpire3']) in timetoexpire3DecodeLut:
				out.write("<td>"+timetoexpire3DecodeLut[str(params['timetoexpire3'])]+"</td>")
			else:
				out.write("<td><i>Missing LUT entry</i></td>")
		out.write("<td>seconds</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>radius3</td>\n")
		out.write("<td>uint</td>\n")
		if 'radius3' in params:
			out.write("	<td>"+str(params['radius3'])+"</td>\n")
			out.write("	<td>"+str(params['radius3'])+"</td>\n")
		out.write("<td>m</td>\n")
		out.write("</tr>\n")
		out.write("\n")
		out.write("<tr>\n")
		out.write("<td>Spare2</td>\n")
		out.write("<td>uint</td>\n")
		if 'Spare2' in params:
			out.write("	<td>"+str(params['Spare2'])+"</td>\n")
			out.write("	<td>"+str(params['Spare2'])+"</td>\n")
		out.write("</tr>\n")
		out.write("</table>\n")


def printKml(params, out=sys.stdout):
	'''KML (Keyhole Markup Language) for Google Earth, but without the header/footer'''
	out.write("\	<Placemark>\n")
	out.write("\t	<name>"+str(params['stationsid'])+"</name>\n")
	out.write("\t\t<description>\n")
	import StringIO
	buf = StringIO.StringIO()
	printHtml(params,buf)
	import cgi
	out.write(cgi.escape(buf.getvalue()))
	out.write("\t\t</description>\n")
	out.write("\t\t<styleUrl>#m_ylw-pushpin_copy0</styleUrl>\n")
	out.write("\t\t<Point>\n")
	out.write("\t\t\t<coordinates>")
	out.write(str(params['center1_longitude']))
	out.write(',')
	out.write(str(params['center1_latitude']))
	out.write(",0</coordinates>\n")
	out.write("\t\t</Point>\n")
	out.write("\t</Placemark>\n")

def printFields(params, out=sys.stdout, format='std', fieldList=None, dbType='postgres'):
	'''Print a whalenotice message to stdout.

	Fields in params:
	  - MessageID(uint): AIS message number.  Must be 8 (field automatically set to "8")
	  - RepeatIndicator(uint): Indicated how many times a message has been repeated
	  - UserID(uint): Unique ship identification number (MMSI)
	  - Spare(uint): Reserved for definition by a regional authority. (field automatically set to "0")
	  - dac(uint): Designated Area Code - 366 for the United States (field automatically set to "366")
	  - fid(uint): Functional IDentifier - 63 for the Whale Notice (field automatically set to "63")
	  - efid(uint): Extended Functional IDentifier.  1 for the Whale Notice (dac+fid+efid defines the exact message type) (field automatically set to "1")
	  - numreports(uint): Number of detection reports filled out in this message
	  - stationid1(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time1_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time1_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time1_min(uint): Time of most recent whale detection.  UTC minutes
	  - center1_longitude(decimal): Center of the detection zone.  East West location
	  - center1_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire1(uint): Seconds from the detection time until the notice expires
	  - radius1(uint): Distance from center of detection zone (lat/lon above)
	  - stationid2(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time2_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time2_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time2_min(uint): Time of most recent whale detection.  UTC minutes
	  - center2_longitude(decimal): Center of the detection zone.  East West location
	  - center2_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire2(uint): Seconds from the detection time until the notice expires
	  - radius2(uint): Distance from center of detection zone (lat/lon above)
	  - stationid3(uint): Identifier of the station that recorded the whale.  Usually a number.
	  - time3_day(uint): Time of most recent whale detection.  UTC day of the month 1..31
	  - time3_hour(uint): Time of most recent whale detection.  UTC hours 0..23
	  - time3_min(uint): Time of most recent whale detection.  UTC minutes
	  - center3_longitude(decimal): Center of the detection zone.  East West location
	  - center3_latitude(decimal): Center of the detection zone.  North South location
	  - timetoexpire3(uint): Seconds from the detection time until the notice expires
	  - radius3(uint): Distance from center of detection zone (lat/lon above)
	  - Spare2(uint): Not used.  Should be set to zero. (field automatically set to "0")
	@param params: Dictionary of field names/values.  
	@param out: File like object to write to
	@rtype: stdout
	@return: text to out
	'''

	if 'std'==format:
		out.write("whalenotice:\n")
		if 'MessageID' in params: out.write("	MessageID:          "+str(params['MessageID'])+"\n")
		if 'RepeatIndicator' in params: out.write("	RepeatIndicator:    "+str(params['RepeatIndicator'])+"\n")
		if 'UserID' in params: out.write("	UserID:             "+str(params['UserID'])+"\n")
		if 'Spare' in params: out.write("	Spare:              "+str(params['Spare'])+"\n")
		if 'dac' in params: out.write("	dac:                "+str(params['dac'])+"\n")
		if 'fid' in params: out.write("	fid:                "+str(params['fid'])+"\n")
		if 'efid' in params: out.write("	efid:               "+str(params['efid'])+"\n")
		if 'numreports' in params: out.write("	numreports:         "+str(params['numreports'])+"\n")
		if 'stationid1' in params: out.write("	stationid1:         "+str(params['stationid1'])+"\n")
		if 'time1_day' in params: out.write("	time1_day:          "+str(params['time1_day'])+"\n")
		if 'time1_hour' in params: out.write("	time1_hour:         "+str(params['time1_hour'])+"\n")
		if 'time1_min' in params: out.write("	time1_min:          "+str(params['time1_min'])+"\n")
		if 'center1_longitude' in params: out.write("	center1_longitude:  "+str(params['center1_longitude'])+"\n")
		if 'center1_latitude' in params: out.write("	center1_latitude:   "+str(params['center1_latitude'])+"\n")
		if 'timetoexpire1' in params: out.write("	timetoexpire1:      "+str(params['timetoexpire1'])+"\n")
		if 'radius1' in params: out.write("	radius1:            "+str(params['radius1'])+"\n")
		if 'stationid2' in params: out.write("	stationid2:         "+str(params['stationid2'])+"\n")
		if 'time2_day' in params: out.write("	time2_day:          "+str(params['time2_day'])+"\n")
		if 'time2_hour' in params: out.write("	time2_hour:         "+str(params['time2_hour'])+"\n")
		if 'time2_min' in params: out.write("	time2_min:          "+str(params['time2_min'])+"\n")
		if 'center2_longitude' in params: out.write("	center2_longitude:  "+str(params['center2_longitude'])+"\n")
		if 'center2_latitude' in params: out.write("	center2_latitude:   "+str(params['center2_latitude'])+"\n")
		if 'timetoexpire2' in params: out.write("	timetoexpire2:      "+str(params['timetoexpire2'])+"\n")
		if 'radius2' in params: out.write("	radius2:            "+str(params['radius2'])+"\n")
		if 'stationid3' in params: out.write("	stationid3:         "+str(params['stationid3'])+"\n")
		if 'time3_day' in params: out.write("	time3_day:          "+str(params['time3_day'])+"\n")
		if 'time3_hour' in params: out.write("	time3_hour:         "+str(params['time3_hour'])+"\n")
		if 'time3_min' in params: out.write("	time3_min:          "+str(params['time3_min'])+"\n")
		if 'center3_longitude' in params: out.write("	center3_longitude:  "+str(params['center3_longitude'])+"\n")
		if 'center3_latitude' in params: out.write("	center3_latitude:   "+str(params['center3_latitude'])+"\n")
		if 'timetoexpire3' in params: out.write("	timetoexpire3:      "+str(params['timetoexpire3'])+"\n")
		if 'radius3' in params: out.write("	radius3:            "+str(params['radius3'])+"\n")
		if 'Spare2' in params: out.write("	Spare2:             "+str(params['Spare2'])+"\n")
	elif 'csv'==format:
		if None == options.fieldList:
			options.fieldList = fieldList
		needComma = False;
		for field in fieldList:
			if needComma: out.write(',')
			needComma = True
			if field in params:
				out.write(str(params[field]))
			# else: leave it empty
		out.write("\n")
	elif 'html'==format:
		printHtml(params,out)
	elif 'sql'==format:
		sqlInsertStr(params,out,dbType=dbType)
	elif 'kml'==format:
		printKml(params,out)
	elif 'kml-full'==format:
		out.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n")
		out.write("<kml xmlns=\"http://earth.google.com/kml/2.1\">\n")
		out.write("<Document>\n")
		out.write("	<name>whalenotice</name>\n")
		printKml(params,out)
		out.write("</Document>\n")
		out.write("</kml>\n")
	else: 
		print "ERROR: unknown format:",format
		assert False

	return # Nothing to return

RepeatIndicatorEncodeLut = {
	'default':'0',
	'do not repeat any more':'3',
	} #RepeatIndicatorEncodeLut

RepeatIndicatorDecodeLut = {
	'0':'default',
	'3':'do not repeat any more',
	} # RepeatIndicatorEncodeLut

timetoexpire1EncodeLut = {
	'No detection/notice active in region':'0',
	} #timetoexpire1EncodeLut

timetoexpire1DecodeLut = {
	'0':'No detection/notice active in region',
	} # timetoexpire1EncodeLut

timetoexpire2EncodeLut = {
	'No detection/notice active in region':'0',
	} #timetoexpire2EncodeLut

timetoexpire2DecodeLut = {
	'0':'No detection/notice active in region',
	} # timetoexpire2EncodeLut

timetoexpire3EncodeLut = {
	'No detection/notice active in region':'0',
	} #timetoexpire3EncodeLut

timetoexpire3DecodeLut = {
	'0':'No detection/notice active in region',
	} # timetoexpire3EncodeLut

######################################################################
# SQL SUPPORT
######################################################################

dbTableName='whalenotice'
'Database table name'

def sqlCreateStr(outfile=sys.stdout, fields=None, extraFields=None
		,addCoastGuardFields=True
		,dbType='postgres'
		):
	'''
	Return the SQL CREATE command for this message type
	@param outfile: file like object to print to.
	@param fields: which fields to put in the create.  Defaults to all.
	@param extraFields: A sequence of tuples containing (name,sql type) for additional fields
	@param addCoastGuardFields: Add the extra fields that come after the NMEA check some from the USCG N-AIS format
	@param dbType: Which flavor of database we are using so that the create is tailored ('sqlite' or 'postgres')
	@type addCoastGuardFields: bool
	@return: sql create string
	@rtype: str

	@see: sqlCreate
	'''
	# FIX: should this sqlCreate be the same as in LaTeX (createFuncName) rather than hard coded?
	outfile.write(str(sqlCreate(fields,extraFields,addCoastGuardFields,dbType=dbType)))

def sqlCreate(fields=None, extraFields=None, addCoastGuardFields=True, dbType='postgres'):
	'''
	Return the sqlhelp object to create the table.

	@param fields: which fields to put in the create.  Defaults to all.
	@param extraFields: A sequence of tuples containing (name,sql type) for additional fields
	@param addCoastGuardFields: Add the extra fields that come after the NMEA check some from the USCG N-AIS format
	@type addCoastGuardFields: bool
	@param dbType: Which flavor of database we are using so that the create is tailored ('sqlite' or 'postgres')
	@return: An object that can be used to generate a return
	@rtype: sqlhelp.create
	'''
	if None == fields: fields = fieldList
	import sqlhelp
	c = sqlhelp.create('whalenotice',dbType=dbType)
	c.addPrimaryKey()
	if 'MessageID' in fields: c.addInt ('MessageID')
	if 'RepeatIndicator' in fields: c.addInt ('RepeatIndicator')
	if 'UserID' in fields: c.addInt ('UserID')
	if 'Spare' in fields: c.addInt ('Spare')
	if 'dac' in fields: c.addInt ('dac')
	if 'fid' in fields: c.addInt ('fid')
	if 'efid' in fields: c.addInt ('efid')
	if 'numreports' in fields: c.addInt ('numreports')
	if 'stationid1' in fields: c.addInt ('stationid1')
	if 'time1_day' in fields: c.addInt ('time1_day')
	if 'time1_hour' in fields: c.addInt ('time1_hour')
	if 'time1_min' in fields: c.addInt ('time1_min')
	if dbType != 'postgres':
		if 'center1_longitude' in fields: c.addDecimal('center1_longitude',8,5)
	if dbType != 'postgres':
		if 'center1_latitude' in fields: c.addDecimal('center1_latitude',8,5)
	if 'timetoexpire1' in fields: c.addInt ('timetoexpire1')
	if 'radius1' in fields: c.addInt ('radius1')
	if 'stationid2' in fields: c.addInt ('stationid2')
	if 'time2_day' in fields: c.addInt ('time2_day')
	if 'time2_hour' in fields: c.addInt ('time2_hour')
	if 'time2_min' in fields: c.addInt ('time2_min')
	if dbType != 'postgres':
		if 'center2_longitude' in fields: c.addDecimal('center2_longitude',8,5)
	if dbType != 'postgres':
		if 'center2_latitude' in fields: c.addDecimal('center2_latitude',8,5)
	if 'timetoexpire2' in fields: c.addInt ('timetoexpire2')
	if 'radius2' in fields: c.addInt ('radius2')
	if 'stationid3' in fields: c.addInt ('stationid3')
	if 'time3_day' in fields: c.addInt ('time3_day')
	if 'time3_hour' in fields: c.addInt ('time3_hour')
	if 'time3_min' in fields: c.addInt ('time3_min')
	if dbType != 'postgres':
		if 'center3_longitude' in fields: c.addDecimal('center3_longitude',8,5)
	if dbType != 'postgres':
		if 'center3_latitude' in fields: c.addDecimal('center3_latitude',8,5)
	if 'timetoexpire3' in fields: c.addInt ('timetoexpire3')
	if 'radius3' in fields: c.addInt ('radius3')
	if 'Spare2' in fields: c.addInt ('Spare2')

	if addCoastGuardFields:
		# c.addInt('cg_s_rssi')     # Relative signal strength indicator
		# c.addInt('cg_d_strength')        # dBm receive strength
		# c.addVarChar('cg_x',10) # Idonno
		c.addInt('cg_t_arrival')        # Receive timestamp from the AIS equipment 'T'
		c.addInt('cg_s_slotnum')        # Slot received in
		c.addVarChar('cg_r',15)   # Receiver station ID  -  should usually be an MMSI, but sometimes is a string
		c.addInt('cg_sec')        # UTC seconds since the epoch

		c.addTimestamp('cg_timestamp') # UTC decoded cg_sec - not actually in the data stream

	if dbType == 'postgres':
		#--- EPSG 4326 : WGS 84
		#INSERT INTO "spatial_ref_sys" ("srid","auth_name","auth_srid","srtext","proj4text") VALUES (4326,'EPSG',4326,'GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.01745329251994328,AUTHORITY["EPSG","9122"]],AUTHORITY["EPSG","4326"]]','+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs ');
		c.addPostGIS('center1','POINT',2,SRID=4326);
		#--- EPSG 4326 : WGS 84
		#INSERT INTO "spatial_ref_sys" ("srid","auth_name","auth_srid","srtext","proj4text") VALUES (4326,'EPSG',4326,'GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.01745329251994328,AUTHORITY["EPSG","9122"]],AUTHORITY["EPSG","4326"]]','+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs ');
		c.addPostGIS('center2','POINT',2,SRID=4326);
		#--- EPSG 4326 : WGS 84
		#INSERT INTO "spatial_ref_sys" ("srid","auth_name","auth_srid","srtext","proj4text") VALUES (4326,'EPSG',4326,'GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.01745329251994328,AUTHORITY["EPSG","9122"]],AUTHORITY["EPSG","4326"]]','+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs ');
		c.addPostGIS('center3','POINT',2,SRID=4326);

	return c

def sqlInsertStr(params, outfile=sys.stdout, extraParams=None, dbType='postgres'):
	'''
	Return the SQL INSERT command for this message type
	@param params: dictionary of values keyed by field name
	@param outfile: file like object to print to.
	@param extraParams: A sequence of tuples containing (name,sql type) for additional fields
	@return: sql create string
	@rtype: str

	@see: sqlCreate
	'''
	outfile.write(str(sqlInsert(params,extraParams,dbType=dbType)))


def sqlInsert(params,extraParams=None,dbType='postgres'):
	'''
	Give the SQL INSERT statement
	@param params: dict keyed by field name of values
	@param extraParams: any extra fields that you have created beyond the normal ais message fields
	@rtype: sqlhelp.insert
	@return: insert class instance
	@todo: allow optional type checking of params?
	@warning: this will take invalid keys happily and do what???
	'''
	import sqlhelp
	i = sqlhelp.insert('whalenotice',dbType=dbType)

	if dbType=='postgres':
		finished = []
		for key in params:
			if key in finished: 
				continue

			if key not in toPgFields and key not in fromPgFields:
				if type(params[key])==Decimal: i.add(key,float(params[key]))
				else: i.add(key,params[key])
			else:
				if key in fromPgFields:
					val = params[key]
				        # Had better be a WKT type like POINT(-88.1 30.321)
					i.addPostGIS(key,val)
					finished.append(key)
				else:
					# Need to construct the type.
					pgName = toPgFields[key]
					#valStr='GeomFromText(\''+pgTypes[pgName]+'('
					valStr=pgTypes[pgName]+'('
					vals = []
					for nonPgKey in fromPgFields[pgName]:
						vals.append(str(params[nonPgKey]))
						finished.append(nonPgKey)
					valStr+=' '.join(vals)+')'
					i.addPostGIS(pgName,valStr)
	else:
		for key in params: 
			if type(params[key])==Decimal: i.add(key,float(params[key]))
			else: i.add(key,params[key])

	if None != extraParams:
		for key in extraParams: 
			i.add(key,extraParams[key])

	return i

######################################################################
# LATEX SUPPORT
######################################################################

def latexDefinitionTable(outfile=sys.stdout
		):
	'''
	Return the LaTeX definition table for this message type
	@param outfile: file like object to print to.
	@type outfile: file obj
	@return: LaTeX table string via the outfile
	@rtype: str

	'''
	o = outfile

	o.write('''
\\begin{table}%[htb]
\\centering
\\begin{tabular}{|l|c|l|}
\\hline
Parameter & Number of bits & Description 
\\\\  \\hline\\hline
MessageID & 6 & AIS message number.  Must be 8 \\\\ \hline 
RepeatIndicator & 2 & Indicated how many times a message has been repeated \\\\ \hline 
UserID & 30 & Unique ship identification number (MMSI) \\\\ \hline 
Spare & 2 & Reserved for definition by a regional authority. \\\\ \hline 
dac & 10 & Designated Area Code - 366 for the United States \\\\ \hline 
fid & 6 & Functional IDentifier - 63 for the Whale Notice \\\\ \hline 
efid & 12 & Extended Functional IDentifier.  1 for the Whale Notice (dac+fid+efid defines the exact message type) \\\\ \hline 
numreports & 2 & Number of detection reports filled out in this message \\\\ \hline 
stationid1 & 8 & Identifier of the station that recorded the whale.  Usually a number. \\\\ \hline 
time1\_day & 5 & Time of most recent whale detection.  UTC day of the month 1..31 \\\\ \hline 
time1\_hour & 5 & Time of most recent whale detection.  UTC hours 0..23 \\\\ \hline 
time1\_min & 6 & Time of most recent whale detection.  UTC minutes \\\\ \hline 
center1\_longitude & 28 & Center of the detection zone.  East West location \\\\ \hline 
center1\_latitude & 27 & Center of the detection zone.  North South location \\\\ \hline 
timetoexpire1 & 16 & Seconds from the detection time until the notice expires \\\\ \hline 
radius1 & 16 & Distance from center of detection zone (lat/lon above) \\\\ \hline 
stationid2 & 8 & Identifier of the station that recorded the whale.  Usually a number. \\\\ \hline 
time2\_day & 5 & Time of most recent whale detection.  UTC day of the month 1..31 \\\\ \hline 
time2\_hour & 5 & Time of most recent whale detection.  UTC hours 0..23 \\\\ \hline 
time2\_min & 6 & Time of most recent whale detection.  UTC minutes \\\\ \hline 
center2\_longitude & 28 & Center of the detection zone.  East West location \\\\ \hline 
center2\_latitude & 27 & Center of the detection zone.  North South location \\\\ \hline 
timetoexpire2 & 16 & Seconds from the detection time until the notice expires \\\\ \hline 
radius2 & 16 & Distance from center of detection zone (lat/lon above) \\\\ \hline 
stationid3 & 8 & Identifier of the station that recorded the whale.  Usually a number. \\\\ \hline 
time3\_day & 5 & Time of most recent whale detection.  UTC day of the month 1..31 \\\\ \hline 
time3\_hour & 5 & Time of most recent whale detection.  UTC hours 0..23 \\\\ \hline 
time3\_min & 6 & Time of most recent whale detection.  UTC minutes \\\\ \hline 
center3\_longitude & 28 & Center of the detection zone.  East West location \\\\ \hline 
center3\_latitude & 27 & Center of the detection zone.  North South location \\\\ \hline 
timetoexpire3 & 16 & Seconds from the detection time until the notice expires \\\\ \hline 
radius3 & 16 & Distance from center of detection zone (lat/lon above) \\\\ \hline 
Spare2 & 21 & Not used.  Should be set to zero.\\\\ \\hline \\hline
Total bits & 424 & Appears to take 2 slots \\\\ \\hline
\\end{tabular}
\\caption{AIS message number 8: Endangered whale notification binary message}
\\label{tab:whalenotice}
\\end{table}
''')

######################################################################
# Text Definition
######################################################################

def textDefinitionTable(outfile=sys.stdout
		,delim='\t'
		):
	'''
	Return the text definition table for this message type
	@param outfile: file like object to print to.
	@type outfile: file obj
	@return: text table string via the outfile
	@rtype: str

	'''
	o = outfile
	o.write('''Parameter'''+delim+'Number of bits'''+delim+'''Description 
MessageID'''+delim+'''6'''+delim+'''AIS message number.  Must be 8
RepeatIndicator'''+delim+'''2'''+delim+'''Indicated how many times a message has been repeated
UserID'''+delim+'''30'''+delim+'''Unique ship identification number (MMSI)
Spare'''+delim+'''2'''+delim+'''Reserved for definition by a regional authority.
dac'''+delim+'''10'''+delim+'''Designated Area Code - 366 for the United States
fid'''+delim+'''6'''+delim+'''Functional IDentifier - 63 for the Whale Notice
efid'''+delim+'''12'''+delim+'''Extended Functional IDentifier.  1 for the Whale Notice (dac+fid+efid defines the exact message type)
numreports'''+delim+'''2'''+delim+'''Number of detection reports filled out in this message
stationid1'''+delim+'''8'''+delim+'''Identifier of the station that recorded the whale.  Usually a number.
time1_day'''+delim+'''5'''+delim+'''Time of most recent whale detection.  UTC day of the month 1..31
time1_hour'''+delim+'''5'''+delim+'''Time of most recent whale detection.  UTC hours 0..23
time1_min'''+delim+'''6'''+delim+'''Time of most recent whale detection.  UTC minutes
center1_longitude'''+delim+'''28'''+delim+'''Center of the detection zone.  East West location
center1_latitude'''+delim+'''27'''+delim+'''Center of the detection zone.  North South location
timetoexpire1'''+delim+'''16'''+delim+'''Seconds from the detection time until the notice expires
radius1'''+delim+'''16'''+delim+'''Distance from center of detection zone (lat/lon above)
stationid2'''+delim+'''8'''+delim+'''Identifier of the station that recorded the whale.  Usually a number.
time2_day'''+delim+'''5'''+delim+'''Time of most recent whale detection.  UTC day of the month 1..31
time2_hour'''+delim+'''5'''+delim+'''Time of most recent whale detection.  UTC hours 0..23
time2_min'''+delim+'''6'''+delim+'''Time of most recent whale detection.  UTC minutes
center2_longitude'''+delim+'''28'''+delim+'''Center of the detection zone.  East West location
center2_latitude'''+delim+'''27'''+delim+'''Center of the detection zone.  North South location
timetoexpire2'''+delim+'''16'''+delim+'''Seconds from the detection time until the notice expires
radius2'''+delim+'''16'''+delim+'''Distance from center of detection zone (lat/lon above)
stationid3'''+delim+'''8'''+delim+'''Identifier of the station that recorded the whale.  Usually a number.
time3_day'''+delim+'''5'''+delim+'''Time of most recent whale detection.  UTC day of the month 1..31
time3_hour'''+delim+'''5'''+delim+'''Time of most recent whale detection.  UTC hours 0..23
time3_min'''+delim+'''6'''+delim+'''Time of most recent whale detection.  UTC minutes
center3_longitude'''+delim+'''28'''+delim+'''Center of the detection zone.  East West location
center3_latitude'''+delim+'''27'''+delim+'''Center of the detection zone.  North South location
timetoexpire3'''+delim+'''16'''+delim+'''Seconds from the detection time until the notice expires
radius3'''+delim+'''16'''+delim+'''Distance from center of detection zone (lat/lon above)
Spare2'''+delim+'''21'''+delim+'''Not used.  Should be set to zero.
Total bits'''+delim+'''424'''+delim+'''Appears to take 2 slots''')


######################################################################
# UNIT TESTING
######################################################################
import unittest
def testParams():
	'''Return a params file base on the testvalue tags.
	@rtype: dict
	@return: params based on testvalue tags
	'''
	params = {}
	params['MessageID'] = 8
	params['RepeatIndicator'] = 1
	params['UserID'] = 1193046
	params['Spare'] = 0
	params['dac'] = 366
	params['fid'] = 63
	params['efid'] = 1
	params['numreports'] = 3
	params['stationid1'] = 76
	params['time1_day'] = 28
	params['time1_hour'] = 23
	params['time1_min'] = 45
	params['center1_longitude'] = Decimal('-122.16328055555556')
	params['center1_latitude'] = Decimal('37.424458333333334')
	params['timetoexpire1'] = 1
	params['radius1'] = 5000
	params['stationid2'] = 76
	params['time2_day'] = 28
	params['time2_hour'] = 23
	params['time2_min'] = 45
	params['center2_longitude'] = Decimal('-122.16328055555556')
	params['center2_latitude'] = Decimal('37.424458333333334')
	params['timetoexpire2'] = 1
	params['radius2'] = 5000
	params['stationid3'] = 76
	params['time3_day'] = 28
	params['time3_hour'] = 23
	params['time3_min'] = 45
	params['center3_longitude'] = Decimal('-122.16328055555556')
	params['center3_latitude'] = Decimal('37.424458333333334')
	params['timetoexpire3'] = 1
	params['radius3'] = 5000
	params['Spare2'] = 0

	return params

class Testwhalenotice(unittest.TestCase):
	'''Use testvalue tag text from each type to build test case the whalenotice message'''
	def testEncodeDecode(self):

		params = testParams()
		bits   = encode(params)
		r      = decode(bits)

		# Check that each parameter came through ok.
		self.failUnlessEqual(r['MessageID'],params['MessageID'])
		self.failUnlessEqual(r['RepeatIndicator'],params['RepeatIndicator'])
		self.failUnlessEqual(r['UserID'],params['UserID'])
		self.failUnlessEqual(r['Spare'],params['Spare'])
		self.failUnlessEqual(r['dac'],params['dac'])
		self.failUnlessEqual(r['fid'],params['fid'])
		self.failUnlessEqual(r['efid'],params['efid'])
		self.failUnlessEqual(r['numreports'],params['numreports'])
		self.failUnlessEqual(r['stationid1'],params['stationid1'])
		self.failUnlessEqual(r['time1_day'],params['time1_day'])
		self.failUnlessEqual(r['time1_hour'],params['time1_hour'])
		self.failUnlessEqual(r['time1_min'],params['time1_min'])
		self.failUnlessAlmostEqual(r['center1_longitude'],params['center1_longitude'],5)
		self.failUnlessAlmostEqual(r['center1_latitude'],params['center1_latitude'],5)
		self.failUnlessEqual(r['timetoexpire1'],params['timetoexpire1'])
		self.failUnlessEqual(r['radius1'],params['radius1'])
		self.failUnlessEqual(r['stationid2'],params['stationid2'])
		self.failUnlessEqual(r['time2_day'],params['time2_day'])
		self.failUnlessEqual(r['time2_hour'],params['time2_hour'])
		self.failUnlessEqual(r['time2_min'],params['time2_min'])
		self.failUnlessAlmostEqual(r['center2_longitude'],params['center2_longitude'],5)
		self.failUnlessAlmostEqual(r['center2_latitude'],params['center2_latitude'],5)
		self.failUnlessEqual(r['timetoexpire2'],params['timetoexpire2'])
		self.failUnlessEqual(r['radius2'],params['radius2'])
		self.failUnlessEqual(r['stationid3'],params['stationid3'])
		self.failUnlessEqual(r['time3_day'],params['time3_day'])
		self.failUnlessEqual(r['time3_hour'],params['time3_hour'])
		self.failUnlessEqual(r['time3_min'],params['time3_min'])
		self.failUnlessAlmostEqual(r['center3_longitude'],params['center3_longitude'],5)
		self.failUnlessAlmostEqual(r['center3_latitude'],params['center3_latitude'],5)
		self.failUnlessEqual(r['timetoexpire3'],params['timetoexpire3'])
		self.failUnlessEqual(r['radius3'],params['radius3'])
		self.failUnlessEqual(r['Spare2'],params['Spare2'])

def addMsgOptions(parser):
	parser.add_option('-d','--decode',dest='doDecode',default=False,action='store_true',
		help='decode a "whalenotice" AIS message')
	parser.add_option('-e','--encode',dest='doEncode',default=False,action='store_true',
		help='encode a "whalenotice" AIS message')
	parser.add_option('--RepeatIndicator-field', dest='RepeatIndicatorField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--UserID-field', dest='UserIDField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--numreports-field', dest='numreportsField',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--stationid1-field', dest='stationid1Field',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time1_day-field', dest='time1_dayField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time1_hour-field', dest='time1_hourField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time1_min-field', dest='time1_minField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--center1_longitude-field', dest='center1_longitudeField',default=Decimal('181'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--center1_latitude-field', dest='center1_latitudeField',default=Decimal('91'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--timetoexpire1-field', dest='timetoexpire1Field',default=65535,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--radius1-field', dest='radius1Field',default=65534,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--stationid2-field', dest='stationid2Field',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time2_day-field', dest='time2_dayField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time2_hour-field', dest='time2_hourField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time2_min-field', dest='time2_minField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--center2_longitude-field', dest='center2_longitudeField',default=Decimal('181'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--center2_latitude-field', dest='center2_latitudeField',default=Decimal('91'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--timetoexpire2-field', dest='timetoexpire2Field',default=65535,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--radius2-field', dest='radius2Field',default=65534,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--stationid3-field', dest='stationid3Field',default=0,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time3_day-field', dest='time3_dayField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time3_hour-field', dest='time3_hourField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--time3_min-field', dest='time3_minField',metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--center3_longitude-field', dest='center3_longitudeField',default=Decimal('181'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--center3_latitude-field', dest='center3_latitudeField',default=Decimal('91'),metavar='decimal',type='string'
		,help='Field parameter value [default: %default]')
	parser.add_option('--timetoexpire3-field', dest='timetoexpire3Field',default=65535,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')
	parser.add_option('--radius3-field', dest='radius3Field',default=65534,metavar='uint',type='int'
		,help='Field parameter value [default: %default]')

def main():
	from optparse import OptionParser
	parser = OptionParser(usage="%prog [options]",
		version="%prog "+__version__)

	parser.add_option('--doc-test',dest='doctest',default=False,action='store_true',
		help='run the documentation tests')
	parser.add_option('--unit-test',dest='unittest',default=False,action='store_true',
		help='run the unit tests')
	parser.add_option('-v','--verbose',dest='verbose',default=False,action='store_true',
		help='Make the test output verbose')

	# FIX: remove nmea from binary messages.  No way to build the whole packet?
	# FIX: or build the surrounding msg 8 for a broadcast?
	typeChoices = ('binary','nmeapayload','nmea') # FIX: what about a USCG type message?
	parser.add_option('-t','--type',choices=typeChoices,type='choice',dest='ioType'
		,default='nmeapayload'
		,help='What kind of string to write for encoding ('+', '.join(typeChoices)+') [default: %default]')


	outputChoices = ('std','html','csv','sql' , 'kml','kml-full')
	parser.add_option('-T','--output-type',choices=outputChoices,type='choice',dest='outputType'
		,default='std'
		,help='What kind of string to output ('+', '.join(outputChoices)+') [default: %default]')

	parser.add_option('-o','--output',dest='outputFileName',default=None,
			  help='Name of the python file to write [default: stdout]')

	parser.add_option('-f','--fields',dest='fieldList',default=None, action='append',
			  choices=fieldList,
			  help='Which fields to include in the output.  Currently only for csv output [default: all]')

	parser.add_option('-p','--print-csv-field-list',dest='printCsvfieldList',default=False,action='store_true',
			  help='Print the field name for csv')

	parser.add_option('-c','--sql-create',dest='sqlCreate',default=False,action='store_true',
			  help='Print out an sql create command for the table.')

	parser.add_option('--latex-table',dest='latexDefinitionTable',default=False,action='store_true',
			  help='Print a LaTeX table of the type')

	parser.add_option('--text-table',dest='textDefinitionTable',default=False,action='store_true',
			  help='Print delimited table of the type (for Word table importing)')
	parser.add_option('--delimt-text-table',dest='delimTextDefinitionTable',default='\t'
			  ,help='Delimiter for text table [default: \'%default\'](for Word table importing)')


	dbChoices = ('sqlite','postgres')
	parser.add_option('-D','--db-type',dest='dbType',default='postgres'
			  ,choices=dbChoices,type='choice'
			  ,help='What kind of database ('+', '.join(dbChoices)+') [default: %default]')

	addMsgOptions(parser)

	(options,args) = parser.parse_args()
	success=True

	if options.doctest:
		import os; print os.path.basename(sys.argv[0]), 'doctests ...',
		sys.argv= [sys.argv[0]]
		if options.verbose: sys.argv.append('-v')
		import doctest
		numfail,numtests=doctest.testmod()
		if numfail==0: print 'ok'
		else: 
			print 'FAILED'
			success=False

	if not success: sys.exit('Something Failed')
	del success # Hide success from epydoc

	if options.unittest:
		sys.argv = [sys.argv[0]]
		if options.verbose: sys.argv.append('-v')
		unittest.main()

	outfile = sys.stdout
	if None!=options.outputFileName:
		outfile = file(options.outputFileName,'w')


	if options.doEncode:
		# First make sure all non required options are specified
		if None==options.RepeatIndicatorField: parser.error("missing value for RepeatIndicatorField")
		if None==options.UserIDField: parser.error("missing value for UserIDField")
		if None==options.numreportsField: parser.error("missing value for numreportsField")
		if None==options.stationid1Field: parser.error("missing value for stationid1Field")
		if None==options.time1_dayField: parser.error("missing value for time1_dayField")
		if None==options.time1_hourField: parser.error("missing value for time1_hourField")
		if None==options.time1_minField: parser.error("missing value for time1_minField")
		if None==options.center1_longitudeField: parser.error("missing value for center1_longitudeField")
		if None==options.center1_latitudeField: parser.error("missing value for center1_latitudeField")
		if None==options.timetoexpire1Field: parser.error("missing value for timetoexpire1Field")
		if None==options.radius1Field: parser.error("missing value for radius1Field")
		if None==options.stationid2Field: parser.error("missing value for stationid2Field")
		if None==options.time2_dayField: parser.error("missing value for time2_dayField")
		if None==options.time2_hourField: parser.error("missing value for time2_hourField")
		if None==options.time2_minField: parser.error("missing value for time2_minField")
		if None==options.center2_longitudeField: parser.error("missing value for center2_longitudeField")
		if None==options.center2_latitudeField: parser.error("missing value for center2_latitudeField")
		if None==options.timetoexpire2Field: parser.error("missing value for timetoexpire2Field")
		if None==options.radius2Field: parser.error("missing value for radius2Field")
		if None==options.stationid3Field: parser.error("missing value for stationid3Field")
		if None==options.time3_dayField: parser.error("missing value for time3_dayField")
		if None==options.time3_hourField: parser.error("missing value for time3_hourField")
		if None==options.time3_minField: parser.error("missing value for time3_minField")
		if None==options.center3_longitudeField: parser.error("missing value for center3_longitudeField")
		if None==options.center3_latitudeField: parser.error("missing value for center3_latitudeField")
		if None==options.timetoexpire3Field: parser.error("missing value for timetoexpire3Field")
		if None==options.radius3Field: parser.error("missing value for radius3Field")
		msgDict={
			'MessageID': '8',
			'RepeatIndicator': options.RepeatIndicatorField,
			'UserID': options.UserIDField,
			'Spare': '0',
			'dac': '366',
			'fid': '63',
			'efid': '1',
			'numreports': options.numreportsField,
			'stationid1': options.stationid1Field,
			'time1_day': options.time1_dayField,
			'time1_hour': options.time1_hourField,
			'time1_min': options.time1_minField,
			'center1_longitude': options.center1_longitudeField,
			'center1_latitude': options.center1_latitudeField,
			'timetoexpire1': options.timetoexpire1Field,
			'radius1': options.radius1Field,
			'stationid2': options.stationid2Field,
			'time2_day': options.time2_dayField,
			'time2_hour': options.time2_hourField,
			'time2_min': options.time2_minField,
			'center2_longitude': options.center2_longitudeField,
			'center2_latitude': options.center2_latitudeField,
			'timetoexpire2': options.timetoexpire2Field,
			'radius2': options.radius2Field,
			'stationid3': options.stationid3Field,
			'time3_day': options.time3_dayField,
			'time3_hour': options.time3_hourField,
			'time3_min': options.time3_minField,
			'center3_longitude': options.center3_longitudeField,
			'center3_latitude': options.center3_latitudeField,
			'timetoexpire3': options.timetoexpire3Field,
			'radius3': options.radius3Field,
			'Spare2': '0',
		}

		bits = encode(msgDict)
		if 'binary'==options.ioType: print str(bits)
		elif 'nmeapayload'==options.ioType:
		    # FIX: figure out if this might be necessary at compile time
		    #print "bitLen",len(bits)
		    bitLen=len(bits)
		    if bitLen%6!=0:
			bits = bits + BitVector(size=(6 - (bitLen%6)))  # Pad out to multiple of 6
		    #print "result:",binary.bitvectoais6(bits)[0]
		    print binary.bitvectoais6(bits)[0]


		# FIX: Do not emit this option for the binary message payloads.  Does not make sense.
		elif 'nmea'==options.ioType: 
		    #bitLen=len(bits)
                    #if bitLen%6!=0:
		    #	bits = bits + BitVector(size=(6 - (bitLen%6)))  # Pad out to multiple of 6
                    import aisutils.uscg as uscg
                    nmea = uscg.create_nmea(bits)
                    print nmea
                    #
                    #


                    #sys.exit("FIX: need to implement creating nmea capability")
		else: sys.exit('ERROR: unknown ioType.  Help!')


	if options.sqlCreate:
		sqlCreateStr(outfile,options.fieldList,dbType=options.dbType)

	if options.latexDefinitionTable:
		latexDefinitionTable(outfile)

	# For conversion to word tables
	if options.textDefinitionTable:
		textDefinitionTable(outfile,options.delimTextDefinitionTable)

	if options.printCsvfieldList:
		# Make a csv separated list of fields that will be displayed for csv
		if None == options.fieldList: options.fieldList = fieldList
		import StringIO
		buf = StringIO.StringIO()
		for field in options.fieldList:
			buf.write(field+',')
		result = buf.getvalue()
		if result[-1] == ',': print result[:-1]
		else: print result

	if options.doDecode:
		if len(args)==0: args = sys.stdin
		for msg in args:
			bv = None

			if msg[0] in ('$','!') and msg[3:6] in ('VDM','VDO'):
				# Found nmea
				# FIX: do checksum
				bv = binary.ais6tobitvec(msg.split(',')[5])
			else: # either binary or nmeapayload... expect mostly nmeapayloads
				# assumes that an all 0 and 1 string can not be a nmeapayload
				binaryMsg=True
				for c in msg:
					if c not in ('0','1'):
						binaryMsg=False
						break
				if binaryMsg:
					bv = BitVector(bitstring=msg)
				else: # nmeapayload
					bv = binary.ais6tobitvec(msg)

			printFields(decode(bv)
				    ,out=outfile
				    ,format=options.outputType
				    ,fieldList=options.fieldList
				    ,dbType=options.dbType
				    )

############################################################
if __name__=='__main__':
    main()
