from odict import odict
from zope.interface import implements
from zope.interface.common.mapping import IEnumerableMapping, IFullMapping
from interfaces import IAttributeAccess


class Unset(object):
    """Used to identify unset values in contrast to None

    use for example by node.parts.nodify.Nodify
    
    XXX: instanciate directly here??
    XXX: yafowil contains another unset object, use the one from node
    XXX: call me UNSET as constant?
         UNSET = object()
    """


def LocationIterator(object):
    """Iterate over an object and all of its parents.

    Copied from ``zope.location.LocationIterator``.

    """
    while object is not None:
        yield object
        object = getattr(object, '__parent__', None)


class Zodict(odict):
    """Mark ordered dict with corresponding interface.
    
    XXX: I think we might not need this any longer.
    """
    implements(IFullMapping)
    
    def __init__(self, data=()):
        odict.__init__(self, data=data)


class ReverseMapping(object):
    """Reversed IEnumerableMapping.
    """
    
    implements(IEnumerableMapping)
    
    def __init__(self, context):
        """Object behaves as adapter for dict like object.
        
        ``context``: a dict like object.
        """
        self.context = context
    
    def __getitem__(self, value):
        for key in self.context:
            if self.context[key] == value:
                return key
        raise KeyError(value)
    
    def get(self, value, default=None):
        try:
            return self[value]
        except KeyError:
            return default

    def __contains__(self, value):
        for key in self.context:
            val = self.context[key]
            if val == value:
                return True
        return False
    
    def keys(self):
        return [val for val in self]

    def __iter__(self):
        for key in self.context:
            yield self.context[key]

    def values(self):
        return [key for key in self.context]

    def items(self):
        return [(v, k) for k, v in self.context.items()]

    def __len__(self):
        return len(self.context)


class AttributeAccess(object):
    """If someone really needs to access the original context (which should not 
    happen), she hast to use ``object.__getattr__(attraccess, 'context')``.
    """
    
    implements(IAttributeAccess)
    
    def __init__(self, context):
        object.__setattr__(self, 'context', context)
    
    def __getattr__(self, name):
        context = object.__getattribute__(self, 'context')
        try:
            return context[name]
        except KeyError:
            raise AttributeError(name)
    
    def __setattr__(self, name, value):
        context = object.__getattribute__(self, 'context')
        context[name] = value
    
    def __getitem__(self, name):
        context = object.__getattribute__(self, 'context')
        return context[name]
    
    def __setitem__(self, name, value):
        context = object.__getattribute__(self, 'context')
        context[name] = value
    
    def __delitem__(self, name):
        context = object.__getattribute__(self, 'context')
        del context[name]


CHARACTER_ENCODING='utf-8'

class StrCodec(object):
    """Encode unicodes to strs and decode strs to unicodes

    We will recursively work on arbitrarily nested structures consisting of
    str, unicode, list, tuple and dict mixed with others, which we won't touch.
    During that process a deep copy is produces leaving the orginal data
    structure intact.
    """
    def __init__(self, encoding=CHARACTER_ENCODING, soft=True):
        """@param encoding: character encoding to decode from/encode to
           @param soft: if True, catch UnicodeDecodeErrors and leave this
           strings as-is.
        """
        self._encoding = encoding
        self._soft = soft

    def encode(self, arg):
        """Return an encoded copy of the argument.
        
        - Strings are decoded and re-encoded to make sure they conform to the
          encoding
        - Unicodees are encoded as string according to encoding
        - Lists/tuples/dicts are recursively worked on
        - Everything else is left untouched
        """
        if isinstance(arg, (list, tuple)):
            arg = arg.__class__(map(self.encode, arg))
        elif isinstance(arg, dict):
            arg = dict([self.encode(t) for t in arg.iteritems()])
        elif isinstance(arg, str):
            arg = self.encode(self.decode(arg))
        elif isinstance(arg, unicode):
            arg = arg.encode(self._encoding)
        return arg

    def decode(self, arg):
        if isinstance(arg, (list, tuple)):
            arg = arg.__class__(map(self.decode, arg))
        elif isinstance(arg, dict):
            arg = dict([self.decode(t) for t in arg.iteritems()])
        elif isinstance(arg, str):
            try:
                arg = arg.decode(self._encoding)
            except UnicodeDecodeError:
                # in soft mode we leave the string, otherwise we raise the
                # exception
                if not self._soft:
                    raise
        return arg

strcodec = StrCodec()
encode = strcodec.encode
decode = strcodec.decode


def instance_property(func):
    """Decorator like ``property``, but underlying function is only called once
    per instance. 
    
    Set instance attribute with '_' prefix. 
    """
    def wrapper(self):
        attribute_name = '_%s' % func.__name__
        if not hasattr(self, attribute_name):
            setattr(self, attribute_name, func(self))
        return getattr(self, attribute_name)
    wrapper.__doc__ = func.__doc__
    return property(wrapper)