import logging
import keyring
import os
import pickle
import sys
import time

from .ui_helpers import _options, app_header, format_msg
from .utils import (get_correct_answer, path_fixer, SetupConfig,
                    install_check, ask_yes_no, password_hasher)

from not_so_tuf import NotSoTuf
from not_so_tuf.package_handler import PackageHandler
from not_so_tuf.key_handler import KeyHandler


log = logging.getLogger()
log.setLevel(logging.DEBUG)
fh_formatter = logging.Formatter('%(asctime)s - %(name)-26s -  %(levelname)-8s -> %(message)s')
ch_formatter = logging.Formatter('%(name)-10s: %(levelname)-8s -> %(message)s')
fh = logging.FileHandler('nst.log', mode='wa')
fh.setFormatter(fh_formatter)

ch = logging.StreamHandler()
fh.setLevel(logging.DEBUG)
ch.setLevel(logging.INFO)
ch.setFormatter(ch_formatter)
fh.setFormatter(fh_formatter)
log.addHandler(ch)
log.addHandler(fh)

class Ui(object):

    def __init__(self):
        if sys.platform == 'win32':
            self._clear = 'cls'
        else:
            self._clear = 'clear'
        try:
            self.config = self.load_config()
        except:
            self.config = SetupConfig()
        self.nst = NotSoTuf(__name__)

    def main(self):
        self.cwd = os.getcwd()
        path = os.path.join(self.cwd, 'config.data')
        if not os.path.exists(path):
            self.initial_setup()
        options = [('Sign Updates', self.sign_updates), ('Keys', self.keys),
                   ('Update Settings', self.settings),
                   ('Quit', self.quit)]
        print app_header()
        next = _options(options)
        return next()

    def initial_setup(self):
        print app_header('Initial Setup')
        msg = ('You should only have to run this once.')
        print format_msg(msg)
        options = [('Begin Setup', self.start_intial_setup),
                   ('Go Back', self.main)]
        next = _options(options)
        return next()

    def start_intial_setup(self):
        print app_header('Setup Assistant')
        print format_msg('Let\'s begin...')

        self.config.APP_NAME = get_correct_answer('Please enter app name',
                                                  default='Not-So-TUF')
        dev_data_dir = get_correct_answer('Enter directory to store'
                                          ' work files',
                                          default='Current Working Directory')
        self.config.DEV_DATA_DIR = path_fixer(dev_data_dir)
        self.config.KEY_LENGTH = get_correct_answer('Enter a key length. '
                                                    'Longer is more secure '
                                                    'but takes longer to '
                                                    'compute!', default='2048')
        priv_key_name = get_correct_answer('Enter a name to give '
                                           'your private key',
                                           default='Not-So-TUF')
        self.config.add_private_key_name(priv_key_name)

        pub_key_name = get_correct_answer('Enter a name to give your'
                                          ' public key',
                                          default='Not-So-TUF')
        self.config.add_public_key_name(pub_key_name)

        self.config.UPDATE_URL = get_correct_answer('Enter your update url',
                                                    required=True)

        self.save_config(self.config)
        package_handler = PackageHandler(self.nst)
        package_handler.setup()
        print 'Making keys...'
        self.make_keys()
        print app_header('Setup Complete')
        print format_msg('Now let\'s update some apps')
        time.sleep(3)

    def settings(self):
        print app_header('Update Settings')
        msg = ('Enter the number of each item you\'d like to update.')
        print format_msg(msg)
        print format_msg('Examples:')
        print format_msg('-->13 ')
        print format_msg('-->23 ')
        print format_msg('-->123')
        print format_msg('Then press Enter')
        print '1. App Name\n2. Dev Data Dir\n3. Update Url'
        answers = raw_input('-->')

        if '1' in answers:
            app_name = get_correct_answer('Enter APP NAME',
                                          default=self.config.APP_NAME)
            self.config.APP_NAME = app_name
        if '2' in answers:
            data_dir = get_correct_answer('Enter directory to store work '
                                          'files',
                                          default=self.config.DEV_DATA_DIR)
            self.config.DEV_DATA_DIR = data_dir
        if '3' in answers:
            url = get_correct_answer('Enter new update url',
                                     default=self.config.UPDATE_URL)
            self.config.UPDATE_URL = url
        self.save_config(self.config)
        print 'Saving new config....'
        time.sleep(3)
        return self.main()

    def save_config(self, obj):
        with open('config.data', 'w') as f:
            f.write(pickle.dumps(obj))
        self.config = self.load_config()
        self.nst.update_config(self.config)

    def load_config(self):
        with open('config.data', 'r') as f:
            return pickle.loads(f.read())

    def sign_updates(self):
        print app_header('Sign Updates')
        msg = 'Make sure updates are in "new" folder then press "1" to start'
        print format_msg(msg)
        options = [('Start', self.start_sign_updates),
                   ('Go Back', self.main)]
        next = _options(options)
        return next()

    def start_sign_updates(self):
        package_handler = PackageHandler(self.nst)
        package_handler.setup()
        package_handler.update_package_list()
        key_handler = KeyHandler(self.nst)
        key_handler.sign_update()
        package_handler.deploy()
        print 'Update signing complete...'
        time.sleep(3)
        self.main()

    def keys(self):
        options = [('Show Public Key', self.show_public_key),
                   ('Go Back', self.main)]
                   # ('Replace Keys', self.main), ('Go Back', self.main)]
        print app_header('Keys')
        next = _options(options)
        return next()

    def show_public_key(self):
        key_handler = KeyHandler(self.nst)
        key_handler.print_public_key()
        raw_input('Press enter to continue')
        self.keys()

    def replace_keys(self):
        key_handler = KeyHandler(self.nst)
        key_handler.make_keys(overwrite=True)

    def make_keys(self):
        key_handler = KeyHandler(self.nst)
        key_handler.make_keys()

    def quit(self):
        print 'See Ya!'
        sys.exit()

    def start(self):
        # self.login()
        self.main()

    def login(self):
        while 1:
            print app_header('Login')
            if install_check() is False:
                ans = ask_yes_no('Confirm Initial Setup', default='no')
                if ans is False:
                    print "Seem like someone tampered with your computer\n\n"
                    raw_input('Press enter to continue')
                    continue
                print app_header('Create Account')
                username = get_correct_answer('Enter a username',
                                              required=True)
                password = get_correct_answer('Enter a password',
                                              required=True, password=True,
                                              create_account=True)

                password_check = keyring.get_password('Not So TUF', username)
                if password_check is None:
                    hashed_password = password_hasher(password)
                    del password
                    keyring.set_password('Not So TUF', username, hashed_password)
                    del hashed_password
            else:
                print app_header('Login')
                while 1:
                    username = get_correct_answer('Username', required=True)
                    keychain_password = keyring.get_password('Not So TUF',
                                                             username)
                    if keychain_password is None:
                        print 'Not a valid user\n\n'
                        raw_input('Press enter to continue')
                        continue
                    break
                while 1:
                    password = get_correct_answer('Password', required=True,
                                                  password=True)
                    if password_hasher(password) != keychain_password:
                        print 'Invalid Password\n\n'
                        raw_input('Press enter to continue')
                        continue
                    break
                break


def main():
    ui = Ui()
    ui.start()
