import logging
import os
import sys
import traceback

logger = logging.getLogger(__name__)

class NotSoTuf(object):
    """There are 2 ways to load config.  The first was is during
    object initialization. The second way is later with :meth:`update_config`

    Examples are shown below::

        DEBUG = True
        SECRET_KEY = 'development key'
        app.config.from_object(__name__)
        Config(object):
            APP_NAME = "NST"

            APP_DATA_DIR = None

            UPDATE_URL = http://www.test-nst.com/updates

        app = NotSoTuf(Config())

        app = NotSoTuf()
        app.update_config(Config())

    Kwargs:
        import_name (str): used to get current directory

        cfg_obj (instance): object with config attributes
    """
    def __init__(self, import_name=None, cfg_obj=None):
        if import_name is None:
            raise UpdaterError(u'You have to pass __name__ to Updater',
                               expected=True)
        self.import_name = import_name
        self.real_path = os.path.dirname(os.path.abspath(self.import_name))
        self.config = Config()
        self.config['APP_DIR'] = self.real_path
        if cfg_obj:
            self.update_config(cfg_obj)

    def update_config(self, obj):
        """Proxy method to update internal config dict

        Args:
            obj (instance): config object
        """
        self.config.from_object(obj)
        if self.config.get('DEV_DATA_DIR', None) is None:
            logger.info('Using default directory')
            self.config['DEV_DATA_DIR'] = os.path.join(self.real_path,
                                                       'data')
        if self.config.get('APP_NAME', None) is None:
            self.config['APP_NAME'] = 'Not_So_TUF'


class Config(dict):
    """Works exactly like a dict but provides ways to fill it from files
    or special dictionaries.  There are two common patterns to populate the
    config.

    You can define the configuration options in the
    module that calls :meth:`from_object`.  It is also possible to tell it
    to use the same module and with that provide the configuration values
    just before the call::

    Loading from modules, only uppercase keys are added to the config.
    This makes it possible to use lowercase values in the config file for
    temporary values that are not added to the config or to define the config
    keys in the same file that implements the application.
    """

    def __init__(self, defaults=None):
        super(Config, self).__init__(defaults or {})

    def from_object(self, obj):
        """Updates the values from the given object.:

        -   an actual object reference: that object is used directly

        Args:
            obj (instance): Object with config attributes

        Objects are classes.

        Just the uppercase variables in that object are stored in the config.
        Example usage::

            from yourapplication import default_config
            app.config.from_object(default_config)
        """
        for key in dir(obj):
            if key.isupper():
                self[key] = getattr(obj, key)

    def __repr__(self):
        return '<%s %s>' % (self.__class__.__name__, dict.__repr__(self))


class STDError(Exception):
    def __init__(self, msg, tb=None, expected=False, cause=None):
        """ tb, if given, is the original traceback *so that it can be
        printed out*.

        If expected is set, this is a normal error message and most
        likely not a bug in Not_So_TUF.
        """
        if not expected:
            msg = msg + (u'; please report this issue on https://github.com'
                         '/JohnyMoSwag/Not-So-TUF/issues')
        super(Exception, self).__init__(msg)

        self.traceback = tb
        self.exc_info = sys.exc_info()  # preserve original exception
        self.cause = cause

    def format_traceback(self):
        if self.traceback is None:
            return None
        return u''.join(traceback.format_tb(self.traceback))


class ClientException(STDError):
    """Raised for Client exceptions"""
    def __init__(self, *args, **kwargs):
        super(CleintException, self).__init__(*args, **kwargs)


class ConfigException(STDError):
    """Raised for Config exceptions"""
    def __init__(self, *args, **kwargs):
        super(ConfigException, self).__init__(*args, **kwargs)


class FileDownloaderError(STDError):
    """Raised for File Downloader exceptions"""
    def __init__(self, *args, **kwargs):
        super(PackageHandlerError, self).__init__(*args, **kwargs)


class KeyHandlerError(STDError):
    """Raised for Key Handler exceptions"""
    def __init__(self, *args, **kwargs):
        super(KeyHandlerError, self).__init__(*args, **kwargs)


class PackageHandlerError(STDError):
    """Raised for Package Handler exceptions"""
    def __init__(self, *args, **kwargs):
        super(PackageHandlerError, self).__init__(*args, **kwargs)


class UpdaterError(STDError):
    """Raised for Updater exceptions"""
    def __init__(self, *args, **kwargs):
        super(UpdaterError, self).__init__(*args, **kwargs)