#!/usr/bin/env python
# -*- coding: utf-8 -*-

import hashlib
import time
import urllib2
import os
import logging
from io import BytesIO

log = logging.getLogger(__name__)


class FileDownloader(object):
    """The FileDownloader object downloads files to memory and
    verifies their hash.  If hash is verified data is either
    written to disk to returned to calling object

    Args:

        filename (str): The name of file to download

        url (str): The url to retrieve the file

        hexdigest str(str): The md5 checksum of the file to download
    """
    def __init__(self, filename, url, hexdigest):
        self.filename = filename
        self.url = url
        self.hexdigest = hexdigest
        self.block_size = 4096 * 4
        self.file_binary_data = None

    def download_verify_write(self):
        """Downloads file then verifies against provided hash
        If hash verfies then writes data to disk

        Returns:
            (bool) Meanings::

                True -- File hash verified

                False -- File hash not verified
        """
        # Downloads and verifies file against
        # provided hash.
        self._download_to_memory()
        check = self._check_hash()
        if check:
            log.info('File hash verified')
            self._write_to_file()
            return True
        else:
            log.error('Cannot verify file hash')
            del self.file_binary_data
            del self.my_file
            return False

    def download_verify_return(self):
        """
        Downloads file to memory, checks against provided hash
        If matched returns binary data

        Returns:
            (data) Meanings::
                data -- If everything verified

                None -- If any check didn't pass
        """
        self._download_to_memory()
        check = self._check_hash()
        if check:
            log.info('File hash verified')
            return self.file_binary_data
        else:
            log.error('Cannot verify file hash')
            return None

    @staticmethod
    def _best_block_size(elapsed_time, bytes):
        """Returns best block size for current internet connection speed

        Returns:
            (int) Size of chuck to read
        """
        new_min = max(bytes / 2.0, 1.0)
        new_max = min(max(bytes * 2.0, 1.0), 4194304)  # Do not surpass 4 MB
        if elapsed_time < 0.001:
            return int(new_max)
        rate = bytes / elapsed_time
        if rate > new_max:
            return int(new_max)
        if rate < new_min:
            return int(new_min)
        return int(rate)

    def _download_to_memory(self):
        """Downloads file to memory.  Keeps internal reference
        """
        try:
            data = urllib2.urlopen(self.url)
        except Exception as e:
            log.error(str(e))
            return None
        self.my_file = BytesIO()
        log.debug('Downloading {} from \n{}'.format(self.filename, self.url))
        while 1:
            start_block = time.time()
            file_data = data.read(self.block_size)
            end_block = time.time()
            if not file_data:
                self.my_file.flush()
                self.my_file.seek(0)
                self.file_binary_data = self.my_file.read()
                log.info('Download Complete')
                break

            self.my_file.write(file_data)
            self.block_size = self._best_block_size(end_block - start_block,
                                                    len(file_data))
            log.debug('Block size: %s' % self.block_size)

    def _write_to_file(self):
        """Writes download data to disk
        """
        # Writes file to disk
        # TODO: Maybe add more verification here
        # Suggestions always welcome
        with open(self.filename, 'wb') as f:
            f.write(self.file_binary_data)

    def _check_hash(self):
        """Checks hash of downloaded file.

        Returns:
            (bool) Meanings::

                True -- If file hash matches

                False -- If file hash mismatch
        """
        # Simple hash returning function
        if self.file_binary_data is None:
            return False
        log.info('Checking file hash')
        log.debug('Update hash: {}'.format(self.hexdigest))
        file_hash = hashlib.md5(self.file_binary_data).hexdigest()
        log.debug('File hash: {}'.format(file_hash))
        if file_hash == self.hexdigest:
            return True
        return False

    @staticmethod
    def _calc_eta(start, now, total, current):
        # Not currently implemented
        # Calculates remaining time of download
        if total is None:
            return '--:--'
        dif = now - start
        if current == 0 or dif < 0.001:  # One millisecond
            return '--:--'
        rate = float(current) / dif
        eta = int((float(total) - float(current)) / rate)
        (eta_mins, eta_secs) = divmod(eta, 60)
        if eta_mins > 99:
            return '--:--'
        return '%02d:%02d' % (eta_mins, eta_secs)

    @staticmethod
    def _calc_progress_percent(x, y):
        percent = float(x) / y * 100
        percent = '%.1f' % percent
        return percent
