import logging
import os
import platform
import shlex
import struct
import subprocess
import sys
import tty

from not_so_tuf.version import __version__

log = logging.getLogger(__name__)

if sys.platform == 'win32':
    clear = 'cls'
else:
    clear = 'clear'


def app_header(*args, **kwargs):
    # Takes a string and adds it to the menu header along side
    # the app name.
    try:
        page = args[0]
    except IndexError:
        page = ""
    window_size = get_terminal_size()[0]

    def add_style():
        app = 'Not so TUF v{}'.format(__version__)
        top = '*' * window_size + '\n'
        bottom = '\n' + '*' * window_size + '\n'
        if len(page) > 1:
            header = app + ' - ' + page
        else:
            header = app + page
        header = header.center(window_size)
        msg = top + header + bottom
        return msg
    os.system(clear)
    return add_style()

class _Getch:
    """Gets a single character from standard input.  Does not echo to the
screen."""
    def __init__(self):
        if sys.platform == 'win32':
            self.impl = _GetchWindows()
        else:
            self.impl = _GetchUnix()

    def __call__(self):
        return self.impl()


class _GetchUnix:
    def __init__(self):
        import tty, sys

    def __call__(self):
        import tty, termios
        fd = sys.stdin.fileno()
        old_settings = termios.tcgetattr(fd)
        try:
            tty.setraw(sys.stdin.fileno())
            ch = sys.stdin.read(1)
        finally:
            termios.tcsetattr(fd, termios.TCSADRAIN, old_settings)
        return ch


class _GetchWindows:
    def __init__(self):
        import msvcrt

    def __call__(self):
        import msvcrt
        return msvcrt.getch()


def format_msg(*args):
    # Takes a string and retuns a string formatted
    # to the center of the terminal window.
    window_size = get_terminal_size()[0]
    def fix():
        formatted = []
        finished = ['\n']
        count = 0
        words = args[0].split(' ')
        for w in words:
            w = w + ' '
            if count + len(w) > window_size / 2:
                finished.append(''.join(formatted).center(window_size))
                finished.append('\n')
                count = len(w)
                formatted = []
                formatted.append(w)
            else:
                formatted.append(w)
                count += len(w)
        finished.append(''.join(formatted).center(window_size))
        finished.append('\n')
        return ''.join(finished)
    return fix()


def _options(*args):
    # Takes a list of tuples(menu_name, call_back) adds menu numbers
    # then prints menu to screen.
    # Gets input from user, then returns the callback
    if len(args) != 1:
        return ""
    if not isinstance(args[0], list):
        return ""

    def add_options():
        getch = _Getch()
        menu = []
        count = 1
        for s in args[0]:
            item = '{}. {}\n'.format(count, s[0])
            menu.append(item)
            count += 1
        print ''.join(menu)
        answers = []
        for a in xrange(1, len(menu) + 1):
            answers.append(str(a))
        while 1:
            ans = getch()
            if ans in answers:
                break
            else:
                log.debug('Not an acceptable answer!')
        return args[0][int(ans) -1][1]
    return add_options()


def get_terminal_size():
    """ getTerminalSize()
     - get width and height of console
     - works on linux, os x, windows, cygwin(windows)
     originally retrieved from:
     http://stackoverflow.com/questions/566746/how-to-get-console-window-width-in-python
    """
    current_os = platform.system()
    tuple_xy = None
    if current_os == 'Windows':
        tuple_xy = _get_terminal_size_windows()
        if tuple_xy is None:
            tuple_xy = _get_terminal_size_tput()
            # needed for window's python in cygwin's xterm!
    if current_os in ['Linux', 'Darwin'] or current_os.startswith('CYGWIN'):
        tuple_xy = _get_terminal_size_linux()
    if tuple_xy is None:
        print "default"
        tuple_xy = (80, 25)      # default value
    return tuple_xy


def _get_terminal_size_windows():
    try:
        from ctypes import windll, create_string_buffer
        # stdin handle is -10
        # stdout handle is -11
        # stderr handle is -12
        h = windll.kernel32.GetStdHandle(-12)
        csbi = create_string_buffer(22)
        res = windll.kernel32.GetConsoleScreenBufferInfo(h, csbi)
        if res:
            (bufx, bufy, curx, cury, wattr,
             left, top, right, bottom,
             maxx, maxy) = struct.unpack("hhhhHhhhhhh", csbi.raw)
            sizex = right - left + 1
            sizey = bottom - top + 1
            return sizex, sizey
    except:
        pass


def _get_terminal_size_tput():
    # get terminal width
    # src: http://stackoverflow.com/questions/263890/how-do-i-find-the-width-height-of-a-terminal-window
    try:
        cols = int(subprocess.check_call(shlex.split('tput cols')))
        rows = int(subprocess.check_call(shlex.split('tput lines')))
        return (cols, rows)
    except:
        pass


def _get_terminal_size_linux():
    def ioctl_GWINSZ(fd):
        try:
            import fcntl
            import termios
            cr = struct.unpack('hh',
                               fcntl.ioctl(fd, termios.TIOCGWINSZ, '1234'))
            return cr
        except:
            pass
    cr = ioctl_GWINSZ(0) or ioctl_GWINSZ(1) or ioctl_GWINSZ(2)
    if not cr:
        try:
            fd = os.open(os.ctermid(), os.O_RDONLY)
            cr = ioctl_GWINSZ(fd)
            os.close(fd)
        except:
            pass
    if not cr:
        try:
            cr = (os.environ['LINES'], os.environ['COLUMNS'])
        except:
            return None
    return int(cr[1]), int(cr[0])