from ftplib import FTP_TLS

from boto.s3.connection import S3Connection
from boto.s3.key import Key
from paramiko import SSHClient
from scp import SCPClient


class BaseUploader(object):
    """Base Uploader.  All plugin uploaders should subclass
    this.

    Args:
        host (str): Service to connect to.  Can be ip or domain.

        update_dir (str): Path to upload on remote server

        file_list (list): List of files to upload

    Kwargs:
        bucket_name (str): Name of bucket on AWS S3

    """

    def __init___(self, host, update_dir, file_list, bucket_name=None):
        self.host = host
        self.update_dir = update_dir
        self.file_list = file_list
        self.bucket = bucket_name
        self.file_count = self._get_filelist_count()
        self.files_completed = 0
        self.failed_uploads = []
        self._connect()

    def upload(self):
        """Uploads all files in file_list"""
        self.file_count = self._get_filelist_count()
        for f in self.file_list:
            complete = self._upload_file(f)
            if complete:
                self.files_completed += 1
            else:
                self.failed_uploads.append(f)
        if len(self.failed_uploads) > 0:
            self._retry_upload()

    def _retry_upload(self):
        """Takes list of failed downloads and try's to reupload them"""
        retry = self.failed_uploads[:]
        self.failed_uploads = []
        for f in retry:
            complete = self._upload_file(f)
            if complete:
                self.files_completed += 1
            else:
                self.failed_uploads.append(f)

    def _reconnect(self):
        """Connects/Reconnects to service"""
        raise NotImplementedError('Must be implemented in subclass.')

    def _connect(self):
        """Connects client attribute to service"""
        raise NotImplementedError('Must be implemented in subclass.')

    def _upload_file(self, filename):
        """Uploads file to remote repository

        Args:
            filename (str): file to upload

        Returns:
            (bool) Meaning::

                True - Upload Successful

                False - Upload Failed
        """
        raise NotImplementedError('Must be implemented in subclass.')

    def _get_filelist_count(self):
        return len(self.file_list)


# TODO: figure out how to upload files
# deeper then just the root directory
class S3Uploader(BaseUploader):

    def __init__(self, *agrs, **kwargs):
        super(S3Uploader, self).__init__(*agrs, **kwargs)

    def _reconnect(self):
        """Connects/Reconnects to S3"""
        s3 = S3Connection(aws_access_key_id='', aws_secert_access_key='')
        bucket = s3.get_bucket(self.bucket)
        k = Key(bucket)
        return k

    def _connect(self):
        """Connects client attribute to S3"""
        self.client = self._reconnect()

    def _upload_file(self, filename):
        """Uploads a single file to S3

        Args:
            filename (str): Name of file to upload.

        Returns:
            (bool) Meanings::

                True - Upload Successful

                False - Upload Failed
        """
        self.client.key = filename
        try:
            self.client.set_contents_from_file(filename)
            self.client.make_public()
            return True
        except:
            self._reconnect()
            return False


class SSHUploader(BaseUploader):

    def __init__(self, *agrs, **kwargs):
        super(SSHUploader).__init__(*agrs, **kwargs)

    def _reconnect(self):
        ssh = SSHClient()
        ssh.load_system_host_keys()
        ssh.connect(self.host, username='', key_filename='')
        scp = SCPClient(ssh.get_transport())
        return scp

    def connect(self):
        self.client = self._reconnect()

    def _upload_file(self, filename):
        try:
            self.client.put(filename, remote_path=self.update_dir)
            return True
        except:
            self._reconnect()
            return False


class FTPUploader(BaseUploader):

    def __init__(self, *args, **kwargs):
        super(FTPUploader).__init__(*args, **kwargs)

    def _reconnect(self):
        ftp_uploader = FTP_TLS(self.host, user='', passwd='')
        ftp_uploader.login()
        ftp_uploader.prot_p()
        return ftp_uploader

    def connect(self):
        self.client = self._reconnect()

    def _upload_file(self, filename):
        try:
            payload = 'STOR {}'.format(filename)
            self.client.storbinary(payload, open(filename, 'rb'))
            return True
        except:
            self._reconnect()
            return False
