import getpass
import hashlib
import logging
import os
import struct
import sys

import beefish

log = logging.getLogger(__name__)


class FileCrypt(object):
    """Smaller wrapper around beefish to make it easier to use
    with not-so-tuf.  Upon initialization automatically adds .enc
    to encrypted files.  Gets original filename from config.

    Args:
        filename (str): The name of the file to encrypt
    """

    def __init__(self, filename=None):
        self.filename = filename
        self.enc_filename = filename + '.enc'
        self.filenames = None
        self.password = None

    def encrypt(self):
        """Encrypts file and deletes original"""
        if self.password is None:
            self._get_password()
        beefish.encrypt_file(self.filename, self.enc_filename, self.password)
        os.remove(self.filename)
        self._del_password()

    def decrypt(self):
        """Decrypts encrypted file. Leaves encrypted file in case user
        enters wrong password and decrypted file is junk
        """
        if self.password is None:
            self._get_password()
        beefish.decrypt_file(self.enc_filename, self.filename, self.password)

    def _del_password(self):
        """Deletes user password once its not needed.
        i.e. when the file been encrypted.
        """
        del self.password
        self.password = None

    def _get_password(self):
        """Gets user passwod without echoing to console"""
        self.password = getpass.getpass('Enter passwword')


class ChDir(object):
    """Step into a directory temporarily. Then return to orignal
    directory.
    """
    def __init__(self, path):
        self.old_dir = os.getcwd()
        self.new_dir = path

    def __enter__(self):
        log.debug('Changing to Directory --> {}'.format(self.new_dir))
        os.chdir(self.new_dir)

    def __exit__(self, *args):
        log.debug('Moving back to Directory --> {}'.format(self.old_dir))
        os.chdir(self.old_dir)


def ask_yes_no(question, default='no'):
    """Will ask a question and keeps prompting until
    answered.

    Args:
        question (str): Question to ask end user

    Kwargs:
        default (str): Default answer if user just press enter at prompt

    Returns:
        bool. Meaning::

            True - Answer is  yes

            False - Answer is no
    """
    default = default.lower()
    yes = ['yes', 'ye', 'y']
    no = ['no', 'n']
    if default in no:
        help = '[N/y]?'
        default = False
    else:
        default = True
        help = '[Y/n]?'
    while 1:
        display = question + '\n' + help
        answer = raw_input(display)
        if answer == '':
            return default
        if answer in yes:
            return True
        elif answer in no:
            return False
        else:
            print 'Please answer yes or no only!\n'
            raw_input('Press enter to continue')
            print '\n\n\n\n'


def rsa_verify(message, signature, key):
    def b(x):
        if sys.version_info[0] == 2:
            return x
        else:
            return x.encode('latin1')
    assert(type(message) == type(b('')))
    block_size = 0
    n = key[0]
    while n:
        block_size += 1
        n >>= 8
    signature = pow(int(signature, 16), key[1], key[0])
    raw_bytes = []
    while signature:
        raw_bytes.insert(0, struct.pack("B", signature & 0xFF))
        signature >>= 8
    signature = (block_size - len
                 (raw_bytes)) * b('\x00') + b('').join(raw_bytes)
    if signature[0:2] != b('\x00\x01'):
        return False
    signature = signature[2:]
    if not b('\x00') in signature:
        return False
    signature = signature[signature.index(b('\x00'))+1:]
    if not signature.startswith(b('\x30\x31\x30\x0D\x06\x09\x60\x86\x48\x01\x65\x03\x04\x02\x01\x05\x00\x04\x20')):
        return False
    signature = signature[19:]
    if signature != hashlib.sha256(message).digest():
        return False
    return True


def _get_package_hashes(filename):
        """Get hash of provided file

        Args:
            package (str): File name to get hash for

        Returns:
            (str) md5 hash
        """
        log.info('Getting package hashes')
        with open(filename, 'rb') as f:
            print filename
            print os.getcwd()
            _hash = hashlib.md5(f.read()).hexdigest()
            log.debug('Hash for file {}: {}'.format(filename, _hash))
        return _hash
