# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>

import bsdiff4
import hashlib
import logging
import os
import sys

from .file_downloader import FileDownloader
from .utils import (ChDir, _version_string_to_tuple,
                    _version_tuple_to_string, platform_)

log = logging.getLogger(__name__)


class Patcher(object):
    """Downloads, verifies, and patches binaries

    Args:
        name (str): Name of binary to patch

        json_data (dict): Info dict with all package meta data

        current_version (str): Version number of currently installed binary

        hightest_version (str): Newest version available

        update_folder (str): Path to update folder to place updated binary in
    """

    def __init__(self, name, json_data, current_version, highest_version,
                 update_folder):
        self.name = name
        self.json_data = json_data
        self.current_version = current_version
        self.highest_version = highest_version
        self.update_folder = update_folder
        self.patch_data = []
        self.patch_binary_data = []
        self.og_binary = None
        self.plat = platform_

    def start(self):
        """Proxy method for :meth:`_verify_installed_binary`,
        :meth:`_get_all_patches`, :meth:`_download_verify_patches`,
        :meth:`_apply_patches_in_memory` & :meth:`_write_update_to_disk`

        Returns:
            (bool) Meanings::

                True - Patch update successful

                False - Failed to pass at least 1 Check.  Failed patch update
        """
        log.info('Starting patch updater...')
        # Check hash on installed binary to begin patching
        binary_check = self._verify_installed_binary()
        if not binary_check:
            log.warning('Binary check failed...')
            return False
        # Getting all required patch meta-data
        all_patches = self._get_all_patches(self.name)
        if not all_patches:
            log.warning('Cannot find all patches...')
            return False

        # Downloading and verify patches as i go
        download_check = self._download_verify_patches()
        if not download_check:
            log.warning('Patch check failed...')
            return False
        self._apply_patches_in_memory()
        self._write_update_to_disk()
        return True

    def _verify_installed_binary(self):
        """Verifies currently installed binary against known hash

        Returns:
            (bool) Meanings::

                True - Binary verified

                False - Binary not verified
        """
        log.info('Checking for current installed binary to patch')
        file_info = self._get_current_filename_and_hash(self.name,
                                                        self.current_version)
        filename = file_info['filename']
        _hash = file_info['hash']
        # I just really like using this ChDir context
        # manager.  Even sent the developer a cup of coffee
        with ChDir(self.update_folder):
            file_hash = self._get_package_hashes(filename)
            if _hash != file_hash:
                log.warning('Binary hash mismatch')
                return False
            if not os.path.exists(filename):
                log.warning('Cannot find binary to patch')
                return False
            with open(filename, 'rb') as f:
                self.og_binary = f.read()
            os.remove(filename)
        log.info('Binary found and verified')
        return True

    # We will take all versions.  Then append any version
    # thats greater then the current version to the list
    # of needed patches.
    def _get_all_patches(self, name):
        """Creates a list of patches to get from current binary to
        latest binary

        Args:
            name (str): name of binary to get patches for.

        Returns:
            (bool) Meanings::

                True - If all patches were found

                False - Failed to find all patches
        """
        needed_patches = []
        versions = sorted(map(_version_string_to_tuple,
                          self.json_data['y_data'][name].keys()))

        log.info('getting required patches')
        for i in versions:
            if i > _version_string_to_tuple(self.current_version):
                needed_patches.append(i)

        # Taking the list of needed patches and extracting the
        # patch data from it. If any loop fails, will return False
        # and start full binary update.
        log.info('Getting patch meta-data')

        for p in needed_patches:
            info = {}
            v_num = _version_tuple_to_string(p)
            try:
                info['patch_name'] = self.json_data['y_data'][name][v_num][self.plat]['patch_name']
                info['patch_url'] = self.json_data['y_data'][name][v_num][self.plat]['patch_url']
                info['patch_md5'] = self.json_data['y_data'][name][v_num][self.plat]['patch_md5']
                self.patch_data.append(info)
            except KeyError:
                log.error('Missing required patch meta-data')
                return False
        return True

    def _download_verify_patches(self):
        """Downloads & verifies all patches

        Returns:
            (bool) Meanings::

                True - All patches have been downloaded and verified

                False - 1 or more patches failed to download and/or verify
        """
        log.debug('Downloading patches')
        for p in self.patch_data:
            fd = FileDownloader(p['patch_name'], p['patch_url'],
                                p['patch_md5'])

            data = fd.download_verify_return()
            if data is not None:
                self.patch_binary_data.append(data)
            else:
                return False
        return True

    def _apply_patches_in_memory(self):
        """Applies a sequence of patches in memory"""
        log.info('Applying patches')
        # Beginning the patch process
        self.new_binary = self.og_binary
        for i in self.patch_binary_data:
            self.new_binary = bsdiff4.patch(self.new_binary, i)

    def _write_update_to_disk(self):
        """Writes updated binary to disk"""
        log.info('Writing update to disk')
        filename = self.json_data['y_data'][self.name][self.highest_version][self.plat]['filename']
        with ChDir(self.update_folder):
            with open(filename, 'wb') as f:
                f.write(self.new_binary)

    def _get_current_filename_and_hash(self, name, version):
        """Returns filename and hash for given name and version

        Args:
            name (str): Name of binary to search for
            version (str): Version number to search for

        Returns:
            (dict) Meanings::

                filename = dict['filename']
                hash = dict['hash']
        """
        info = {}
        info['filename'] = self.json_data['y_data'][name][version][self.plat]['filename']
        info['hash'] = self.json_data['y_data'][name][version][self.plat]['md5']
        return info

    def _get_package_hashes(self, package):
        """Get hash of provided package

        Args:
            package (str): Name of package to get hash from

        Returns:
            (str) md5 hash
        """
        # Returns has of provided package
        log.debug('Getting package hashes')
        with open(package, 'rb') as f:
            _hash = hashlib.md5(f.read()).hexdigest()
            log.debug('Hash for file {}: {}'.format(package, _hash))
        return _hash
