# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

from getpass import getpass
import hashlib
import logging
import os

from not_so_tuf.utils import _cwd

log = logging.getLogger(__name__)


class SetupConfig(object):
    # If left None "Not_So_TUF" will be used
    APP_NAME = None

    # Directory for updater to place verified updates.
    # If left blank will be place in the users home
    # directory Unix ".Not_So_TUF"
    # windows "Not_So_Tuf"
    APP_DATA_DIR = None

    # If set more debug info will be printed to console
    DEBUG = False

    # Work directory on dev machine for framework to
    # do its business. sign updates, get hashes etc...
    # If None a data folder will be created in the
    # current directory
    DEV_DATA_DIR = None

    # Length of keys to sign and verify files with
    # If left None 2048 key size will be used
    KEY_LENGTH = None

    # Name made for your private key. If left
    # None "Not_So_TUF.pem" will be used
    PRIVATE_KEY_NAME = None

    # Public Key used by your app to verify update data
    # REQUIRED
    PUBLIC_KEY = None

    # Name made for your public key.  If left
    # None "Not_So_TUF.pub" will be used
    PUBLIC_KEY_NAME = None

    # Online repository where you host your packages
    # and version file
    # REQUIRED
    UPDATE_URL = None

    # Support for patch updates
    UPDATE_PATCHES = True

    # Upload Setup
    REMOTE_DIR = None
    HOST = None

    # SSH settings
    # Path to ssh key of server
    SSH_USERNAME = None
    SSH_KEY_PATH = ''

    # FTP settings
    FTP_USERNAME = None
    FTP_PASSWORD = None

    # S3 settings
    ACCESS_KEY_ID = None
    SECRET_ACCESS_KEY = None
    BUCKET_NAME = None

    def add_private_key_name(self, name):
        self.PRIVATE_KEY_NAME = name + '.pem'

    def add_public_key_name(self, name):
        self.PUBLIC_KEY_NAME = name + '.pub'


def get_correct_answer(question, default=None, required=False):
    while 1:
        if default is None:
            msg = 'None'
        else:
            msg = ('\n[DEFAULT] -> {}\nPress Enter To '
                   'Use Default'.format(default))
        prompt = question + msg + '\n--> '
        answer = raw_input(prompt)
        if answer == '' and required:
            print 'You have to enter a value\n\n'
            raw_input('Press enter to continue')
            print '\n\n'
            continue
        if answer == '' and default is not None:
            answer = default
        _ans = ask_yes_no('You entered {}, is this correct?'.format(answer))
        if _ans:
            return answer


def ask_yes_no(question, default='no'):
    default = default.lower()
    yes = ['yes', 'ye', 'y']
    no = ['no', 'n']
    if default in no:
        help = '[N/y]?'
        default = False
    else:
        default = True
        help = '[Y/n]?'
    while 1:
        display = question + '\n' + help
        answer = raw_input(display)
        if answer == '':
            return default
        if answer in yes:
            return True
        elif answer in no:
            return False
        else:
            print 'Please answer yes or no only!\n'
            raw_input('Press enter to continue')
            print '\n\n\n\n'


# Fixed path to work without quotes
def path_fixer(path):
    # Removing ' & " in case user used them
    # for path with spaces
    path.replace("'", "")
    path.replace('"', '')

    # Correcting the path with spaces to work
    # without quotes
    return path.replace(' ', '\ ')


# Explains itself
def sha_digest(data):
    return hashlib.sha256(data).hexdigest()


# Makes inputting directory more like shell
def _directory_fixer(_dir):
    if _dir == 'Current Working Directory':
        log.debug('Using cwd for Current Working Directory')
        _dir = _cwd
    elif _dir.startswith('~'):
        log.debug('Expanding ~ to full user path')
        _dir = _dir[2:]
        _dir = os.path.join(os.path.expanduser('~'), _dir)
    return _dir


# Used to prevent sending out updates with debug
# turned on...
# If there is a better way to do this please chime in!
def dev_machine_check():
    files = os.listdir(os.getcwd())
    if 'app.py' in files and 'make_binary.py' in files:
        log.debug('We are on the dev machine.')
        return logging.DEBUG
    else:
        log.debug('I am not on the dev machine.')
        return logging.INFO
