# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

from ftplib import FTP_TLS
import logging
import os
import sys
import time

from . import UploaderError
from .utils import _remove_hidden_stuff_mac, ChDir

try:
    from boto.exception import S3ResponseError
    from boto.s3.connection import S3Connection
    from boto.s3.key import Key
except ImportError:
    boto = None

from paramiko import SSHClient
from scp import SCPClient

log = logging.getLogger(__name__)


class Uploader(object):
    """Uploads updates to configured servers.  SSH, SFTP, S3
    Will automatically pick the correct uploader depending on
    what is configured thorough the config object

    Sets up client with config values from obj

        Args:
            obj (instance): config object
    """
    def __init__(self, app=None):
        if app:
            self.init_app(app)
        else:
            raise UploaderError(u'Must pass an instance of NST')

    def init_app(self, obj):
        """Sets up client with config values from obj

        Args:
            obj (instance): config object
        """
        self.data_dir = obj.config.get('DEV_DATA_DIR', None)
        self.data_dir = os.path.join(self.data_dir, 'nst-data')
        self.deploy_dir = os.path.join(self.data_dir, 'deploy')
        self.remote_dir = obj.config.get('REMOTE_DIR', None)
        self.host = obj.config.get('HOST', None)
        # SSH Info
        self.ssh_username = obj.config.get('SSH_USERNAME', None)
        self.ssh_key_path = obj.config.get('SSH_KEY_PATH')

        # FTP Info
        self.ftp_username = obj.config.get('FTP_USERNAME', None)
        self.ftp_password = obj.config.get('FTP_PASSWORD', None)

        # S3 Info
        self.access_key = obj.config.get('ACCESS_KEY_ID', None)
        self.secret_key = obj.config.get('SECRET_ACCESS_KEY', None)
        self.bucket_name = obj.config.get('BUCKET_NAME', None)
        self.ready = False
        self.uplaoder = None

    def upload(self):
        """Proxy function that calls the upload method on the received uploader
        Only calls the upload method if an uploader is found.
        """
        if self.ready:
            self.uploader.deploy_dir = self.deploy_dir
            self.uploader.upload()

    def set_uploader(self, requested_uploader):
        """Returns an uploader object. 1 of S3, SCP, SFTP.
        SFTP uploaders not supported at this time.

        Args:
            requested_uploader (string): Either s3 or scp

        Returns:
            object (instance): Uploader object
        """
        log.debug('Requested uploader type: {}'.format(
                  type(requested_uploader)))

        if isinstance(requested_uploader, str):
            requested_uploader = requested_uploader.lower()
        else:
            raise UploaderError('You must provide a string to set_uploader',
                                expected=True)

        files = _remove_hidden_stuff_mac(os.listdir(self.deploy_dir))
        if 's3' in requested_uploader and len(requested_uploader) == 2:
            if self.access_key is None and self.secret_key is not None \
                    and self.bucket_name is not None:
                    self.ready = True
                    self.uploader = S3Uploader(file_list=files,
                                               aws_access_id=self.access_key,
                                               aws_secret_key=self.secret_key,
                                               bucket_name=self.bucket_name)
            else:
                log.error('You must provide all correct s3 settings')

        elif 'scp' in requested_uploader and len(requested_uploader) == 3:
            if self.host is not None and self.ssh_username is not None:
                self.uplaoder = True
                self.uploader = SSHUploader(file_list=files,
                                            remote_dir=self.remote_dir,
                                            host=self.host,
                                            username=self.ssh_username,
                                            ssh_key_file=self.ssh_key_path)
            else:
                log.error('You must provide all correct scp settings.')
        else:
            log.error('You must provide a valid option. Either s3 or scp')

        return
        # ToDo: Add stp to elif statement once implemented!
        if self.host is not None and self.ftp_username is not None:
            if self.ftp_password is not None:
                pass
            else:
                raise UploaderError(u'You must provide ftp password',
                                    expected=True)

        if getattr(sys, 'frozen', False):
            raise UploaderError(u'Make sure you have all config options set',
                                expected=True)
        raise UploaderError(u'Make sure you have all config options set',
                            expected=True)


class BaseUploader(object):
    """Base Uploader.  All uploaders should subclass
    this base class

    Kwargs:
        file_list (list): List of files to upload

        host (str): Either ip or domain name of remote servers

        bucket_name (str): Name of bucket on AWS S3

        remote_dir (str): The directory on remote server to upload files to.

        username (str): login username of remote server

        password (str): login password of remote server

        ssh_key_file (str): full path to ssh pub key on local machine

        aws_access_id (str): aws access id for S3 bucket account owner

        aws_secret_key (str): aws secret key for S3 bucket account owner

    """
    def __init__(self, file_list=None, host=None, bucket_name=None,
                 remote_dir=None, username=None,
                 password=None, ssh_key_file=None, aws_access_id=None,
                 aws_secret_key=None):
        self.file_list = file_list
        self.host = host
        self.bucket = bucket_name
        self.remote_dir = remote_dir
        self.username = username
        self.password = password
        self.ssh_key_file = ssh_key_file
        self.access_id = aws_access_id
        self.secret_key = aws_secret_key
        self.failed_uploads = []
        self.deploy_dir = None

    def upload(self):
        """Uploads all files in file_list"""
        self.files_completed = 1
        self.file_count = self._get_filelist_count()
        for f in self.file_list:
            msg = 'Uploading: {}' .format(f)
            msg2 = ' - File {} of {}\n'.format(self.files_completed,
                                               self.file_count)
            print msg + msg2
            complete = self._upload_file(f)
            if complete:
                log.debug('{} uploaded successfully'.format(f))
                self.files_completed += 1
            else:
                log.warning('{} failed to upload.  will retry'.format(f))
                self.failed_uploads.append(f)
        if len(self.failed_uploads) > 0:
            self._retry_upload()
        if len(self.failed_uploads) < 1:
            print "\nUpload Complete"
            time.sleep(3)
        else:
            print 'The following files were not uploaded'
            for i in self.failed_uploads:
                log.error('{} failed to upload'.format(i))
                print i

    def _retry_upload(self):
        """Takes list of failed downloads and try's to reupload them"""
        retry = self.failed_uploads[:]
        self.failed_uploads = []
        failed_count = len(retry)
        count = 1
        for f in retry:
            msg = 'Retyring: {} - File {} of {}\n'.format(f,
                                                          count,
                                                          failed_count)
            print msg
            complete = self._upload_file(f)
            if complete:
                log.debug('{} uploaded on retry'.format(f))
                count += 1
            else:
                self.failed_uploads.append(f)
        if len(self.failed_uploads) > 0:
            print '\nThe following files failed to upload...'
            for f in self.failed_uploads:
                print f
            time.sleep(3)
        else:
            print '\nUpload complete'

    def _connect(self):
        """Connects client attribute to service"""
        raise NotImplementedError('Must be implemented in subclass.')

    def _upload_file(self, filename):
        """Uploads file to remote repository

        Args:
            filename (str): file to upload

        Returns:
            (bool) Meaning::

                True - Upload Successful

                False - Upload Failed
        """
        raise NotImplementedError('Must be implemented in subclass.')

    def _get_filelist_count(self):
        return len(self.file_list)


class SSHUploader(BaseUploader):

    def __init__(self, **kwargs):
        super(SSHUploader, self).__init__(**kwargs)
        self.host = kwargs['host']
        self.remote_dir = kwargs['remote_dir']
        self.file_list = kwargs['file_list']
        self.username = kwargs['username']
        self.ssh_key_file = kwargs['ssh_key_file']
        self._connect()

    def _connect(self):
        self.ssh = SSHClient()
        self.ssh.load_system_host_keys()
        self.ssh.connect(self.host, username=self.username,
                         key_filename=self.ssh_key_file)
        self.client = SCPClient(self.ssh.get_transport(),
                                progress=ssh_progress)

    def _upload_file(self, filename):
        with ChDir(self.deploy_dir):
            try:
                self.client.put(filename, remote_path=self.remote_dir)
                return True
            except Exception as e:
                log.error(e, exc_info=True)
                self._connect()
                return False


def ssh_progress(filename, size, sent):
    percent = float(sent) / size * 100
    sys.stdout.write('\r%.1f' % percent)
    sys.stdout.flush()


# TODO: figure out how to upload files
# deeper then just the root directory
class S3Uploader(BaseUploader):

    def __init__(self, **kwargs):
        super(S3Uploader, self).__init__(**kwargs)
        self.file_list = kwargs['file_list']
        self.access_key = kwargs['aws_access_id']
        self.secret_key = kwargs['aws_secret_key']
        self.bucket_name = kwargs['bucket_name']
        self.deploy_dir = None
        self._connect()

    def _connect(self):
        """Connects client attribute to S3"""
        self.s3 = S3Connection(aws_access_key_id=self.access_key,
                               aws_secret_access_key=self.secret_key,
                               debug=True)
        bucket = self.s3.get_bucket(self.bucket_name)
        self.client = Key(bucket)

    def _upload_file(self, filename):
        """Uploads a single file to S3

        Args:
            filename (str): Name of file to upload.

        Returns:
            (bool) Meanings::

                True - Upload Successful

                False - Upload Failed
        """
        with ChDir(self.deploy_dir):
            self.client.key = filename
            try:
                self.client.set_contents_from_filename(filename,
                                                   cb=s3_progress)
                self.client.make_public()
                print 'Did that upload boy'
                return True
            except S3ResponseError as e:
                log.error(str(e), exc_info=True)
                self._connect()
                print 'Oopss didn\'t upload that boy'
                return False


def s3_progress(current, total):
    percent = float(current) / total * 100
    sys.stdout.write('\r%.1f' % percent)
    sys.stdout.flush()


class SFTPUploader(BaseUploader):

    def __init__(self, *args, **kwargs):
        super(SFTPUploader, self).__init__()

    def _reconnect(self):
        sftp_uploader = FTP_TLS(self.host, user='', passwd='')
        sftp_uploader.login()
        sftp_uploader.prot_p()
        return sftp_uploader

    def _connect(self):
        self.client = self._reconnect()

    def _upload_file(self, filename):
        try:
            payload = 'STOR {}'.format(filename)
            self.client.storbinary(payload, open(filename, 'rb'))
            return True
        except:
            self._reconnect()
            return False
