# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import base64
import getpass
import hashlib
import logging
import os
import platform
import struct
import sys

from not_so_tuf import NotSoTufError

try:
    import beefish
    import keyring
    import pbkdf2
except ImportError:
    raise NotSoTufError('Must have beefish, keyring & pbkdf2 installed.',
                        expected=True)

log = logging.getLogger(__name__)

if getattr(sys, 'frozen', False):
    log.debug('Running frozen')
    # we are running in a |PyInstaller| bundle
    _cwd = os.path.dirname(sys.argv[0])
else:
    log.debug('Running thawed out.  e.g. not frozen :)')
    # we are running in a normal Python environment
    _cwd = os.getcwd()

if sys.platform == 'win32':
    platform_ = 'win'
elif sys.platform == 'darwin':
    platform_ = 'mac'
else:
    if platform.uname()[-1] == 'x86_64':
        platform_ = 'nix64'
    elif platform.uname()[-1] != 'x86_64':
        platform_ = 'nix'


class FileCrypt(object):
    """Smaller wrapper around beefish to make it easier to use
    with not-so-tuf.  Upon initialization automatically adds .enc
    to encrypted files.  Gets original filename from config.

    Args:
        filename (str): The name of the file to encrypt
    """

    def __init__(self, filename=None):
        self.filename = filename
        self.enc_filename = filename + '.enc'
        self.filenames = None
        self.password = None
        self.tries = 0
        self._get_password()

    def encrypt(self):
        """Encrypts file and deletes original"""
        if self.password is None:
            self._get_password()
        try:
            beefish.encrypt_file(self.filename, self.enc_filename,
                                 self.password)
            os.remove(self.filename)
        except IOError:
            log.warning('Cannot find original file')
        self._del_internal_password()

    def decrypt(self):
        """Decrypts encrypted file. Leaves encrypted file in case user
        enters wrong password and decrypted file is junk
        """
        if self.password is None:
            self._get_password()
        try:
            beefish.decrypt_file(self.enc_filename, self.filename,
                                 self.password)
        except IOError:
            log.warning('Cannot find encrypted file.')

    def _del_internal_password(self):
        """Deletes user password once its not needed.
        i.e. when the file been encrypted.
        """
        del self.password
        self.password = None

    def _get_password(self):
        """Gets user passwod without echoing to console"""
        username = self.make_username()
        app_name = 'NST-' + username
        log.info('Looking for keyring password')
        keyring_password = keyring.get_password(app_name, username)
        if keyring_password is None:
            log.warning('No keyring password found')
            self._set_password()
            keyring_password = keyring.get_password(app_name, username)
        while self.tries < 2:
            attempted_password = getpass.getpass('Enter login password')
            if keyring_password == pbkdf2.crypt(attempted_password,
                                                keyring_password):
                self.password = keyring_password
                break
            else:
                self.tries += 1
                raw_input('\nInvalid Password.  Press enter to try again')
        if self.password is None:
            sys.exit(0)

    def _set_password(self):
        """Asks user for password twice, hashes password then sets password
        in keyring
        """
        while 1:
            print 'Create a new password.\n\n'
            password1 = getpass.getpass('Enter password\n--> ')
            password2 = getpass.getpass('Enter password again\n--> ')
            if password1 == password2:
                break
            else:
                msg = 'Passwords don\'t match.  Press enter to try again.'
                raw_input(msg)
        username = self.make_username()
        app_name = 'NST-' + username
        hashed = pbkdf2.crypt(password1)
        keyring.set_password(app_name, username, hashed)

    def _del_password(self):
        username = self.make_username()
        app_name = 'NST-' + username
        self._get_password()
        if self.password:
            del self.self.password
            keyring.delete_password(app_name, username)
        else:
            pass

    def make_username(self):
        """Makes username by base54 encoding current working directory.
        """
        return base64.b64encode(_cwd)


class ChDir(object):
    """Step into a directory temporarily. Then return to orignal
    directory.
    """
    def __init__(self, path):
        self.old_dir = os.getcwd()
        self.new_dir = path

    def __enter__(self):
        log.debug('Changing to Directory --> {}'.format(self.new_dir))
        os.chdir(self.new_dir)

    def __exit__(self, *args):
        log.debug('Moving back to Directory --> {}'.format(self.old_dir))
        os.chdir(self.old_dir)


def ask_yes_no(question, default='no'):
    """Will ask a question and keeps prompting until
    answered.

    Args:
        question (str): Question to ask end user

    Kwargs:
        default (str): Default answer if user just press enter at prompt

    Returns:
        bool. Meaning::

            True - Answer is  yes

            False - Answer is no
    """
    default = default.lower()
    yes = ['yes', 'ye', 'y']
    no = ['no', 'n']
    if default in no:
        help = '[N/y]?'
        default = False
    else:
        default = True
        help = '[Y/n]?'
    while 1:
        display = question + '\n' + help
        answer = raw_input(display)
        if answer == '':
            return default
        if answer in yes:
            return True
        elif answer in no:
            return False
        else:
            print 'Please answer yes or no only!\n'
            raw_input('Press enter to continue')
            print '\n\n\n\n'


# Big Thanks to FiloSottile for making this lib for us
# All Files can be found in this gist
# https://gist.github.com/FiloSottile/4340076
def rsa_verify(message, signature, key):
    def b(x):
        if sys.version_info[0] == 2:
            return x
        else:
            return x.encode('latin1')
    assert(type(message) == type(b('')))
    block_size = 0
    n = key[0]
    while n:
        block_size += 1
        n >>= 8
    signature = pow(int(signature, 16), key[1], key[0])
    raw_bytes = []
    while signature:
        raw_bytes.insert(0, struct.pack("B", signature & 0xFF))
        signature >>= 8
    signature = (block_size - len
                 (raw_bytes)) * b('\x00') + b('').join(raw_bytes)
    if signature[0:2] != b('\x00\x01'):
        return False
    signature = signature[2:]
    if not b('\x00') in signature:
        return False
    signature = signature[signature.index(b('\x00'))+1:]
    if not signature.startswith(b('\x30\x31\x30\x0D\x06\x09\x60\x86\x48\x01\x65\x03\x04\x02\x01\x05\x00\x04\x20')):
        return False
    signature = signature[19:]
    if signature != hashlib.sha256(message).digest():
        return False
    return True


def _get_package_hashes(filename):
        """Get hash of provided file

        Args:
            package (str): File name to get hash for

        Returns:
            (str) md5 hash
        """
        log.info('Getting package hashes')
        with open(filename, 'rb') as f:
            _hash = hashlib.md5(f.read()).hexdigest()
            log.debug('Hash for file {}: {}'.format(filename, _hash))
        return _hash


def _remove_hidden_stuff_mac(files):
        """Removes hidden files from file list

        Args:
            files (list): list of files

        Returns:
            (list) without .DS_Store and .AppleDouble
        """
        # Does what the name entails.
        # ToDo: Implement a better way of handling this if there is one
        # Any suggestions are warmly welcomed :)
        new_list = []
        for l in files:
            if '.DS_Store' not in l and '.AppleDouble' not in l:
                new_list.append(l)
        return new_list


def _version_string_to_tuple(version):
        return tuple(map(int, version.split('.')))


def _version_tuple_to_string(version):
    return '.'.join(map(str, version))
