#!/usr/bin/env python

# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import logging
import os
import pickle
import shutil
import sys
import time

# cli specific imports
from .ui_utils import menu_options, app_header, format_msg
from .utils import (get_correct_answer, ask_yes_no, path_fixer,
                    SetupConfig, _directory_fixer)

# framework stuff
from not_so_tuf import NotSoTuf
from not_so_tuf.client import Client
from not_so_tuf.key_handler import KeyHandler
from not_so_tuf.package_handler import PackageHandler
from not_so_tuf.uploader import Uploader
from not_so_tuf.utils import FileCrypt, _cwd
from not_so_tuf.version import __version__


log = logging.getLogger(__name__)
log.setLevel(logging.DEBUG)
h = logging.NullHandler()
h.setLevel(logging.DEBUG)
log.addHandler(h)


class Ui(object):

    def __init__(self):
        if sys.platform == 'win32':
            log.debug('Working on windows')
            self._clear = 'cls'
        else:
            log.debug('Working on Unix')
            self._clear = 'clear'
        try:
            self.config = self.load_config()
            log.debug('Loaded config.data file')
        except Exception as e:
            log.warning('config file not found')
            log.error(e, exc_info=True)
            self.config = SetupConfig()
        self.nst = NotSoTuf(__name__, self.config)

    def main(self):
        log.debug('Main menu')
        self.cwd = _cwd
        dec_path = os.path.join(self.cwd, 'config.data')
        enc_path = os.path.join(self.cwd, 'config.data.enc')
        if not os.path.exists(dec_path) and not os.path.exists(enc_path):
            self.initial_setup()
        options = [('Sign Updates', self.sign_updates),
                   ('Upload', self.upload), ('Keys', self.keys),
                   ('Settings', self.settings), ('Quit', self.quit)]

        print app_header()
        next = menu_options(options)
        return next()

    def sign_updates(self):
        log.debug('Sign Updates Menu')
        print app_header('Sign Updates')
        msg = 'Make sure updates are in "new" folder then press "1" to start'
        print format_msg(msg)
        options = [('Start', self.start_sign_updates),
                   ('Go Back', self.main)]
        next = menu_options(options)
        return next()

    def start_sign_updates(self):
        log.debug('Signing updates')
        print app_header('Signing updates...')
        print format_msg('Please wait...')
        package_handler = PackageHandler(self.nst)
        package_handler.setup()
        package_handler.update_package_list()
        key_handler = KeyHandler(self.nst)
        key_handler.sign_update()
        package_handler.deploy()
        print 'Update signing complete...'
        time.sleep(3)
        self.main()

    def upload(self):
        log.debug('Upload')
        self.uploader = Uploader(self.nst)
        print app_header('Upload')
        msg = ('Select destination to begin upload.  If you don\'t have any '
               'upload settings, go to settings menu, then update settings or '
               'you\'ll get an error.')
        print format_msg(msg)
        options = [('S3', self.s3_upload), ('SCP', self.scp_upload),
                   ('Go Back', self.main)]
        next = menu_options(options)
        return next()

    def s3_upload(self):
        log.debug('S3 Uploader')
        print app_header('S3 Uploader')
        self.uploader.set_uploader('s3')
        if self.uploader.ready:
            self.uploader.upload()
        else:
            del self.uploader
            msg = ('I told you to update your settings right? Right! Press '
                   'enter to return to main menu and please this time update '
                   'your settings.')
            print format_msg(msg)
            raw_input()
            self.main()

    def scp_upload(self):
        log.debug('SCP Uploader')
        print app_header('SCP Uploader')
        self.uploader.set_uploader('scp')
        if self.uploader.ready:
            self.uploader.upload()
        else:
            del self.uploader
            msg = ('I told you to update your settings right? Right! Press '
                   'enter to return to main menu and please this time update '
                   'your settings.')
            print format_msg(msg)
            raw_input()
            self.main()

    def keys(self):
        log.debug('Keys Menu')
        options = [('Show Public Key', self.show_public_key),
                   ('Make Decrypted Copy of Private Key',
                    self.private_key_copy),
                   ('Go Back', self.main)]
                   # ('Replace Keys', self.main), ('Go Back', self.main)]
        print app_header('Keys')
        next = menu_options(options)
        return next()

    def show_public_key(self):
        log.debug('Show public key')
        key_handler = KeyHandler(self.nst)
        key_handler.print_public_key()
        raw_input(format_msg('Press enter to continue'))
        self.keys()

    def private_key_copy(self):
        log.debug('Copied private key')
        key_handler = KeyHandler(self.nst)
        key_handler.copy_decrypted_private_key()
        msg = 'Private key decrypted. Press enter to continue'
        raw_input(format_msg(msg))
        self.keys()

    def replace_keys(self):
        log.debug('Replacing keys')
        key_handler = KeyHandler(self.nst)
        key_handler.make_keys(overwrite=True)

    def make_keys(self):
        log.debug('Making keys')
        key_handler = KeyHandler(self.nst)
        key_handler.make_keys()

    def settings(self):
        log.debug('Settings Menu')
        print app_header('Settings')
        options = [('View Settings', self.view_settings),
                   ('Update Settings', self.update_settings),
                   ('Decrypt and Copy Config', self.copy_decrypted_config),
                   ('Initial Setup', self.initial_setup),
                   ('Delete Password', self.delete_password),
                   ('Go Back', self.main)]

        next = menu_options(options)
        return next()

    def delete_password(self):
        log.warning('About to delete password!')
        print app_header('Proceed with Caution')
        msg = ('Deleting you password will render you encrypted config and '
               'private key file useless. Make sure you have a decrypted '
               'copy of both.')
        print format_msg(msg)
        answer = ask_yes_no('Are you sure you want to continue?\nSelecting no '
                            'will return you to the main menu!', default='no')
        if not answer:
            self.main()

        answer = ask_yes_no('Would you like to make decrypted copies of '
                            'both files now?  If you choose no there is '
                            'not going back.  You have been warned.',
                            default='y')
        if answer:
            self.copy_decrypted_config
            self.copy_decrypted_private_key
        self.delete_password()

    def view_settings(self):
        log.debug('View Settings Menu')
        for k, v in self.config.__dict__.items():
            print k.ljust(20), v
        raw_input('Press enter to continue')
        return self.settings()

    def update_settings(self):
        log.debug('Update Settings Menu')
        print app_header('Update Settings')
        msg = ('Enter the number of each item you\'d like to update.')
        print format_msg(msg)
        print format_msg('Examples:')
        print format_msg('-->13 ')
        print format_msg('-->235 ')
        print format_msg('-->513')
        print format_msg('Then press Enter')
        print ('1. App Name\n2. Dev Data Dir\n3. Update Url\n4. Update SSH '
               'settings\n5. Update S3 settings\n')
        answers = raw_input('-->')

        if '1' in answers:
            app_name = get_correct_answer('Enter APP NAME',
                                          default=self.config.APP_NAME)
            self.config.APP_NAME = app_name
        if '2' in answers:
            data_dir = get_correct_answer('Enter directory to store work '
                                          'files',
                                          default=self.config.DEV_DATA_DIR)
            self.config.DEV_DATA_DIR = _directory_fixer(data_dir)
        if '3' in answers:
            url = get_correct_answer('Enter new update url',
                                     default=self.config.UPDATE_URL)
            self.config.UPDATE_URL = url

        if '4' in answers:
            remote_dir = get_correct_answer('Enter remote path',
                                            default=self.config.REMOTE_DIR)
            ssh_host = get_correct_answer('Enter ssh host',
                                          default=self.config.HOST)

            ssh_username = get_correct_answer('Enter ssh username',
                                              default=self.config.SSH_USERNAME)

            ssh_key_path = get_correct_answer('Enter path to key file',
                                              default=self.config.SSH_KEY_PATH)

            self.config.REMOTE_DIR = remote_dir
            self.config.HOST = ssh_host
            self.config.SSH_USERNAME = ssh_username
            self.config.SSH_KEY_PATH = _directory_fixer(ssh_key_path)

        if '5' in answers:
            self.config.ACCESS_KEY_ID = get_correct_answer('Enter access key',
                                                           required=True)
            self.config.SECRET_ACCESS_KEY = get_correct_answer('Enter secret',
                                                               required=True)
            self.config.BUCKET_NAME = get_correct_answer('Enter bucket name',
                                                         required=True)

        self.save_config(self.config)
        print 'Saving new config....'
        time.sleep(3)
        return self.main()

    def quit(self):
        log.debug('Quitting')
        print 'See Ya!'
        sys.exit()

    def initial_setup(self):
        log.debug('Initial Setup Menu')
        print app_header('Initial Setup')
        msg = ('You should only have to run this once.')
        print format_msg(msg)
        options = [('Begin Setup', self.start_intial_setup),
                   ('Go Back', self.main)]
        next = menu_options(options)
        return next()

    def start_intial_setup(self):
        log.debug('Starting initial setup')
        print app_header('Setup Assistant')
        print format_msg('Let\'s begin...')

        self.config.APP_NAME = get_correct_answer('Please enter app name',
                                                  default='Not-So-TUF')
        dev_data_dir = get_correct_answer('Enter directory to store'
                                          ' work files',
                                          default='Current Working Directory')

        dev_data_dir = _directory_fixer(dev_data_dir)

        self.config.DEV_DATA_DIR = path_fixer(dev_data_dir)
        self.config.KEY_LENGTH = get_correct_answer('Enter a key length. '
                                                    'Longer is more secure '
                                                    'but takes longer to '
                                                    'compute!', default='2048')
        priv_key_name = get_correct_answer('Enter a name to give '
                                           'your private key',
                                           default='Not-So-TUF')
        self.config.add_private_key_name(priv_key_name)

        pub_key_name = get_correct_answer('Enter a name to give your'
                                          ' public key',
                                          default='Not-So-TUF')
        self.config.add_public_key_name(pub_key_name)

        self.config.UPDATE_URL = get_correct_answer('Enter your update url',
                                                    required=True)

        self.config.UPDATE_PATCHES = ask_yes_no('Would you like to enable '
                                                'patch updates?',
                                                default='yes')

        answer1 = ask_yes_no('Would you like to add scp settings?',
                             default='no')

        answer2 = ask_yes_no('Would you like to add S3 settings?',
                             default='no')

        # answer3 = ask_yes_no('Would you like to add ftp settings?',
                             # default='no')

        answer3 = False

        if answer1 or answer3:
            self.config.REMOTE_DIR = get_correct_answer('Enter remote dir',
                                                        required=True)
            self.config.HOST = get_correct_answer('Enter host', required=True)

        if answer1:
            self.config.SSH_USERNAME = get_correct_answer('Enter usernmae',
                                                          required=True)

            key_path = get_correct_answer('Enter path to ssh key',
                                          required=True)
            self.config.SSH_KEY_PATH = _directory_fixer(key_path)

        if answer2:
            self.config.ACCESS_KEY_ID = get_correct_answer('Enter access key'
                                                           ' ID',
                                                           required=True)
            self.config.SECRET_ACCESS_KEY = get_correct_answer('Enter secret ',
                                                               'Access Key',
                                                               required=True)
            self.config.BUCKET_NAME = get_correct_answer('Enter bucket name',
                                                         reuqired=True)

        if answer3:
            self.config.FTP_USERNAME = get_correct_answer('Enter ftp username',
                                                          required=True)
            self.config.FTP_PASSWORD = get_correct_answer('Enter ftp password',
                                                          required=True)

        self.save_config(self.config)
        package_handler = PackageHandler(self.nst)
        package_handler.setup()
        print 'Making keys...'
        self.make_keys()
        print app_header('Setup Complete')
        print format_msg('Now let\'s update some apps')
        time.sleep(3)

    def save_config(self, obj):
        self.nst.update_config(obj)
        log.debug('Saving Config')
        filename = os.path.join(_cwd, 'config.data')
        fc = FileCrypt(filename)
        fc.decrypt()
        with open(filename, 'w') as f:
            f.write(pickle.dumps(obj))
        fc.encrypt()

    def load_config(self):
        log.debug('Loading Config')
        filename = os.path.join(_cwd, 'config.data')
        fc = FileCrypt(filename)
        fc.decrypt()
        with open(filename, 'r') as f:
            config_data = pickle.loads(f.read())
        fc.encrypt()
        return config_data

    def copy_decrypted_config(self):
        log.debug('Attempting to copy decrypted config')
        filename = os.path.join(_cwd, 'config.data')
        fc = FileCrypt(filename)
        fc.decrypt()
        try:
            shutil.copy(filename, filename + ' copy')
        except Exception as e:
            log.error(str(e), exc_info=True)
            log.error('Cannot copy decrypted config file')
        fc.encrypt()
        msg = 'Decrypted config file copied.  Press enter to continue'
        raw_input(format_msg(msg))
        self.settings()

    def start(self):
        self.main()


class ClientConfig(object):

    UPDATE_URL = 'https://s3-us-west-1.amazonaws.com/not-so-tuf/'
    PUBLIC_KEY = (16464922665638673681992463013273935612664219094906980149897613483558764958275503471833999796774944451919679352299199781040261183626243789235159282484489660866211751942888040512642103075255732190154437871072326800313353343988071645793491504044388864374910418983325794095950121251375913490389709129928619827303368362856224581261135920008510653620012541583622789200202174035822993035846677533547389325544764270670429010858826140320098873827212627840823767544126373206301938873536107735052145489782939917808169280407467079055393335221958452100971113008041835393071679231080475062804424979544387709688694450412240360063013L, 65537)


def main():
    try:
        if getattr(sys, 'frozen', False):
            nst = NotSoTuf(__name__, ClientConfig())
            client = Client(nst)
            print 'Current Version {}'.format(__version__)
            print '\nChecking for updates...'
            update = client.update_check('Not So TUF', __version__)
            if update and not client.up_to_date:
                client.install_restart()
        try:
            ui = Ui()
            ui.start()
        except KeyboardInterrupt:
            # ToDo: Add clean up stuff here
            print '\n\nExited by user'
    except Exception as e:
        print str(e)
        log.error(str(e), exc_info=True)
