# The MIT License (MIT)
#
# Copyright (c) 2014 JohnyMoSwag <johnymoswag@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import hashlib
from io import BytesIO
import logging
import time
import sys
import urllib2


log = logging.getLogger(__name__)
c = logging.StreamHandler()
c.setLevel(logging.INFO)
log.addHandler(c)


class FileDownloader(object):
    """The FileDownloader object downloads files to memory and
    verifies their hash.  If hash is verified data is either
    written to disk to returned to calling object

    Args:

        filename (str): The name of file to download

        url (str): The url to retrieve the file

        hexdigest str(str): The md5 checksum of the file to download
    """
    def __init__(self, filename, url, hexdigest):
        self.start = time.time()
        self.filename = filename
        self.url = url
        self.hexdigest = hexdigest
        self.block_size = 4096 * 4
        self.file_binary_data = None

    def download_verify_write(self):
        """Downloads file then verifies against provided hash
        If hash verfies then writes data to disk

        Returns:
            (bool) Meanings::

                True - Hash verified

                False - Hash not verified
        """
        # Downloads and verifies file against
        # provided hash.
        self._download_to_memory()
        check = self._check_hash()
        if check:
            log.debug('File hash verified')
            self._write_to_file()
            return True
        else:
            log.debug('Cannot verify file hash')
            del self.file_binary_data
            del self.my_file
            return False

    def download_verify_return(self):
        """
        Downloads file to memory, checks against provided hash
        If matched returns binary data

        Returns:
            (data) Meanings::

                Data - If everything verified

                None - If any verification didn't pass
        """
        self._download_to_memory()
        check = self._check_hash()
        if check:
            log.debug('File hash verified')
            return self.file_binary_data
        else:
            log.debug('Cannot verify file hash')
            return None

    @staticmethod
    def _best_block_size(elapsed_time, bytes):
        """Returns best block size for current internet connection speed

        Returns:
            (int) Size of chuck to read
        """
        new_min = max(bytes / 2.0, 1.0)
        new_max = min(max(bytes * 2.0, 1.0), 4194304)  # Do not surpass 4 MB
        if elapsed_time < 0.001:
            return int(new_max)
        rate = bytes / elapsed_time
        if rate > new_max:
            return int(new_max)
        if rate < new_min:
            return int(new_min)
        return int(rate)

    def _download_to_memory(self):
        """Downloads file to memory.  Keeps internal reference
        """
        try:
            data = urllib2.urlopen(self.url)
        except urllib2.HTTPError:
            log.warning('Might have had spaces in an S3 url...')
            self.url = self.url.replace(' ', '+')
            log.warning('S3 updated url {}'.format(self.url))
            data = urllib2.urlopen(self.url)
        except Exception as e:
            log.error(str(e), exc_info=True)
            return None
        content_length = self._get_content_length(data)
        self.my_file = BytesIO()
        log.debug('Downloading {} from \n{}'.format(self.filename, self.url))
        recieved_data = 0
        while 1:
            start_block = time.time()
            file_data = data.read(self.block_size)
            recieved_data += len(file_data)
            end_block = time.time()
            if not file_data:
                print '\n'
                self.my_file.flush()
                self.my_file.seek(0)
                self.file_binary_data = self.my_file.read()
                log.info('Download Complete')
                log.debug('Finished in {} seconds'.format(time.time() -
                          self.start))
                break
            percent = self._calc_progress_percent(recieved_data,
                                                  content_length)
            sys.stdout.write('\r{} Percent Complete'.format(percent))
            sys.stdout.flush()
            self.my_file.write(file_data)
            self.block_size = self._best_block_size(end_block - start_block,
                                                    len(file_data))
            log.debug('Block size: %s' % self.block_size)

    def _write_to_file(self):
        """Writes download data to disk
        """
        # Writes file to disk
        # TODO: Maybe add more verification here
        # Suggestions always welcome
        with open(self.filename, 'wb') as f:
            f.write(self.file_binary_data)

    def _check_hash(self):
        """Checks hash of downloaded file.

        Returns:
            (bool) Meanings::

                True - Hash verified

                False - Hash not verified
        """
        # Simple hash returning function
        if self.file_binary_data is None:
            return False
        log.debug('Checking file hash')
        log.debug('Update hash: {}'.format(self.hexdigest))
        file_hash = hashlib.md5(self.file_binary_data).hexdigest()
        log.debug('File hash: {}'.format(file_hash))
        if file_hash == self.hexdigest:
            return True
        return False

    def _get_content_length(self, data):
        try:
            content_length = int(data.headers["Content-Length"])
            log.debug('Got content length of: %s', content_length)
            return content_length
        except Exception as e:
            log.error(e, exc_info=True)
            return 100000000

    @staticmethod
    def _calc_eta(start, now, total, current):
        # Not currently implemented
        # Calculates remaining time of download
        if total is None:
            return '--:--'
        dif = now - start
        if current == 0 or dif < 0.001:  # One millisecond
            return '--:--'
        rate = float(current) / dif
        eta = int((float(total) - float(current)) / rate)
        (eta_mins, eta_secs) = divmod(eta, 60)
        if eta_mins > 99:
            return '--:--'
        return '%02d:%02d' % (eta_mins, eta_secs)

    def _calc_progress_percent(self, x, y):
        percent = float(x) / y * 100
        percent = '%.1f' % percent
        return percent
